<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Customer;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class MigrateCustomers extends Command
{
    protected $signature = 'migrate:customers {file}';
    protected $description = 'Migrate customers from CSV to ZedBankOS';

    public function handle()
    {
        $filePath = $this->argument('file');

        if (!file_exists($filePath)) {
            $this->error("File not found at: " . $filePath);
            return;
        }

        $handle = fopen($filePath, "r");
        
        // READ THE FIRST ROW AS HEADERS
        $rawHeaders = fgetcsv($handle);
        if (!$rawHeaders) {
            $this->error("File is empty or invalid.");
            return;
        }

        // Clean headers: remove BOM or whitespace
        $headers = array_map(function($header) {
            return trim(preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $header));
        }, $rawHeaders);

        $this->info("Starting migration...");

        $count = 0;
        while (($data = fgetcsv($handle)) !== FALSE) {
            // Skip empty rows or rows that don't match header count
            if (count($headers) !== count($data) || empty($data[0])) {
                continue;
            }

            $row = array_combine($headers, $data);

            // 1. Process Name
            $fullName = trim($row['Name'] ?? '');
            $nameParts = explode(' ', $fullName);
            $firstName = $nameParts[0] ?? 'N/A';
            $surname = count($nameParts) > 1 ? end($nameParts) : '';
            $middleName = (count($nameParts) > 2) ? $nameParts[1] : null;

            // 2. Process Date (Handling DD/MM/YYYY format)
            $dob = null;
            $rawDate = $row['Date of birth ( or incorporation)'] ?? null;
            if (!empty($rawDate)) {
                try {
                    $dob = Carbon::createFromFormat('d/m/Y', $rawDate)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $dob = Carbon::parse($rawDate)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $dob = null;
                    }
                }
            }

            // 3. Process Phone (Prepend 0 if it starts with 97, 96, etc.)
            $phone = $row['Phone Number'] ?? '';
            if (!empty($phone) && strlen($phone) == 9) {
                $phone = '0' . $phone;
            }

            try {
                // We use find to check existence so we can manually handle the customer_number increment safely
                $customer = Customer::find($row['ID']);

                $customerData = [
                    'cgroup'            => $row['Customer Group'] ?? 'Individual',
                    'first_name'        => $firstName,
                    'middle_name'       => $middleName,
                    'surname'           => $surname,
                    'company'           => $row['Company'] ?? null,
                    'income'            => (float) str_replace(',', '', $row['Estimated Monthly Income'] ?? 0),
                    'phone'             => $phone,
                    'email'             => !empty($row['Email']) ? trim($row['Email']) : null,
                    'town'              => $row['Town'] ?? null,
                    'address'           => $row['Address'] ?? null,
                    'province'          => $row['Province'] ?? null,
                    'occupation'        => $row['Occupation/Nature of business'] ?? '',
                    'status'            => $row['Status'] ?? 'Active',
                    'idtype'            => $row['ID Type'] ?? 'NRC',
                    'number'            => $row['ID No.'] ?? '',
                    'tpin'              => (int) ($row['TPIN'] ?? 0),
                    'dob'               => $dob,
                    'pass'              => null, 
                    'verification_status' => 'verified',
                ];

                if (!$customer) {
                    // Manually set ID and trigger customer number generation for new records
                    $customerData['id'] = $row['ID'];
                    
                    // Force the generation here to ensure it increments correctly in the loop
                    $customerData['customer_number'] = Customer::generateCustomerNumber();
                    
                    Customer::create($customerData);
                } else {
                    // Update existing record
                    $customer->update($customerData);
                }

                $count++;
                $this->output->write('.');
            } catch (\Exception $e) {
                $this->error("\nError migrating ID {$row['ID']}: " . $e->getMessage());
            }
        }

        fclose($handle);
        $this->info("\nMigration finished. Total records: $count");
    }
}