<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ChartOfAccount;
use App\Models\JournalEntry;
use App\Models\Receivable;
use App\Models\SupplierInvoice;
use App\Models\CashBankAccount;
use App\Models\InternalCashBankTransfer;
use App\Models\PettyCash;

class AccountingController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        $companyId = $user->companyid;
        
        // Initialize statistics
        $stats = [];
        
        // Chart of Accounts statistics
        $stats['accountsCount'] = ChartOfAccount::count();
        
        // Journal Entries statistics
        $stats['journalEntriesCount'] = JournalEntry::count();
        
        // Receivables statistics
        $stats['totalReceivables'] = Receivable::where('status', 'outstanding')->sum('amount');
        
        // Payables statistics (simplified calculation)
        $stats['totalPayables'] = SupplierInvoice::where('status', 'unpaid')->sum('total_amount');
        
        // Financial position (simplified calculations)
        $stats['totalAssets'] = $this->calculateTotalAssets();
        $stats['totalLiabilities'] = $this->calculateTotalLiabilities();
        $stats['totalEquity'] = $stats['totalAssets'] - $stats['totalLiabilities'];
        $stats['netIncome'] = $this->calculateNetIncome();
        
        // Recent journal entries - FIXED: using 'items' instead of 'journalEntryItems'
        $recentJournalEntries = JournalEntry::with(['items'])
            ->orderBy('entry_date', 'desc')
            ->take(5)
            ->get();

        // Cash and Banking Data
        if ($companyId) {
            $cashBankAccounts = CashBankAccount::with(['glAccount'])
                ->where('companyid', $companyId)
                ->where('isActive', true)
                ->orderBy('accountType')
                ->orderBy('accountName')
                ->take(6) // Show up to 6 accounts
                ->get();

            $recentTransfers = InternalCashBankTransfer::with(['fromAccount', 'toAccount'])
                ->where('companyid', $companyId)
                ->orderBy('created_at', 'desc')
                ->take(5)
                ->get();

            $recentPettyCash = PettyCash::with(['sourceAccount', 'requester'])
                ->where('company_id', $companyId)
                ->orderBy('created_at', 'desc')
                ->take(5)
                ->get();

            // Calculate total balances
            $stats['cashBankAccounts'] = $cashBankAccounts;
            $stats['recentCashTransfers'] = $recentTransfers;
            $stats['recentPettyCash'] = $recentPettyCash;
            $stats['totalCashBalance'] = CashBankAccount::where('companyid', $companyId)
                ->where('isActive', true)
                ->sum('current_balance');
            $stats['cashBalance'] = CashBankAccount::where('companyid', $companyId)
                ->where('isActive', true)
                ->where('accountType', 'Cash')
                ->sum('current_balance');
            $stats['bankBalance'] = CashBankAccount::where('companyid', $companyId)
                ->where('isActive', true)
                ->where('accountType', 'Bank')
                ->sum('current_balance');
            $stats['mobileMoneyBalance'] = CashBankAccount::where('companyid', $companyId)
                ->where('isActive', true)
                ->where('accountType', 'Mobile Money')
                ->sum('current_balance');
        } else {
            // Set defaults if no company ID
            $stats['cashBankAccounts'] = collect();
            $stats['recentCashTransfers'] = collect();
            $stats['recentPettyCash'] = collect();
            $stats['totalCashBalance'] = 0;
            $stats['cashBalance'] = 0;
            $stats['bankBalance'] = 0;
            $stats['mobileMoneyBalance'] = 0;
        }

        return view('accounting.index', array_merge($stats, [
            'recentJournalEntries' => $recentJournalEntries
        ]));
    }

    private function calculateTotalAssets()
    {
        // Include cash and bank accounts in total assets
        $companyId = auth()->user()->companyid;
        $cashBankAssets = $companyId ? 
            CashBankAccount::where('companyid', $companyId)
                ->where('isActive', true)
                ->sum('current_balance') : 0;
        
        $chartAccountAssets = ChartOfAccount::where('account_type', 'asset')->sum('current_balance');
        
        return $cashBankAssets + $chartAccountAssets;
    }

    private function calculateTotalLiabilities()
    {
        // Simplified calculation - in real scenario, sum all liability accounts
        return ChartOfAccount::where('account_type', 'liability')->sum('current_balance');
    }

    private function calculateNetIncome()
    {
        // Simplified calculation - in real scenario, calculate from income and expense accounts
        $revenue = ChartOfAccount::where('account_type', 'revenue')->sum('current_balance');
        $expenses = ChartOfAccount::where('account_type', 'expense')->sum('current_balance');
        
        return $revenue - $expenses;
    }
}