<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\BusinessProcessMapping;
use App\Models\ChartOfAccount;

class BusinessProcessMappingController extends Controller
{
    /**
     * Display a listing of business process mappings.
     */
    public function index()
    {
        $mappings = BusinessProcessMapping::with(['account', 'creator'])
            ->orderBy('nameofthebusinessprocess')
            ->get();
            
        $processNames = BusinessProcessMapping::getProcessNames();
        $accounts = ChartOfAccount::active()->orderBy('code')->get();
        
        return view('accounting.business-process-mappings.index', compact('mappings', 'processNames', 'accounts'));
    }

    /**
     * Show the form for creating a new business process mapping.
     */
    public function create()
    {
        // Not used - we use modal
        abort(404);
    }

    /**
     * Store a newly created business process mapping.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nameofthebusinessprocess' => 'required|string|max:255',
            'businessprocessdescription' => 'nullable|string',
            'accountid' => 'required|exists:chart_of_accounts,id',
        ]);

        try {
            // Check if process already exists
            $existingMapping = BusinessProcessMapping::where('nameofthebusinessprocess', $validated['nameofthebusinessprocess'])->first();
            
            if ($existingMapping) {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Business process mapping already exists for: ' . $validated['nameofthebusinessprocess']
                    ], 422);
                }
                
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Business process mapping already exists for: ' . $validated['nameofthebusinessprocess']);
            }

            $mapping = BusinessProcessMapping::create([
                'nameofthebusinessprocess' => $validated['nameofthebusinessprocess'],
                'businessprocessdescription' => $validated['businessprocessdescription'] ?? null,
                'accountid' => $validated['accountid'],
                'createdby' => auth()->id(),
                'companyid' => 1,
                'branchid' => 1,
            ]);

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Business process mapping created successfully!',
                    'mapping' => $mapping->load('account', 'creator')
                ]);
            }

            return redirect()->route('admin.accounting.settings.business-process-mappings.index')
                ->with('success', 'Business process mapping created successfully!');

        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error creating business process mapping: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->withInput()
                ->with('error', 'Error creating business process mapping: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified business process mapping.
     */
    public function show($id)
    {
        // Not used - we use modal
        abort(404);
    }

    /**
     * Show the form for editing the specified business process mapping.
     */
    public function edit($id)
    {
        // Not used - we use modal
        abort(404);
    }

    /**
     * Update the specified business process mapping.
     */
    public function update(Request $request, $id)
    {
        $mapping = BusinessProcessMapping::findOrFail($id);
        
        $validated = $request->validate([
            'nameofthebusinessprocess' => 'required|string|max:255',
            'businessprocessdescription' => 'nullable|string',
            'accountid' => 'required|exists:chart_of_accounts,id',
        ]);

        try {
            // Check if process already exists (excluding current record)
            $existingMapping = BusinessProcessMapping::where('nameofthebusinessprocess', $validated['nameofthebusinessprocess'])
                ->where('businessmappingid', '!=', $id)
                ->first();
            
            if ($existingMapping) {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Business process mapping already exists for: ' . $validated['nameofthebusinessprocess']
                    ], 422);
                }
                
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Business process mapping already exists for: ' . $validated['nameofthebusinessprocess']);
            }

            $mapping->update([
                'nameofthebusinessprocess' => $validated['nameofthebusinessprocess'],
                'businessprocessdescription' => $validated['businessprocessdescription'] ?? null,
                'accountid' => $validated['accountid'],
            ]);

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Business process mapping updated successfully!',
                    'mapping' => $mapping->load('account', 'creator')
                ]);
            }

            return redirect()->route('admin.accounting.settings.business-process-mappings.index')
                ->with('success', 'Business process mapping updated successfully!');

        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error updating business process mapping: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->withInput()
                ->with('error', 'Error updating business process mapping: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified business process mapping.
     */
    public function destroy(Request $request, $id)
    {
        $mapping = BusinessProcessMapping::findOrFail($id);
        
        try {
            $mapping->delete();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Business process mapping deleted successfully!'
                ]);
            }

            return redirect()->route('admin.accounting.settings.business-process-mappings.index')
                ->with('success', 'Business process mapping deleted successfully!');

        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error deleting business process mapping: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->route('admin.accounting.settings.business-process-mappings.index')
                ->with('error', 'Error deleting business process mapping: ' . $e->getMessage());
        }
    }

    /**
     * Get account for a specific process (API endpoint).
     */
    public function getAccountForProcess($processName)
    {
        try {
            $account = BusinessProcessMapping::getAccountForProcess($processName);
            
            if (!$account) {
                return response()->json([
                    'success' => false,
                    'error' => 'No account mapping found for process: ' . $processName
                ], 404);
            }

            return response()->json([
                'success' => true,
                'process_name' => $processName,
                'account' => $account,
                'account_code' => $account->code,
                'account_name' => $account->name
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Error retrieving account mapping: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get mapping data for editing (AJAX endpoint).
     */
    public function getMappingData($id)
    {
        try {
            $mapping = BusinessProcessMapping::with('account')->findOrFail($id);

            return response()->json([
                'success' => true,
                'mapping' => $mapping
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Error retrieving mapping data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk delete business process mappings.
     */
    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:business_process_mappings,businessmappingid'
        ]);

        try {
            $count = BusinessProcessMapping::whereIn('businessmappingid', $validated['ids'])->delete();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => $count . ' business process mapping(s) deleted successfully!'
                ]);
            }

            return redirect()->route('admin.accounting.settings.business-process-mappings.index')
                ->with('success', $count . ' business process mapping(s) deleted successfully!');

        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error deleting business process mappings: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->route('admin.accounting.settings.business-process-mappings.index')
                ->with('error', 'Error deleting business process mappings: ' . $e->getMessage());
        }
    }

    /**
     * Export business process mappings.
     */
    public function export(Request $request)
    {
        try {
            $mappings = BusinessProcessMapping::with(['account', 'creator'])
                ->orderBy('nameofthebusinessprocess')
                ->get();

            $data = [
                ['Business Process', 'Description', 'GL Account Code', 'GL Account Name', 'Created By', 'Created Date']
            ];

            foreach ($mappings as $mapping) {
                $data[] = [
                    $mapping->nameofthebusinessprocess,
                    $mapping->businessprocessdescription ?? '',
                    $mapping->account->code ?? 'N/A',
                    $mapping->account->name ?? 'N/A',
                    $mapping->creator->name ?? 'System',
                    $mapping->created_at->format('Y-m-d H:i:s')
                ];
            }

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'data' => $data
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Export functionality ready. Data prepared for ' . count($mappings) . ' mappings.'
            ]);

        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error exporting business process mappings: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->route('admin.accounting.settings.business-process-mappings.index')
                ->with('error', 'Error exporting business process mappings: ' . $e->getMessage());
        }
    }

    /**
     * Validate process name uniqueness.
     */
    public function validateProcessName(Request $request)
    {
        $validated = $request->validate([
            'nameofthebusinessprocess' => 'required|string|max:255',
            'exclude_id' => 'nullable|exists:business_process_mappings,businessmappingid'
        ]);

        try {
            $query = BusinessProcessMapping::where('nameofthebusinessprocess', $validated['nameofthebusinessprocess']);
            
            if (!empty($validated['exclude_id'])) {
                $query->where('businessmappingid', '!=', $validated['exclude_id']);
            }

            $exists = $query->exists();

            return response()->json([
                'success' => true,
                'exists' => $exists,
                'message' => $exists ? 'Process name already exists' : 'Process name is available'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Error validating process name: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get statistics about business process mappings.
     */
    public function getStatistics()
    {
        try {
            $totalMappings = BusinessProcessMapping::count();
            $mappingsByProcess = BusinessProcessMapping::groupBy('nameofthebusinessprocess')
                ->selectRaw('nameofthebusinessprocess, COUNT(*) as count')
                ->get();

            return response()->json([
                'success' => true,
                'statistics' => [
                    'total_mappings' => $totalMappings,
                    'mappings_by_process' => $mappingsByProcess
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Error retrieving statistics: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Create default mappings (optional - can be removed if not needed).
     */
    public function createDefaultMappings(Request $request)
    {
        try {
            $createdCount = BusinessProcessMapping::createDefaultMappings();
            
            return response()->json([
                'success' => true,
                'message' => $createdCount . ' default mappings created successfully.',
                'count' => $createdCount
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating default mappings: ' . $e->getMessage()
            ], 500);
        }
    }
}