<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Company;
use App\Models\Currency;
use App\Models\FinancialYear;
use App\Models\CompanySettingsHistory;
use Illuminate\Support\Facades\Auth;

class CompanyController extends Controller
{
    /**
     * Display company profile and settings
     */
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->companyid) {
            return redirect()->route('admin.accounting.index')
                ->with('error', 'User is not associated with a company.');
        }

        $company = Company::with(['reportingCurrency', 'currentFinancialYear', 'owner'])
            ->findOrFail($user->companyid);

        // Get available currencies for dropdown
        $currencies = Currency::where('is_active', true)
            ->orderBy('code')
            ->get();

        // Get available financial years for dropdown (excluding closed years)
        $financialYears = FinancialYear::availableForSelection($company->id)
            ->get();

        // Get settings change history
        $settingsHistory = CompanySettingsHistory::where('company_id', $company->id)
            ->with('changedBy')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        return view('accounting.company.index', compact(
            'company',
            'currencies',
            'financialYears',
            'settingsHistory'
        ));
    }

    /**
     * Update reporting currency
     */
    public function updateReportingCurrency(Request $request)
    {
        $request->validate([
            'currency_id' => 'required|exists:currencies,id',
            'reason' => 'nullable|string|max:500'
        ]);

        $user = Auth::user();
        $company = Company::findOrFail($user->companyid);

        // Check if currency is active
        $currency = Currency::find($request->currency_id);
        if (!$currency->is_active) {
            return redirect()->back()
                ->with('error', 'Selected currency is not active.');
        }

        // Update reporting currency
        $company->updateReportingCurrency(
            $request->currency_id,
            $request->reason,
            $user->id
        );

        return redirect()->back()
            ->with('success', 'Reporting currency updated successfully.');
    }

    /**
     * Update current financial year
     */
    public function updateFinancialYear(Request $request)
    {
        $request->validate([
            'financial_year_id' => 'required|exists:financial_years,id',
            'reason' => 'nullable|string|max:500'
        ]);

        $user = Auth::user();
        $company = Company::findOrFail($user->companyid);

        // Check if financial year belongs to company
        $financialYear = FinancialYear::find($request->financial_year_id);
        if ($financialYear->company_id != $company->id) {
            return redirect()->back()
                ->with('error', 'Selected financial year does not belong to your company.');
        }

        // Check if financial year is closed
        if ($financialYear->is_closed) {
            return redirect()->back()
                ->with('error', 'Cannot select a closed financial year.');
        }

        // Update financial year
        $company->updateFinancialYear(
            $request->financial_year_id,
            $request->reason,
            $user->id
        );

        return redirect()->back()
            ->with('success', 'Current financial year updated successfully.');
    }

    /**
     * Update company profile
     */
   
    /**
     * View settings change history
     */
    public function history()
    {
        $user = Auth::user();
        $company = Company::findOrFail($user->companyid);

        $history = CompanySettingsHistory::where('company_id', $company->id)
            ->with('changedBy')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('accounting.company.history', compact('company', 'history'));
    }

    /**
     * Get available financial years for dropdown (AJAX)
     */
    public function getFinancialYears(Request $request)
    {
        $user = Auth::user();
        $company = Company::findOrFail($user->companyid);

        $financialYears = FinancialYear::availableForSelection($company->id)
            ->get()
            ->map(function ($year) {
                return [
                    'id' => $year->id,
                    'text' => $year->year . ' (' . $year->formatted_date_range . ')'
                ];
            });

        return response()->json($financialYears);
    }

    /**
     * Get available currencies for dropdown (AJAX)
     */
    public function getCurrencies(Request $request)
    {
        $currencies = Currency::where('is_active', true)
            ->orderBy('code')
            ->get()
            ->map(function ($currency) {
                return [
                    'id' => $currency->id,
                    'text' => $currency->code . ' - ' . $currency->name . ' (' . $currency->symbol . ')'
                ];
            });

        return response()->json($currencies);
    }
    
    
    /**
 * Update company profile
 */
public function updateProfile(Request $request)
{
    $user = Auth::user();
    $company = Company::findOrFail($user->companyid);

    $validated = $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|email|max:255|unique:companies,email,' . $company->id,
        'phone' => 'nullable|string|max:20',
        'address' => 'nullable|string|max:500',
        'town' => 'nullable|string|max:100',
        'country' => 'required|string|max:100',
        'postal_code' => 'nullable|string|max:20',
        'contact_person' => 'nullable|string|max:255',
        'contact_person_phone' => 'nullable|string|max:20',
        'contact_person_email' => 'nullable|email|max:100',
        'website' => 'nullable|url|max:255',
        'industry' => 'nullable|string|max:100',
        'company_size' => 'required|in:1-10,11-50,51-200,201-500,500+',
        'company_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:5120', // 5MB max
        'remove_logo' => 'nullable|boolean',
    ]);

    // Handle logo removal
    if ($request->has('remove_logo') && $request->remove_logo) {
        $company->deleteLogo();
        $validated['company_logo'] = null;
    }
    // Handle logo upload
    elseif ($request->hasFile('company_logo')) {
        $path = $company->uploadLogo($request->file('company_logo'));
        $validated['company_logo'] = $path;
        
        // Remove the file from validated array as we'll handle separately
        unset($validated['company_logo']);
    }

    $company->update($validated);

    return redirect()->back()
        ->with('success', 'Company profile updated successfully.');
}
    
}