<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Currency;

class CurrencyController extends Controller
{
    public function index()
    {
        $currencies = Currency::orderBy('code')->get();
        return view('accounting.settings.currencies.index', compact('currencies'));
    }

    public function create()
    {
        return view('accounting.settings.currencies.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|unique:currencies,code|size:3',
            'name' => 'required|string|max:255',
            'symbol' => 'required|string|max:10',
            'symbol_native' => 'nullable|string|max:10',
            'decimal_digits' => 'required|integer|min:0|max:4',
            'rounding' => 'required|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        // Add authenticated user as creator
        $validated['created_by'] = auth()->id();
        $validated['updated_by'] = auth()->id();

        Currency::create($validated);

        return redirect()->route('admin.accounting.settings.currencies.index')
            ->with('success', 'Currency created successfully.');
    }

    public function show($id)
    {
        $currency = Currency::findOrFail($id);
        return view('accounting.settings.currencies.show', compact('currency'));
    }

    public function edit($id)
    {
        $currency = Currency::findOrFail($id);
        return view('accounting.settings.currencies.edit', compact('currency'));
    }

    public function update(Request $request, $id)
    {
        $currency = Currency::findOrFail($id);
        
        $validated = $request->validate([
            'code' => 'required|unique:currencies,code,' . $id . '|size:3',
            'name' => 'required|string|max:255',
            'symbol' => 'required|string|max:10',
            'symbol_native' => 'nullable|string|max:10',
            'decimal_digits' => 'required|integer|min:0|max:4',
            'rounding' => 'required|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        // Add authenticated user as updater
        $validated['updated_by'] = auth()->id();

        $currency->update($validated);

        return redirect()->route('admin.accounting.settings.currencies.index')
            ->with('success', 'Currency updated successfully.');
    }

    public function destroy($id)
    {
        $currency = Currency::findOrFail($id);
        
        // Check if currency is used in chart of accounts
        if ($currency->chartOfAccounts()->count() > 0) {
            return redirect()->route('admin.accounting.settings.currencies.index')
                ->with('error', 'Cannot delete currency that is used in chart of accounts.');
        }

        // Check if currency is used in currency rates
        if ($currency->baseCurrencyRates()->count() > 0 || $currency->targetCurrencyRates()->count() > 0) {
            return redirect()->route('admin.accounting.settings.currencies.index')
                ->with('error', 'Cannot delete currency that is used in currency rates.');
        }

        $currency->delete();

        return redirect()->route('admin.accounting.settings.currencies.index')
            ->with('success', 'Currency deleted successfully.');
    }
}