<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\FinancialYear;

class FinancialYearController extends Controller
{
    public function index()
    {
        $financialYears = FinancialYear::orderBy('start_date', 'desc')->get();
        return view('accounting.settings.financial-years.index', compact('financialYears'));
    }

    public function create()
    {
        // Check if there's an active financial year
        $activeYear = FinancialYear::where('is_active', true)->first();
        
        return view('accounting.settings.financial-years.create', compact('activeYear'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'year' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_active' => 'boolean',
            'is_closed' => 'boolean',
        ]);

        // Check if trying to create an active year when one already exists
        if ($validated['is_active'] ?? false) {
            $activeYear = FinancialYear::where('is_active', true)->first();
            if ($activeYear) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'There is already an active financial year (' . $activeYear->year . '). Please deactivate it first or create this year as inactive.');
            }
        }

        // Check for date overlaps
        $overlappingYear = FinancialYear::where(function($query) use ($validated) {
            $query->whereBetween('start_date', [$validated['start_date'], $validated['end_date']])
                  ->orWhereBetween('end_date', [$validated['start_date'], $validated['end_date']])
                  ->orWhere(function($q) use ($validated) {
                      $q->where('start_date', '<=', $validated['start_date'])
                        ->where('end_date', '>=', $validated['end_date']);
                  });
        })->first();

        if ($overlappingYear) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'This financial year overlaps with an existing year (' . $overlappingYear->year . '). Please adjust the dates.');
        }

        // Add authenticated user as creator
        $validated['created_by'] = auth()->id();
        $validated['updated_by'] = auth()->id();

        FinancialYear::create($validated);

        return redirect()->route('admin.accounting.settings.financial-years.index')
            ->with('success', 'Financial year created successfully.');
    }

    public function show($id)
    {
        $financialYear = FinancialYear::findOrFail($id);
        return view('accounting.settings.financial-years.show', compact('financialYear'));
    }

    public function edit($id)
    {
        $financialYear = FinancialYear::findOrFail($id);
        $activeYear = FinancialYear::where('is_active', true)->where('id', '!=', $id)->first();
        
        return view('accounting.settings.financial-years.edit', compact('financialYear', 'activeYear'));
    }

    public function update(Request $request, $id)
    {
        $financialYear = FinancialYear::findOrFail($id);
        
        $validated = $request->validate([
            'year' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_active' => 'boolean',
            'is_closed' => 'boolean',
        ]);

        // Check if trying to activate a year when another active year exists
        if ($validated['is_active'] ?? false) {
            $activeYear = FinancialYear::where('is_active', true)->where('id', '!=', $id)->first();
            if ($activeYear) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'There is already an active financial year (' . $activeYear->year . '). Please deactivate it first.');
            }
        }

        // Check for date overlaps with other years
        $overlappingYear = FinancialYear::where('id', '!=', $id)
            ->where(function($query) use ($validated) {
                $query->whereBetween('start_date', [$validated['start_date'], $validated['end_date']])
                      ->orWhereBetween('end_date', [$validated['start_date'], $validated['end_date']])
                      ->orWhere(function($q) use ($validated) {
                          $q->where('start_date', '<=', $validated['start_date'])
                            ->where('end_date', '>=', $validated['end_date']);
                      });
            })->first();

        if ($overlappingYear) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'This financial year overlaps with another year (' . $overlappingYear->year . '). Please adjust the dates.');
        }

        // Add authenticated user as updater
        $validated['updated_by'] = auth()->id();

        $financialYear->update($validated);

        return redirect()->route('admin.accounting.settings.financial-years.index')
            ->with('success', 'Financial year updated successfully.');
    }

    public function destroy($id)
    {
        $financialYear = FinancialYear::findOrFail($id);
        
        // Prevent deletion of active financial year
        if ($financialYear->is_active) {
            return redirect()->route('admin.accounting.settings.financial-years.index')
                ->with('error', 'Cannot delete an active financial year. Please deactivate it first.');
        }

        // Prevent deletion if there are journal entries
        if ($financialYear->journalEntries()->count() > 0) {
            return redirect()->route('admin.accounting.settings.financial-years.index')
                ->with('error', 'Cannot delete financial year that has journal entries.');
        }

        $financialYear->delete();

        return redirect()->route('admin.accounting.settings.financial-years.index')
            ->with('success', 'Financial year deleted successfully.');
    }

    // Show close financial year form
    public function showClose($id)
    {
        $financialYear = FinancialYear::findOrFail($id);
        
        // Check if already closed
        if ($financialYear->is_closed) {
            return redirect()->route('admin.accounting.settings.financial-years.index')
                ->with('error', 'This financial year is already closed.');
        }

        return view('accounting.settings.financial-years.close', compact('financialYear'));
    }

    // Close financial year
    public function close(Request $request, $id)
    {
        $financialYear = FinancialYear::findOrFail($id);
        
        // Validate closure
        $request->validate([
            'confirmation' => 'required|in:CLOSE_YEAR',
        ]);

        // Check if already closed
        if ($financialYear->is_closed) {
            return redirect()->route('admin.accounting.settings.financial-years.index')
                ->with('error', 'This financial year is already closed.');
        }

        // Close the financial year
        $financialYear->close();

        return redirect()->route('admin.accounting.settings.financial-years.index')
            ->with('success', 'Financial year closed successfully.');
    }

    // Reopen financial year
    public function reopen($id)
    {
        $financialYear = FinancialYear::findOrFail($id);
        
        // Check if actually closed
        if (!$financialYear->is_closed) {
            return redirect()->route('admin.accounting.settings.financial-years.index')
                ->with('error', 'This financial year is not closed.');
        }

        // Reopen the financial year
        $financialYear->reopen();

        return redirect()->route('admin.accounting.settings.financial-years.index')
            ->with('success', 'Financial year reopened successfully.');
    }
}