<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\InternalCashBankTransfer;
use App\Models\CashBankAccount;
use App\Models\CashBook;
use App\Models\GeneralLedgerEntry;
use App\Models\ChartOfAccount;
use App\Models\Company;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class InternalTransferController extends Controller
{
    public function __construct()
    {
        // Removed middleware permission check
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $query = InternalCashBankTransfer::with(['fromAccount', 'toAccount', 'creator'])
            ->where('companyid', $companyId);

        // Apply search filter
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function($q) use ($search) {
                $q->where('transferid', 'like', "%{$search}%")
                  ->orWhere('details', 'like', "%{$search}%")
                  ->orWhere('reference', 'like', "%{$search}%")
                  ->orWhere('referenceno', 'like', "%{$search}%")
                  ->orWhereHas('fromAccount', function($q) use ($search) {
                      $q->where('accountName', 'like', "%{$search}%")
                        ->orWhere('accountNumber', 'like', "%{$search}%");
                  })
                  ->orWhereHas('toAccount', function($q) use ($search) {
                      $q->where('accountName', 'like', "%{$search}%")
                        ->orWhere('accountNumber', 'like', "%{$search}%");
                  });
            });
        }

        // Apply date range filter
        if ($request->filled('date_from')) {
            $query->whereDate('transactiondate', '>=', $request->input('date_from'));
        }
        
        if ($request->filled('date_to')) {
            $query->whereDate('transactiondate', '<=', $request->input('date_to'));
        }

        // Apply status filter
        if ($request->filled('status')) {
            $query->where('transferstatus', $request->input('status'));
        }

        // Apply account filter
        if ($request->filled('account_id')) {
            $accountId = $request->input('account_id');
            $query->where(function($q) use ($accountId) {
                $q->where('accountfrom', $accountId)
                  ->orWhere('accountto', $accountId);
            });
        }

        // Get all accounts for filter dropdown
        $accounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->orderBy('accountName')
            ->get();

        $transfers = $query->orderBy('transactiondate', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(20)
            ->appends($request->except('page'));

        return view('accounting.cash-banking.internal-transfers.index', compact('transfers', 'accounts'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $accounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->orderBy('accountName')
            ->get();

        $currencies = DB::table('currencies')->get();

        return view('accounting.cash-banking.internal-transfers.create', compact('accounts', 'currencies'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $validated = $request->validate([
            'transactiondate' => 'required|date',
            'accountfrom' => 'required|exists:cash_bank_accounts,cashBankId',
            'accountto' => [
                'required',
                'exists:cash_bank_accounts,cashBankId',
                'different:accountfrom'
            ],
            'details' => 'nullable|string|max:500',
            'currency' => 'required|string|max:10',
            'exchangerate' => 'required|numeric|min:0.0001',
            'amountsent' => 'required|numeric|min:0.01',
            'reference' => 'nullable|string|max:255',
            'referenceno' => 'nullable|string|max:255',
            'transferstatus' => 'required|in:Pending,Sent'
        ]);

        // If transfer is being sent, check balance
        if ($validated['transferstatus'] === 'Sent') {
            $sourceAccount = CashBankAccount::find($validated['accountfrom']);
            if (!$sourceAccount) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Source account not found.');
            }
            
            if (!$sourceAccount->hasSufficientBalance($validated['amountsent'])) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Insufficient balance in source account. Available balance: ' . $sourceAccount->formatted_balance);
            }
        }

        DB::transaction(function () use ($validated, $companyId) {
            $transfer = InternalCashBankTransfer::create([
                'transactiondate' => $validated['transactiondate'],
                'accountfrom' => $validated['accountfrom'],
                'accountto' => $validated['accountto'],
                'details' => $validated['details'],
                'currency' => $validated['currency'],
                'exchangerate' => $validated['exchangerate'],
                'amountsent' => $validated['amountsent'],
                'amountreceived' => $validated['amountsent'] * $validated['exchangerate'],
                'reference' => $validated['reference'],
                'referenceno' => $validated['referenceno'],
                'createdby' => auth()->id(),
                'companyid' => $companyId,
                'branchid' => auth()->user()->branchid ?? 1,
                'transferstatus' => $validated['transferstatus'],
                'sent_at' => $validated['transferstatus'] === 'Sent' ? now() : null,
                'sent_by' => $validated['transferstatus'] === 'Sent' ? auth()->id() : null
            ]);

            // Process transfer if status is Sent
            if ($validated['transferstatus'] === 'Sent') {
                $this->processTransfer($transfer);
            }
        });

        $message = $validated['transferstatus'] === 'Sent' 
            ? 'Internal transfer created and posted successfully.' 
            : 'Internal transfer saved as draft.';

        return redirect()->route('admin.accounting.cash-banking.internal-transfers.index')
            ->with('success', $message);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transfer = InternalCashBankTransfer::with(['fromAccount', 'toAccount', 'creator', 'sentByUser'])
            ->where('transferid', $id)
            ->where('companyid', $companyId)
            ->firstOrFail();

        // Get cashbook entries for this transfer
        $cashbookEntries = CashBook::where('transfer_id', $id)
            ->with('account')
            ->orderBy('transaction_date', 'asc')
            ->get();

        // Get general ledger entries for this transfer
        $ledgerEntries = GeneralLedgerEntry::where('reference_type', 'internal_transfer')
            ->where('reference_id', $id)
            ->with('account')
            ->orderBy('entrydate', 'asc')
            ->get();

        return view('accounting.cash-banking.internal-transfers.show', compact(
            'transfer', 
            'cashbookEntries', 
            'ledgerEntries'
        ));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transfer = InternalCashBankTransfer::where('transferid', $id)
            ->where('companyid', $companyId)
            ->where('transferstatus', 'Pending')
            ->firstOrFail();

        $accounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->orderBy('accountName')
            ->get();

        $currencies = DB::table('currencies')->get();

        return view('accounting.cash-banking.internal-transfers.edit', compact('transfer', 'accounts', 'currencies'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transfer = InternalCashBankTransfer::where('transferid', $id)
            ->where('companyid', $companyId)
            ->where('transferstatus', 'Pending')
            ->firstOrFail();

        $validated = $request->validate([
            'transactiondate' => 'required|date',
            'accountfrom' => 'required|exists:cash_bank_accounts,cashBankId',
            'accountto' => [
                'required',
                'exists:cash_bank_accounts,cashBankId',
                'different:accountfrom'
            ],
            'details' => 'nullable|string|max:500',
            'currency' => 'required|string|max:10',
            'exchangerate' => 'required|numeric|min:0.0001',
            'amountsent' => 'required|numeric|min:0.01',
            'reference' => 'nullable|string|max:255',
            'referenceno' => 'nullable|string|max:255',
            'transferstatus' => 'required|in:Pending,Sent'
        ]);

        // If status is being changed to Sent, check balance
        if ($validated['transferstatus'] === 'Sent') {
            $sourceAccount = CashBankAccount::find($validated['accountfrom']);
            if (!$sourceAccount) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Source account not found.');
            }
            
            if (!$sourceAccount->hasSufficientBalance($validated['amountsent'])) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Insufficient balance in source account. Available balance: ' . $sourceAccount->formatted_balance);
            }
        }

        DB::transaction(function () use ($transfer, $validated) {
            $oldStatus = $transfer->transferstatus;
            
            $transferData = [
                'transactiondate' => $validated['transactiondate'],
                'accountfrom' => $validated['accountfrom'],
                'accountto' => $validated['accountto'],
                'details' => $validated['details'],
                'currency' => $validated['currency'],
                'exchangerate' => $validated['exchangerate'],
                'amountsent' => $validated['amountsent'],
                'amountreceived' => $validated['amountsent'] * $validated['exchangerate'],
                'reference' => $validated['reference'],
                'referenceno' => $validated['referenceno'],
                'transferstatus' => $validated['transferstatus']
            ];

            // Add sent information if changing to Sent
            if ($oldStatus === 'Pending' && $validated['transferstatus'] === 'Sent') {
                $transferData['sent_at'] = now();
                $transferData['sent_by'] = auth()->id();
            }

            $transfer->update($transferData);

            // Process transfer if status changed from Pending to Sent
            if ($oldStatus === 'Pending' && $validated['transferstatus'] === 'Sent') {
                $this->processTransfer($transfer);
            }
        });

        $message = $validated['transferstatus'] === 'Sent' 
            ? 'Internal transfer updated and posted successfully.' 
            : 'Internal transfer updated.';

        return redirect()->route('admin.accounting.cash-banking.internal-transfers.index')
            ->with('success', $message);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transfer = InternalCashBankTransfer::where('transferid', $id)
            ->where('companyid', $companyId)
            ->where('transferstatus', 'Pending')
            ->firstOrFail();

        DB::transaction(function () use ($transfer) {
            // Delete cashbook entries if any (shouldn't exist for pending transfers)
            CashBook::where('transfer_id', $transfer->transferid)->delete();
            
            // Delete general ledger entries if any (shouldn't exist for pending transfers)
            GeneralLedgerEntry::where('reference_type', 'internal_transfer')
                ->where('reference_id', $transfer->transferid)
                ->delete();
            
            $transfer->delete();
        });

        return redirect()->route('admin.accounting.cash-banking.internal-transfers.index')
            ->with('success', 'Internal transfer deleted successfully.');
    }

    /**
     * Get accounts for select2
     */
    public function getAccounts(Request $request)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return response()->json(['error' => 'User not associated with a company'], 400);
        }

        $accounts = CashBankAccount::where('companyid', $companyId)
            ->where('isActive', true)
            ->when($request->has('q'), function ($query) use ($request) {
                $query->where('accountName', 'like', '%' . $request->q . '%')
                      ->orWhere('accountNumber', 'like', '%' . $request->q . '%');
            })
            ->orderBy('accountName')
            ->get(['cashBankId as id', 'accountName as text', 'currency', 'current_balance']);

        return response()->json($accounts);
    }

    /**
     * Get account balance
     */
    public function getAccountBalance($accountId)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        if (!$companyId) {
            return response()->json(['error' => 'User not associated with a company'], 400);
        }

        $account = CashBankAccount::where('cashBankId', $accountId)
            ->where('companyid', $companyId)
            ->first();

        if (!$account) {
            return response()->json(['error' => 'Account not found'], 404);
        }

        return response()->json([
            'balance' => $account->current_balance,
            'formatted_balance' => $account->formatted_balance,
            'currency' => $account->currency
        ]);
    }

    /**
     * Send a pending transfer
     */
    public function send(Request $request, string $id)
    {
        // Get the user's company ID
        $companyId = auth()->user()->companyid;
        
        // Check if company ID exists
        if (!$companyId) {
            return redirect()->back()
                ->with('error', 'User is not associated with a company. Please contact administrator.');
        }

        $transfer = InternalCashBankTransfer::where('transferid', $id)
            ->where('companyid', $companyId)
            ->where('transferstatus', 'Pending')
            ->firstOrFail();

        DB::transaction(function () use ($transfer) {
            // Check balance
            $sourceAccount = CashBankAccount::find($transfer->accountfrom);
            if (!$sourceAccount) {
                throw new \Exception('Source account not found.');
            }
            
            if (!$sourceAccount->hasSufficientBalance($transfer->amountsent)) {
                throw new \Exception('Insufficient balance in source account. Available balance: ' . $sourceAccount->formatted_balance);
            }

            // Update transfer status with sent information
            $transfer->update([
                'transferstatus' => 'Sent',
                'sent_at' => now(),
                'sent_by' => auth()->id()
            ]);

            // Process the transfer
            $this->processTransfer($transfer);
        });

        return redirect()->route('admin.accounting.cash-banking.internal-transfers.show', $transfer->transferid)
            ->with('success', 'Transfer has been successfully sent and posted to ledger.');
    }

    /**
     * Process a transfer (update balances and post to ledgers)
     */
    private function processTransfer($transfer)
    {
        $sourceAccount = CashBankAccount::find($transfer->accountfrom);
        $destinationAccount = CashBankAccount::find($transfer->accountto);
        
        if (!$sourceAccount || !$destinationAccount) {
            throw new \Exception('One or both accounts not found.');
        }

        // Update account balances
        $sourceAccount->updateBalance($transfer->amountsent, 'credit');
        $destinationAccount->updateBalance($transfer->amountreceived, 'debit');

        // Post to CashBook
        $this->postToCashBook($transfer, $sourceAccount, $destinationAccount);

        // Post to General Ledger
        $this->postToGeneralLedger($transfer, $sourceAccount, $destinationAccount);
    }

    /**
     * Post transfer to CashBook
     */
    private function postToCashBook($transfer, $sourceAccount, $destinationAccount)
    {
        // Get the user's company ID and branch ID
        $companyId = auth()->user()->companyid;
        $branchId = auth()->user()->branchid ?? 1;
        
        // Generate reference number
        $referenceNumber = $transfer->reference ?? 'TRF-' . str_pad($transfer->transferid, 6, '0', STR_PAD_LEFT);
        
        // Get source account balance before transaction
        $sourceBalanceBefore = $sourceAccount->current_balance + $transfer->amountsent;
        
        // Get destination account balance before transaction
        $destinationBalanceBefore = $destinationAccount->current_balance - $transfer->amountreceived;
        
        // Source Account Entry (Outflow - Credit)
        CashBook::create([
            'companyid' => $companyId,
            'branchid' => $branchId,
            'account_id' => $sourceAccount->cashBankId,
            'transaction_date' => $transfer->transactiondate,
            'transaction_type' => CashBook::TRANSACTION_TRANSFER_OUT,
            'reference_number' => $referenceNumber,
            'transfer_id' => $transfer->transferid,
            'description' => $transfer->details ?? 'Transfer to ' . $destinationAccount->accountName,
            'amount' => -$transfer->amountsent, // Negative for outflow/credit
            'balance_before' => $sourceBalanceBefore,
            'balance_after' => $sourceAccount->current_balance,
            'currency' => $transfer->currency,
            'exchange_rate' => $transfer->exchangerate,
            'status' => 'posted',
            'created_by' => auth()->id(),
            'updated_by' => auth()->id()
        ]);

        // Destination Account Entry (Inflow - Debit)
        CashBook::create([
            'companyid' => $companyId,
            'branchid' => $branchId,
            'account_id' => $destinationAccount->cashBankId,
            'transaction_date' => $transfer->transactiondate,
            'transaction_type' => CashBook::TRANSACTION_TRANSFER_IN,
            'reference_number' => $referenceNumber,
            'transfer_id' => $transfer->transferid,
            'description' => $transfer->details ?? 'Transfer from ' . $sourceAccount->accountName,
            'amount' => $transfer->amountreceived, // Positive for inflow/debit
            'balance_before' => $destinationBalanceBefore,
            'balance_after' => $destinationAccount->current_balance,
            'currency' => $transfer->currency,
            'exchange_rate' => $transfer->exchangerate,
            'status' => 'posted',
            'created_by' => auth()->id(),
            'updated_by' => auth()->id()
        ]);
    }

    /**
     * Post transfer to General Ledger
     */
    private function postToGeneralLedger($transfer, $sourceAccount, $destinationAccount)
    {
        // Get the user's company ID and branch ID
        $companyId = auth()->user()->companyid;
        $branchId = auth()->user()->branchid ?? 1;
        
        // Generate reference number
        $referenceNumber = $transfer->reference ?? 'TRF-' . str_pad($transfer->transferid, 6, '0', STR_PAD_LEFT);
        
        // Find GL accounts for cash/bank accounts
        $sourceGLAccountId = $sourceAccount->gl_account_id;
        $destinationGLAccountId = $destinationAccount->gl_account_id;
        
        // If no GL accounts are linked, try to find them by account type
        if (!$sourceGLAccountId) {
            $sourceGLAccountId = $this->findGLAccountByType($sourceAccount->accountType, $companyId);
        }
        
        if (!$destinationGLAccountId) {
            $destinationGLAccountId = $this->findGLAccountByType($destinationAccount->accountType, $companyId);
        }

        // Check if GL accounts were found
        if (!$sourceGLAccountId || !$destinationGLAccountId) {
            throw new \Exception('GL accounts not configured for cash/bank accounts. Please link GL accounts to cash/bank accounts.');
        }

        // Debit Entry (Destination Account) - INFLOW
        GeneralLedgerEntry::create([
            'entrydate' => $transfer->transactiondate,
            'description' => $transfer->details ?? 'Internal transfer from ' . $sourceAccount->accountName,
            'accountid' => $destinationGLAccountId,
            'referencedocument' => 'INTERNAL_TRANSFER',
            'reference_type' => 'internal_transfer',
            'reference_id' => $transfer->transferid,
            'documentno' => $referenceNumber,
            'entrytype' => 'debit',
            'amount' => $transfer->amountreceived,
            'createdby' => auth()->id(),
            'companyid' => $companyId,
            'branchid' => $branchId,
            'original_currency' => $transfer->currency,
            'transtype' => 'cash_transfer'
        ]);

        // Credit Entry (Source Account) - OUTFLOW
        GeneralLedgerEntry::create([
            'entrydate' => $transfer->transactiondate,
            'description' => $transfer->details ?? 'Internal transfer to ' . $destinationAccount->accountName,
            'accountid' => $sourceGLAccountId,
            'referencedocument' => 'INTERNAL_TRANSFER',
            'reference_type' => 'internal_transfer',
            'reference_id' => $transfer->transferid,
            'documentno' => $referenceNumber,
            'entrytype' => 'credit',
            'amount' => $transfer->amountsent,
            'createdby' => auth()->id(),
            'companyid' => $companyId,
            'branchid' => $branchId,
            'original_currency' => $transfer->currency,
            'transtype' => 'cash_transfer'
        ]);
    }

    /**
     * Find GL account by cash/bank account type
     */
    private function findGLAccountByType($accountType, $companyId)
    {
        // Map cash/bank account types to GL account names
        $mapping = [
            'Bank' => ['Bank', 'Current Account', 'Savings Account', 'Bank Account'],
            'Cash' => ['Cash', 'Cash in Hand', 'Petty Cash', 'Cash Account'],
            'Mobile Money' => ['Mobile Money', 'Airtel Money', 'MTN Money', 'Zamtel Kwacha', 'Mobile Wallet'],
            'Petty Cash' => ['Petty Cash', 'Cash', 'Petty Cash Account'],
            'Credit Card' => ['Credit Card', 'Card Account'],
        ];

        $searchTerms = $mapping[$accountType] ?? [$accountType];
        
        // Search for the GL account
        $glAccount = ChartOfAccount::where('company_id', $companyId)
            ->where(function($query) use ($searchTerms) {
                foreach ($searchTerms as $term) {
                    $query->orWhere('name', 'like', "%{$term}%");
                }
            })
            ->where('is_active', true)
            ->first();

        if (!$glAccount) {
            // Try to find any asset account
            $glAccount = ChartOfAccount::where('company_id', $companyId)
                ->whereHas('accountType', function($query) {
                    $query->where('normal_balance', 'debit'); // Asset accounts
                })
                ->where('is_active', true)
                ->first();
        }

        return $glAccount ? $glAccount->id : null;
    }
}