<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\LoanType;
use App\Models\CreditScoreTier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class OfferController extends Controller
{
    public function index(Request $request)
    {
        $query = DB::table('product_credit_score_tier as pct')
            ->join('loantype as p', 'pct.product_id', '=', 'p.id')
            ->join('credit_score_tiers as cst', 'pct.credit_score_tier_id', '=', 'cst.id')
            ->select(
                'pct.id',
                'p.product as product_name',
                'cst.name as credit_score_tier',
                'cst.min_score',
                'cst.max_score',
                'pct.daily_rate',
                'pct.monthly_rate',
                'pct.annual_rate',
                'pct.min_amount',
                'pct.max_amount',
                'pct.min_duration', 
                'pct.max_duration', 
                'pct.is_active',
                'pct.is_finalized', 
                'p.status as product_status'
            );

        // Filter by active status if requested
        if ($request->has('status') && $request->status !== 'all') {
            if ($request->status === 'active') {
                $query->where('pct.is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('pct.is_active', false);
            }
        } 

        // Filter by product if requested
        if ($request->has('product_id')) {
            $query->where('pct.product_id', $request->product_id);
        }

        // Filter by credit score tier if requested
        if ($request->has('tier_id')) {
            $query->where('pct.credit_score_tier_id', $request->tier_id);
        }

        $offers = $query->orderBy('p.product')
                        ->orderBy('cst.min_score', 'desc')
                        ->paginate(20)
                        ->withQueryString();

        $stats = [
            'total' => DB::table('product_credit_score_tier')->count(),
            'active' => DB::table('product_credit_score_tier')->where('is_active', true)->count(),
            'inactive' => DB::table('product_credit_score_tier')->where('is_active', false)->count(),
        ];

        $products = LoanType::where('status', 'Active')->orderBy('product')->get();
        $creditScoreTiers = CreditScoreTier::active()->ordered()->get();

        return view('accounting.offers.index', [
            'offers' => $offers,
            'stats' => $stats,
            'products' => $products,
            'creditScoreTiers' => $creditScoreTiers,
            'filters' => $request->only(['status', 'product_id', 'tier_id']),
        ]);
    }


    public function create()
    {
        $products = LoanType::where('status', 'Active')->orderBy('product')->get();
        $creditScoreTiers = CreditScoreTier::active()->ordered()->get();

        return view('accounting.offers.create', [
            'products' => $products,
            'creditScoreTiers' => $creditScoreTiers,
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_id' => 'required|exists:loantype,id',
            'credit_score_tier_id' => 'required|exists:credit_score_tiers,id',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'required|numeric|gt:min_amount',
            'min_duration' => 'required|integer|min:1', 
            'max_duration' => 'required|integer|gt:min_duration', 
            'daily_rate' => 'nullable|numeric|min:0|max:100',
        ]);

        // Check if this product-tier combination already exists
        $exists = DB::table('product_credit_score_tier')
            ->where('product_id', $request->product_id)
            ->where('credit_score_tier_id', $request->credit_score_tier_id)
            ->exists();

        if ($exists) {
            return redirect()->back()
                ->withErrors(['credit_score_tier_id' => 'This product already has an offer for the selected credit score tier.'])
                ->withInput();
        }

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::table('product_credit_score_tier')->insert([
                'product_id' => $request->product_id,
                'credit_score_tier_id' => $request->credit_score_tier_id,
                'min_amount' => $request->min_amount,
                'max_amount' => $request->max_amount,
                'min_duration' => $request->min_duration, 
                'max_duration' => $request->max_duration, 
                'daily_rate' => $request->daily_rate,
                'monthly_rate' => 0, 
                'annual_rate' => 0,  
                'is_active' => false, 
                'is_finalized' => false,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return redirect()->route('admin.accounting.offers.index')
                ->with('success', 'Offer created successfully!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error creating offer: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function edit($id)
    {
        $offer = DB::table('product_credit_score_tier')->where('id', $id)->first();

        if (!$offer) {
            return redirect()->route('admin.accounting.offers.index')
                ->with('error', 'Offer not found.');
        }
        
        // RESTRICTION: Cannot edit if finalized
        if (isset($offer->is_finalized) && $offer->is_finalized) {
             return redirect()->route('admin.accounting.offers.index')
                ->with('error', 'Activated offers cannot be edited.');
        }

        $products = LoanType::where('status', 'Active')->orderBy('product')->get();
        $creditScoreTiers = CreditScoreTier::active()->ordered()->get();

        return view('accounting.offers.edit', [
            'offer' => $offer,
            'products' => $products,
            'creditScoreTiers' => $creditScoreTiers,
        ]);
    }

    public function update(Request $request, $id)
    {
        $offer = DB::table('product_credit_score_tier')->where('id', $id)->first();
        
        // RESTRICTION: Cannot update if finalized
        if (isset($offer->is_finalized) && $offer->is_finalized) {
            return redirect()->route('admin.accounting.offers.index')
               ->with('error', 'Activated offers cannot be updated.');
        }

        $validator = Validator::make($request->all(), [
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'required|numeric|gt:min_amount',
            'min_duration' => 'required|integer|min:1', 
            'max_duration' => 'required|integer|gt:min_duration', 
            'daily_rate' => 'nullable|numeric|min:0|max:100',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::table('product_credit_score_tier')
                ->where('id', $id)
                ->update([
                    'min_amount' => $request->min_amount,
                    'max_amount' => $request->max_amount,
                    'min_duration' => $request->min_duration, 
                    'max_duration' => $request->max_duration, 
                    'daily_rate' => $request->daily_rate,
                    'monthly_rate' => 0,
                    'annual_rate' => 0,
                    'is_active' => $request->has('is_active'),
                    'updated_at' => now(),
                ]);

            return redirect()->route('admin.accounting.offers.index')
                ->with('success', 'Offer updated successfully!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error updating offer: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function destroy($id)
    {
        $offer = DB::table('product_credit_score_tier')->where('id', $id)->first();
        
        // RESTRICTION: Cannot delete if finalized
        if (isset($offer->is_finalized) && $offer->is_finalized) {
            return redirect()->route('admin.accounting.offers.index')
               ->with('error', 'Activated offers cannot be deleted.');
        }
        
        try {
            DB::table('product_credit_score_tier')->where('id', $id)->delete();

            return redirect()->route('admin.accounting.offers.index')
                ->with('success', 'Offer deleted successfully!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error deleting offer: ' . $e->getMessage());
        }
    }

    public function toggleStatus($id)
    {
        try {
            $offer = DB::table('product_credit_score_tier')->where('id', $id)->first();

            if (!$offer) {
                return response()->json(['success' => false, 'message' => 'Offer not found'], 404);
            }
            
            // RESTRICTION: Cannot toggle status if finalized
            if (isset($offer->is_finalized) && $offer->is_finalized) {
                 return response()->json(['success' => false, 'message' => 'Activated offers cannot have their status toggled.'], 403);
            }


            DB::table('product_credit_score_tier')
                ->where('id', $id)
                ->update([
                    'is_active' => !$offer->is_active,
                    'updated_at' => now(),
                ]);

            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }
    
    /**
     * Show the activation confirmation form.
     */
    public function activateForm($id)
    {
        $offer = DB::table('product_credit_score_tier as pct')
            ->join('loantype as p', 'pct.product_id', '=', 'p.id')
            ->join('credit_score_tiers as cst', 'pct.credit_score_tier_id', '=', 'cst.id')
            ->select(
                'pct.id',
                'p.product as product_name',
                'cst.name as credit_score_tier',
                'cst.min_score',
                'cst.max_score',
                'pct.min_amount',
                'pct.max_amount',
                'pct.min_duration',
                'pct.max_duration',
                'pct.is_active',
                'pct.is_finalized',
                'pct.daily_rate' // ADDED: The missing field
            )
            ->where('pct.id', $id)
            ->first();

        if (!$offer) {
            return redirect()->route('admin.accounting.offers.index')
                ->with('error', 'Offer not found.');
        }

        if (isset($offer->is_finalized) && $offer->is_finalized) {
            return redirect()->route('admin.accounting.offers.index')
                ->with('error', 'Offer is already permanently activated.');
        }

        return view('accounting.offers.activate-offer', compact('offer'));
    }

    /**
     * Finalize and activate the offer.
     */
    public function activate(Request $request, $id)
    {
        $offer = DB::table('product_credit_score_tier')->where('id', $id)->first();

        if (!$offer || (isset($offer->is_finalized) && $offer->is_finalized)) {
            return redirect()->route('admin.accounting.offers.index')
                ->with('error', 'Offer cannot be activated.');
        }

        try {
            DB::table('product_credit_score_tier')
                ->where('id', $id)
                ->update([
                    'is_finalized' => true,
                    'is_active' => true,
                    'updated_at' => now(),
                ]);

            return redirect()->route('admin.accounting.offers.index')
                ->with('success', 'Offer successfully activated and finalized. It is now protected from further edits/deletions.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error during activation: ' . $e->getMessage());
        }
    }

    /**
     * Get offers for a specific customer based on their credit score
     */
    public function getOffersForCustomer($customerId)
    {
        // Get customer's current credit score
        $creditScore = DB::table('customer_credit_scores')
            ->where('customer_id', $customerId)
            ->where('is_current', true)
            ->value('credit_score');

        if (!$creditScore) {
            return collect(); // No credit score available
        }

        // Get the credit score tier for this score
        $tier = CreditScoreTier::where('min_score', '<=', $creditScore)
            ->where('max_score', '>=', $creditScore)
            ->where('is_active', true)
            ->first();

        if (!$tier) {
            return collect(); // No tier found for this score
        }

        // Get offers for this tier
        $offers = DB::table('product_credit_score_tier as pct')
            ->join('loantype as p', 'pct.product_id', '=', 'p.id')
            ->join('credit_score_tiers as cst', 'pct.credit_score_tier_id', '=', 'cst.id')
            ->where('pct.credit_score_tier_id', $tier->id)
            ->where('pct.is_active', true)
            ->where('p.status', 'Active')
            ->select(
                'p.id as product_id',
                'p.product as product_name',
                'p.details',
                'cst.name as credit_score_tier',
                'pct.daily_rate',
                'pct.monthly_rate',
                'pct.annual_rate',
                'pct.min_amount',
                'pct.max_amount',
                'p.prate',
                'p.monthlyrate',
                'p.yearlyrate'
            )
            ->orderBy('p.product')
            ->get();

        return $offers;
    }
}