<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\PayrollSetting;
use App\Models\PayeTaxBand;
use App\Models\PayrollPeriod;
use App\Models\Company;
use App\Models\Employee;
use App\Models\PayrollRun;
use App\Models\PayrollEntry;
use App\Models\FinancialYear;
use App\Models\EarningType;
use App\Models\EmployeeEarning;
use App\Models\PayrollAdjustment;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\BusinessProcessMapping; // ADDED
use App\Models\JournalEntry; // ADDED
use App\Models\ChartOfAccount; // ADDED
use App\Models\LeaveAccrualTransaction; // ADDED

class PayrollController extends Controller
{
    // Dashboard
    // Dashboard
    public function index()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $currentFinancialYear = $company->currentFinancialYear;
        
        $payeSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'paye')
            ->first();
            
        $napsaSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'napsa')
            ->first();
            
        $nhimaSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'nhima')
            ->first();
            
        $payrollPeriods = PayrollPeriod::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->orderBy('start_date', 'desc')
            ->paginate(10);

        // NEW: Get payroll account mappings
        $payrollMappings = BusinessProcessMapping::getPayrollMappings();

        return view('accounting.payroll.index', compact(
            'company',
            'currentFinancialYear',
            'payeSettings',
            'napsaSettings',
            'nhimaSettings',
            'payrollPeriods',
            'payrollMappings' // NEW
        ));
    }
    
    public function employeesEdit($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employee = Employee::with(['earnings.earningType', 'company'])
            ->where('company_id', $company->id)
            ->findOrFail($id);
        
        $totalEarnings = $employee->calculateMonthlyEarnings();
        $taxableEarnings = $employee->calculateTaxableEarnings();
        $nonTaxableEarnings = $employee->calculateNonTaxableEarnings();
        $earningsBreakdown = $employee->getEarningsBreakdown();
        
        return view('accounting.payroll.employees.edit', compact(
            'company',
            'employee',
            'totalEarnings',
            'taxableEarnings',
            'nonTaxableEarnings',
            'earningsBreakdown'
        ));
    }

    public function employeesUpdate(Request $request, $id)
    {
        $request->validate([
            'employee_number' => 'required|string|max:50|unique:employees,employee_number,' . $id,
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'email' => 'required|email|unique:employees,email,' . $id,
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'hire_date' => 'required|date',
            'employment_type' => 'required|in:permanent,contract,temporary,casual',
            'gender' => 'required|in:male,female,other',
            'national_id' => 'nullable|string|max:50',
            'napsa_number' => 'nullable|string|max:50',
            'nhima_number' => 'nullable|string|max:50',
            'payment_method' => 'required|in:bank,cash,mobile_money',
            'bank_name' => 'required_if:payment_method,bank|nullable|string|max:100',
            'bank_account_number' => 'required_if:payment_method,bank|nullable|string|max:50',
            'bank_account_name' => 'required_if:payment_method,bank|nullable|string|max:100',
            'bank_branch' => 'nullable|string|max:100',
            'earnings' => 'nullable|array',
            'earnings.*.id' => 'nullable|exists:employee_earnings,id',
            'earnings.*.earning_type_id' => 'required|exists:earning_types,id',
            'earnings.*.amount' => 'required|numeric|min:0',
            'earnings.*.frequency' => 'required|in:monthly,quarterly,annually,one-time',
            'earnings.*.effective_date' => 'required|date',
            'earnings.*.is_recurring' => 'required|boolean',
            'earnings.*.notes' => 'nullable|string',
            'earnings.*._delete' => 'nullable|boolean',
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->findOrFail($id);
        
        DB::beginTransaction();
        
        try {
            // Update employee
            $employee->update([
                'employee_number' => $request->employee_number,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'phone' => $request->phone,
                'date_of_birth' => $request->date_of_birth,
                'hire_date' => $request->hire_date,
                'employment_type' => $request->employment_type,
                'gender' => $request->gender,
                'national_id' => $request->national_id,
                'napsa_number' => $request->napsa_number,
                'nhima_number' => $request->nhima_number,
                'payment_method' => $request->payment_method,
                'bank_name' => $request->bank_name,
                'bank_account_number' => $request->bank_account_number,
                'bank_account_name' => $request->bank_account_name,
                'bank_branch' => $request->bank_branch,
                'is_active' => $request->is_active ?? true,
                'notes' => $request->notes,
                'updated_by' => auth()->id()
            ]);
            
            // Handle earnings updates
            if ($request->has('earnings')) {
                $existingEarningIds = [];
                
                foreach ($request->earnings as $earningData) {
                    // Check if earning type belongs to same company
                    $earningType = EarningType::where('id', $earningData['earning_type_id'])
                        ->where('company_id', $company->id)
                        ->first();
                    
                    if (!$earningType) {
                        throw new \Exception('Invalid earning type selected.');
                    }
                    
                    if (isset($earningData['id']) && $earningData['id']) {
                        // Update existing earning
                        $employeeEarning = EmployeeEarning::where('id', $earningData['id'])
                            ->where('employee_id', $employee->id)
                            ->first();
                        
                        if ($employeeEarning) {
                            // Check if marked for deletion
                            if (isset($earningData['_delete']) && $earningData['_delete']) {
                                $employeeEarning->delete();
                            } else {
                                $employeeEarning->update([
                                    'earning_type_id' => $earningType->id,
                                    'amount' => $earningData['amount'],
                                    'frequency' => $earningData['frequency'],
                                    'effective_date' => $earningData['effective_date'],
                                    'is_recurring' => $earningData['is_recurring'] ?? true,
                                    'notes' => $earningData['notes'] ?? null,
                                    'updated_by' => auth()->id()
                                ]);
                                $existingEarningIds[] = $employeeEarning->id;
                            }
                        }
                    } else {
                        // Create new earning (if not marked for deletion)
                        if (!(isset($earningData['_delete']) && $earningData['_delete'])) {
                            $newEarning = EmployeeEarning::create([
                                'employee_id' => $employee->id,
                                'earning_type_id' => $earningType->id,
                                'amount' => $earningData['amount'],
                                'currency' => $company->reporting_currency ?? 'ZMW',
                                'effective_date' => $earningData['effective_date'],
                                'end_date' => null,
                                'is_recurring' => $earningData['is_recurring'] ?? true,
                                'frequency' => $earningData['frequency'],
                                'notes' => $earningData['notes'] ?? null,
                                'is_active' => true,
                                'created_by' => auth()->id()
                            ]);
                            $existingEarningIds[] = $newEarning->id;
                        }
                    }
                }
                
                // Delete earnings that were removed from the form
                EmployeeEarning::where('employee_id', $employee->id)
                    ->whereNotIn('id', $existingEarningIds)
                    ->delete();
            }
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.employees.show', $employee->id)
                ->with('success', 'Employee updated successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to update employee: ' . $e->getMessage())
                ->withInput();
        }
    }

    // PAYE Tax Bands
    public function payeIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $currentFinancialYear = $company->currentFinancialYear;
        
        $payeSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'paye')
            ->first();
            
        $taxBands = [];
        if ($payeSettings) {
            $taxBands = PayeTaxBand::where('payroll_setting_id', $payeSettings->id)
                ->orderBy('band_number')
                ->get();
        }

        // Get all financial years for the company
        $financialYears = FinancialYear::where('company_id', $company->id)
            ->orderBy('start_date', 'desc')
            ->get();

        return view('accounting.payroll.paye.index', compact(
            'company',
            'currentFinancialYear',
            'payeSettings',
            'taxBands',
            'financialYears'
        ));
    }

    public function payeStore(Request $request)
    {
        $request->validate([
            'financial_year_id' => 'required|exists:financial_years,id',
            'year' => 'required|string|max:9',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'tax_bands' => 'required|array|min:1',
            'tax_bands.*.band_number' => 'required|integer',
            'tax_bands.*.description' => 'required|string|max:255',
            'tax_bands.*.monthly_from' => 'required|numeric|min:0',
            'tax_bands.*.monthly_to' => 'required|numeric|min:0',
            'tax_bands.*.annual_from' => 'required|numeric|min:0',
            'tax_bands.*.annual_to' => 'required|numeric|min:0',
            'tax_bands.*.tax_rate' => 'required|numeric|min:0|max:100',
        ]);

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        DB::beginTransaction();

        try {
            // Check if setting exists for this financial year
            $existingSetting = PayrollSetting::where('company_id', $company->id)
                ->where('financial_year_id', $request->financial_year_id)
                ->where('setting_type', 'paye')
                ->first();

            if ($existingSetting) {
                // Update existing setting
                $existingSetting->update([
                    'year' => $request->year,
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'updated_by' => auth()->id()
                ]);
                $payrollSetting = $existingSetting;
                
                // Delete existing tax bands
                PayeTaxBand::where('payroll_setting_id', $payrollSetting->id)->delete();
            } else {
                // Create new setting
                $payrollSetting = PayrollSetting::create([
                    'company_id' => $company->id,
                    'financial_year_id' => $request->financial_year_id,
                    'setting_type' => 'paye',
                    'year' => $request->year,
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'is_active' => true,
                    'created_by' => auth()->id()
                ]);
            }

            // Create tax bands
            foreach ($request->tax_bands as $band) {
                PayeTaxBand::create([
                    'payroll_setting_id' => $payrollSetting->id,
                    'company_id' => $company->id,
                    'band_number' => $band['band_number'],
                    'description' => $band['description'],
                    'monthly_from' => $band['monthly_from'],
                    'monthly_to' => $band['monthly_to'],
                    'annual_from' => $band['annual_from'],
                    'annual_to' => $band['annual_to'],
                    'tax_rate' => $band['tax_rate'],
                    'is_active' => true,
                    'created_by' => auth()->id()
                ]);
            }

            DB::commit();

            return redirect()->route('admin.accounting.payroll.paye.index')
                ->with('success', 'PAYE tax bands updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to update PAYE tax bands: ' . $e->getMessage())
                ->withInput();
        }
    }

    // NAPSA Settings
    public function nAPSAIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $currentFinancialYear = $company->currentFinancialYear;
        
        $napsaSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'napsa')
            ->first();

        $financialYears = FinancialYear::where('company_id', $company->id)
            ->orderBy('start_date', 'desc')
            ->get();

        return view('accounting.payroll.napsa.index', compact(
            'company',
            'currentFinancialYear',
            'napsaSettings',
            'financialYears'
        ));
    }

    public function nAPSAStore(Request $request)
    {
        $request->validate([
            'financial_year_id' => 'required|exists:financial_years,id',
            'year' => 'required|string|max:9',
            'monthly_rate' => 'required|numeric|min:0|max:100',
            'monthly_ceiling' => 'required|numeric|min:0',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
        ]);

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        // Calculate annual values
        $annual_ceiling = $request->monthly_ceiling * 12;

        DB::beginTransaction();

        try {
            // Check if setting exists for this financial year
            $existingSetting = PayrollSetting::where('company_id', $company->id)
                ->where('financial_year_id', $request->financial_year_id)
                ->where('setting_type', 'napsa')
                ->first();

            if ($existingSetting) {
                // Update existing setting
                $existingSetting->update([
                    'year' => $request->year,
                    'monthly_rate' => $request->monthly_rate,
                    'annual_rate' => $request->monthly_rate,
                    'monthly_ceiling' => $request->monthly_ceiling,
                    'annual_ceiling' => $annual_ceiling,
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'updated_by' => auth()->id()
                ]);

                $message = 'NAPSA settings updated successfully.';
            } else {
                // Create new setting
                PayrollSetting::create([
                    'company_id' => $company->id,
                    'financial_year_id' => $request->financial_year_id,
                    'setting_type' => 'napsa',
                    'year' => $request->year,
                    'monthly_rate' => $request->monthly_rate,
                    'annual_rate' => $request->monthly_rate,
                    'monthly_ceiling' => $request->monthly_ceiling,
                    'annual_ceiling' => $annual_ceiling,
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'is_active' => true,
                    'created_by' => auth()->id()
                ]);

                $message = 'NAPSA settings created successfully.';
            }

            DB::commit();

            return redirect()->route('admin.accounting.payroll.napsa.index')
                ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to save NAPSA settings: ' . $e->getMessage())
                ->withInput();
        }
    }

    // NHIMA Settings
    public function nHIMAIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $currentFinancialYear = $company->currentFinancialYear;
        
        $nhimaSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'nhima')
            ->first();

        $financialYears = FinancialYear::where('company_id', $company->id)
            ->orderBy('start_date', 'desc')
            ->get();

        return view('accounting.payroll.nhima.index', compact(
            'company',
            'currentFinancialYear',
            'nhimaSettings',
            'financialYears'
        ));
    }

    public function nHIMAStore(Request $request)
    {
        $request->validate([
            'financial_year_id' => 'required|exists:financial_years,id',
            'year' => 'required|string|max:9',
            'monthly_rate' => 'required|numeric|min:0|max:100',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
        ]);

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        DB::beginTransaction();

        try {
            // Check if setting exists for this financial year
            $existingSetting = PayrollSetting::where('company_id', $company->id)
                ->where('financial_year_id', $request->financial_year_id)
                ->where('setting_type', 'nhima')
                ->first();

            if ($existingSetting) {
                // Update existing setting
                $existingSetting->update([
                    'year' => $request->year,
                    'monthly_rate' => $request->monthly_rate,
                    'annual_rate' => $request->monthly_rate,
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'updated_by' => auth()->id()
                ]);

                $message = 'NHIMA settings updated successfully.';
            } else {
                // Create new setting
                PayrollSetting::create([
                    'company_id' => $company->id,
                    'financial_year_id' => $request->financial_year_id,
                    'setting_type' => 'nhima',
                    'year' => $request->year,
                    'monthly_rate' => $request->monthly_rate,
                    'annual_rate' => $request->monthly_rate,
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'is_active' => true,
                    'created_by' => auth()->id()
                ]);

                $message = 'NHIMA settings created successfully.';
            }

            DB::commit();

            return redirect()->route('admin.accounting.payroll.nhima.index')
                ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to save NHIMA settings: ' . $e->getMessage())
                ->withInput();
        }
    }

    // Payroll Periods
    public function periodsIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $currentFinancialYear = $company->currentFinancialYear;
        
        $payrollPeriods = PayrollPeriod::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->orderBy('start_date', 'desc')
            ->paginate(20);

        $financialYears = FinancialYear::where('company_id', $company->id)
            ->orderBy('start_date', 'desc')
            ->get();

        return view('accounting.payroll.periods.index', compact(
            'company',
            'currentFinancialYear',
            'payrollPeriods',
            'financialYears'
        ));
    }

    public function periodsStore(Request $request)
    {
        $request->validate([
            'financial_year_id' => 'required|exists:financial_years,id',
            'period_name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'payment_date' => 'required|date|after_or_equal:end_date',
            'cutoff_date' => 'required|date|after_or_equal:start_date|before_or_equal:end_date',
        ]);

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        try {
            PayrollPeriod::create([
                'company_id' => $company->id,
                'financial_year_id' => $request->financial_year_id,
                'period_name' => $request->period_name,
                'period_type' => 'monthly',
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'payment_date' => $request->payment_date,
                'cutoff_date' => $request->cutoff_date,
                'status' => 'open',
                'is_active' => true,
                'created_by' => auth()->id()
            ]);

            return redirect()->route('admin.accounting.payroll.periods.index')
                ->with('success', 'Payroll period created successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create payroll period: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function generatePeriods(Request $request)
    {
        $request->validate([
            'financial_year_id' => 'required|exists:financial_years,id'
        ]);

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $financialYear = FinancialYear::findOrFail($request->financial_year_id);

        // Check if periods already exist for this financial year
        $existingPeriods = PayrollPeriod::where('company_id', $company->id)
            ->where('financial_year_id', $financialYear->id)
            ->count();

        if ($existingPeriods > 0) {
            return redirect()->back()
                ->with('error', 'Payroll periods already exist for this financial year. Please delete existing periods first.');
        }

        try {
            $periods = PayrollPeriod::generateMonthlyPeriods($company->id, $financialYear->id, $financialYear);
            
            // Insert all periods at once
            PayrollPeriod::insert($periods);

            return redirect()->route('admin.accounting.payroll.periods.index')
                ->with('success', 'Monthly payroll periods generated successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to generate periods: ' . $e->getMessage());
        }
    }

    public function periodsUpdate(Request $request, $id)
    {
        $request->validate([
            'period_name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'payment_date' => 'required|date|after_or_equal:end_date',
            'cutoff_date' => 'required|date|after_or_equal:start_date|before_or_equal:end_date',
            'status' => 'required|in:open,closed,processing',
        ]);

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($id);

        try {
            $period->update([
                'period_name' => $request->period_name,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'payment_date' => $request->payment_date,
                'cutoff_date' => $request->cutoff_date,
                'status' => $request->status,
                'updated_by' => auth()->id()
            ]);

            return redirect()->route('admin.accounting.payroll.periods.index')
                ->with('success', 'Payroll period updated successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update payroll period: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function periodsDestroy($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($id);

        try {
            $period->delete();

            return redirect()->route('admin.accounting.payroll.periods.index')
                ->with('success', 'Payroll period deleted successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete payroll period: ' . $e->getMessage());
        }
    }

    // Calculate PAYE
    public function calculatePAYE(Request $request)
    {
        $request->validate([
            'monthly_salary' => 'required|numeric|min:0',
            'financial_year_id' => 'required|exists:financial_years,id'
        ]);

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        $payeSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $request->financial_year_id)
            ->where('setting_type', 'paye')
            ->first();

        if (!$payeSettings) {
            return response()->json([
                'success' => false,
                'message' => 'No PAYE settings found for the selected financial year.'
            ], 404);
        }

        $taxBands = PayeTaxBand::where('payroll_setting_id', $payeSettings->id)
            ->orderBy('band_number')
            ->get();

        $monthlySalary = $request->monthly_salary;
        $totalTax = 0;
        $bandCalculations = [];

        foreach ($taxBands as $band) {
            if ($monthlySalary >= $band->monthly_from) {
                $taxableAmount = min($monthlySalary, $band->monthly_to) - $band->monthly_from;
                $bandTax = ($taxableAmount > 0) ? ($taxableAmount * $band->tax_rate) / 100 : 0;
                $totalTax += $bandTax;

                $bandCalculations[] = [
                    'band' => $band->description,
                    'range' => $band->monthly_range,
                    'taxable_amount' => number_format($taxableAmount, 2),
                    'rate' => number_format($band->tax_rate, 2) . '%',
                    'tax' => number_format($bandTax, 2)
                ];
            }
        }

        return response()->json([
            'success' => true,
            'monthly_salary' => number_format($monthlySalary, 2),
            'total_tax' => number_format($totalTax, 2),
            'net_salary' => number_format($monthlySalary - $totalTax, 2),
            'band_calculations' => $bandCalculations
        ]);
    }

    // Get settings for financial year
    public function getSettingsForYear(Request $request)
    {
        $request->validate([
            'financial_year_id' => 'required|exists:financial_years,id',
            'setting_type' => 'required|in:paye,napsa,nhima'
        ]);

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        $setting = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $request->financial_year_id)
            ->where('setting_type', $request->setting_type)
            ->first();

        if ($setting && $request->setting_type === 'paye') {
            $taxBands = PayeTaxBand::where('payroll_setting_id', $setting->id)
                ->orderBy('band_number')
                ->get();

            return response()->json([
                'success' => true,
                'setting' => $setting,
                'tax_bands' => $taxBands
            ]);
        }

        return response()->json([
            'success' => true,
            'setting' => $setting
        ]);
    }

    public function periodsEdit($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'period' => $period
        ]);
    }

    public function periodsClose($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($id);

        try {
            $period->closePeriod();

            return response()->json([
                'success' => true,
                'message' => 'Payroll period closed successfully.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to close period: ' . $e->getMessage()
            ], 500);
        }
    }

    public function periodsReopen($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($id);

        try {
            $period->reopenPeriod();

            return response()->json([
                'success' => true,
                'message' => 'Payroll period reopened successfully.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to reopen period: ' . $e->getMessage()
            ], 500);
        }
    }
    
    // Employee Management
    public function employeesIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employees = Employee::where('company_id', $company->id)
            ->orderBy('first_name')
            ->paginate(20);
        
        return view('accounting.payroll.employees.index', compact('company', 'employees'));
    }

    public function employeesCreate()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        return view('accounting.payroll.employees.create', compact('company'));
    }

    public function employeesStore(Request $request)
    {
        $request->validate([
            'employee_number' => 'required|string|max:50|unique:employees,employee_number',
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'email' => 'required|email|unique:employees,email',
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'hire_date' => 'required|date',
            'employment_type' => 'required|in:permanent,contract,temporary,casual',
            'gender' => 'required|in:male,female,other',
            'national_id' => 'nullable|string|max:50',
            'napsa_number' => 'nullable|string|max:50',
            'nhima_number' => 'nullable|string|max:50',
            'payment_method' => 'required|in:bank,cash,mobile_money',
            'bank_name' => 'required_if:payment_method,bank|nullable|string|max:100',
            'bank_account_number' => 'required_if:payment_method,bank|nullable|string|max:50',
            'bank_account_name' => 'required_if:payment_method,bank|nullable|string|max:100',
            'bank_branch' => 'nullable|string|max:100',
            'earnings' => 'required|array|min:1',
            'earnings.*.earning_type_id' => 'required|exists:earning_types,id',
            'earnings.*.amount' => 'required|numeric|min:0',
            'earnings.*.frequency' => 'required|in:monthly,quarterly,annually,one-time',
            'earnings.*.effective_date' => 'required|date',
            'earnings.*.is_recurring' => 'required|boolean',
            'earnings.*.notes' => 'nullable|string',
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        DB::beginTransaction();
        
        try {
            // Create employee
            $employee = Employee::create([
                'company_id' => $company->id,
                'employee_number' => $request->employee_number,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->email,
                'phone' => $request->phone,
                'date_of_birth' => $request->date_of_birth,
                'hire_date' => $request->hire_date,
                'employment_type' => $request->employment_type,
                'gender' => $request->gender,
                'national_id' => $request->national_id,
                'napsa_number' => $request->napsa_number,
                'nhima_number' => $request->nhima_number,
                'payment_method' => $request->payment_method,
                'bank_name' => $request->bank_name,
                'bank_account_number' => $request->bank_account_number,
                'bank_account_name' => $request->bank_account_name,
                'bank_branch' => $request->bank_branch,
                'is_active' => $request->is_active ?? true,
                'notes' => $request->notes,
                'created_by' => auth()->id()
            ]);
            
            // Create employee earnings
            foreach ($request->earnings as $earningData) {
                // Check if earning type belongs to same company
                $earningType = EarningType::where('id', $earningData['earning_type_id'])
                    ->where('company_id', $company->id)
                    ->first();
                
                if (!$earningType) {
                    throw new \Exception('Invalid earning type selected.');
                }
                
                EmployeeEarning::create([
                    'employee_id' => $employee->id,
                    'earning_type_id' => $earningType->id,
                    'amount' => $earningData['amount'],
                    'currency' => $company->reporting_currency ?? 'ZMW',
                    'effective_date' => $earningData['effective_date'],
                    'end_date' => null,
                    'is_recurring' => $earningData['is_recurring'] ?? true,
                    'frequency' => $earningData['frequency'],
                    'notes' => $earningData['notes'] ?? null,
                    'is_active' => true,
                    'created_by' => auth()->id()
                ]);
            }
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.employees.show', $employee->id)
                ->with('success', 'Employee created successfully with earnings.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to create employee: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function employeesDestroy($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->findOrFail($id);
        
        try {
            // Check if employee has payroll entries
            $hasPayrollEntries = PayrollEntry::where('employee_id', $employee->id)->exists();
            
            if ($hasPayrollEntries) {
                return redirect()->back()
                    ->with('error', 'Cannot delete employee with payroll history. Please deactivate instead.');
            }
            
            $employee->delete();
            
            return redirect()->route('admin.accounting.payroll.employees.index')
                ->with('success', 'Employee deleted successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete employee: ' . $e->getMessage());
        }
    }

    // Payroll Processing Methods
    public function processingIndex(Request $request)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $currentFinancialYear = $company->currentFinancialYear;
        
        $status = $request->get('status', 'all'); // Get status from query parameter
        
        // Base query
        $query = PayrollEntry::where('company_id', $company->id)
            ->with(['employee', 'payrollPeriod', 'payrollRun']);
        
        // Apply status filter
        switch ($status) {
            case 'calculated':
                $query->where('calculation_status', 'calculated')
                      ->where('payment_status', 'pending');
                $title = "Payroll Pending Review";
                $description = "Payroll entries that have been calculated and need approval";
                break;
                
            case 'approved':
                $query->where('calculation_status', 'approved')
                      ->where('payment_status', 'pending');
                $title = "Approved Payroll";
                $description = "Payroll entries that are approved and ready for payment";
                break;
                
            case 'paid':
                $query->where('calculation_status', 'approved')
                      ->where('payment_status', 'paid')
                      ->where('updated_at', '>=', now()->subDays(30));
                $title = "Paid Payroll";
                $description = "Payroll entries that have been paid in the last 30 days";
                break;
                
            default:
                $title = "All Payroll Processing";
                $description = "All payroll entries";
                break;
        }
        
        $payrollEntries = $query->orderBy('created_at', 'desc')
            ->paginate(20);
        
        // Get counts for all statuses
        $counts = $this->getPayrollMenuCounts();
        
        return view('accounting.payroll.processing.index', compact(
            'company',
            'currentFinancialYear',
            'payrollEntries',
            'title',
            'description',
            'status',
            'counts'
        ));
    }

    public function processingCreate($periodId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        // Check if payroll already exists for this period
        $existingRun = PayrollRun::where('company_id', $company->id)
            ->where('payroll_period_id', $periodId)
            ->first();
        
        if ($existingRun) {
            return redirect()->route('admin.accounting.payroll.processing.review', $existingRun->id)
                ->with('info', 'Payroll run already exists for this period.');
        }
        
        $employees = Employee::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('first_name')
            ->get();
        
        $currentFinancialYear = $company->currentFinancialYear;
        $payeSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'paye')
            ->first();
        
        $napsaSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'napsa')
            ->first();
        
        $nhimaSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'nhima')
            ->first();
        
        return view('accounting.payroll.processing.create', compact(
            'company',
            'period',
            'employees',
            'payeSettings',
            'napsaSettings',
            'nhimaSettings'
        ));
    }

    public function processingCalculate(Request $request)
    {
        $request->validate([
            'payroll_period_id' => 'required|exists:payroll_periods,id',
            'selected_employees' => 'required|array|min:1',
            'selected_employees.*' => 'exists:employees,id'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($request->payroll_period_id);
        
        // Get payroll settings
        $currentFinancialYear = $company->currentFinancialYear;
        $payeSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'paye')
            ->first();
        
        $napsaSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'napsa')
            ->first();
        
        $nhimaSettings = PayrollSetting::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->where('setting_type', 'nhima')
            ->first();
        
        if (!$payeSettings || !$napsaSettings || !$nhimaSettings) {
            return redirect()->back()
                ->with('error', 'Please configure PAYE, NAPSA, and NHIMA settings before processing payroll.')
                ->withInput();
        }
        
        // Get PAYE tax bands
        $taxBands = PayeTaxBand::where('payroll_setting_id', $payeSettings->id)
            ->orderBy('band_number')
            ->get();
        
        DB::beginTransaction();
        
        try {
            // Create payroll run
            $runNumber = 'PR' . date('Ymd') . str_pad(PayrollRun::where('company_id', $company->id)->count() + 1, 4, '0', STR_PAD_LEFT);
            
            $payrollRun = PayrollRun::create([
                'company_id' => $company->id,
                'payroll_period_id' => $period->id,
                'run_number' => $runNumber,
                'run_date' => now(),
                'status' => 'draft',
                'created_by' => auth()->id()
            ]);
            
            $totalGross = 0;
            $totalTaxableEarnings = 0;
            $totalNonTaxableEarnings = 0;
            $totalPaye = 0;
            $totalNapsaEmployee = 0;
            $totalNapsaEmployer = 0;
            $totalNhimaEmployee = 0;
            $totalNhimaEmployer = 0;
            $totalOtherDeductions = 0;
            $totalNetPay = 0;
            
            // Process each selected employee
            foreach ($request->selected_employees as $employeeId) {
                $employee = Employee::with('activeEarnings.earningType')
                    ->findOrFail($employeeId);
                
                // Calculate regular earnings using new system
                $grossSalary = $employee->calculateMonthlyEarnings();
                $taxableEarnings = $employee->calculateTaxableEarnings();
                $nonTaxableEarnings = $employee->calculateNonTaxableEarnings();
                $earningsBreakdown = $employee->getEarningsBreakdown();
                
                // Get payroll adjustments for this employee and period
                $adjustments = PayrollAdjustment::where('company_id', $company->id)
                    ->where('payroll_period_id', $period->id)
                    ->where('employee_id', $employee->id)
                    ->whereNotNull('approved_by') // Only approved adjustments
                    ->with('earningType')
                    ->get();
                
                $adjustmentDetails = [];
                $adjustmentTotal = 0;
                $adjustmentTaxable = 0;
                $adjustmentNonTaxable = 0;
                
                foreach ($adjustments as $adjustment) {
                    $adjustmentAmount = $adjustment->amount;
                    $adjustmentTotal += $adjustmentAmount;
                    
                    if ($adjustment->is_taxable) {
                        $adjustmentTaxable += $adjustmentAmount;
                    } else {
                        $adjustmentNonTaxable += $adjustmentAmount;
                    }
                    
                    $adjustmentDetails[] = [
                        'type' => $adjustment->earningType->name,
                        'code' => $adjustment->earningType->code,
                        'description' => $adjustment->description,
                        'amount' => $adjustmentAmount,
                        'taxable' => $adjustment->is_taxable,
                        'one_time' => $adjustment->is_one_time,
                        'notes' => $adjustment->notes
                    ];
                }
                
                // Add adjustments to totals
                $grossSalary += $adjustmentTotal;
                $taxableEarnings += $adjustmentTaxable;
                $nonTaxableEarnings += $adjustmentNonTaxable;
                
                // Calculate PAYE on taxable earnings only
                $payeTax = $this->calculatePAYETax($taxableEarnings, $taxBands);
                
                // Calculate NAPSA (Employee contribution: 5%, Employer contribution: 5% up to ceiling)
                $napsaEmployee = 0;
                $napsaEmployer = 0;
                $basicSalary = $employee->getEarningsByType('basic');
                
                // UPDATED: Calculate NAPSA for all employees regardless of napsa_number
                if ($napsaSettings && $basicSalary > 0) {
                    // Employee contribution: 5% of basic salary
                    $napsaEmployee = min($basicSalary * ($napsaSettings->monthly_rate / 100), $napsaSettings->monthly_ceiling);
                    
                    // Employer contribution: 5% of basic salary up to ceiling
                    $napsaEmployer = min($basicSalary * ($napsaSettings->monthly_rate / 100), $napsaSettings->monthly_ceiling);
                }
                
                // Calculate NHIMA (Employee contribution: 1%, Employer contribution: 1%)
                $nhimaEmployee = 0;
                $nhimaEmployer = 0;
                $grossEarnings = $grossSalary;
                
                // UPDATED: Calculate NHIMA for all employees regardless of nhima_number
                if ($nhimaSettings && $grossEarnings > 0) {
                    $nhimaEmployee = $grossEarnings * ($nhimaSettings->monthly_rate / 100);
                    $nhimaEmployer = $grossEarnings * ($nhimaSettings->monthly_rate / 100);
                }
                
                // Calculate other deductions (you can expand this)
                $otherDeductions = 0;
                
                $totalDeductions = $payeTax + $napsaEmployee + $nhimaEmployee + $otherDeductions;
                $netPay = $grossSalary - $totalDeductions;
                
                // Create payroll entry with adjustments included
                PayrollEntry::create([
                    'payroll_run_id' => $payrollRun->id,
                    'employee_id' => $employee->id,
                    'payroll_period_id' => $period->id,
                    'gross_salary' => $grossSalary,
                    'taxable_earnings' => $taxableEarnings,
                    'non_taxable_earnings' => $nonTaxableEarnings,
                    'paye_tax' => $payeTax,
                    'napsa_employee' => $napsaEmployee,
                    'napsa_employer' => $napsaEmployer,
                    'nhima_employee' => $nhimaEmployee,
                    'nhima_employer' => $nhimaEmployer,
                    'other_deductions' => $otherDeductions,
                    'total_deductions' => $totalDeductions,
                    'net_pay' => $netPay,
                    'earnings_breakdown' => json_encode([
                        'regular_breakdown' => $earningsBreakdown,
                        'regular_details' => $employee->activeEarnings->map(function($earning) {
                            return [
                                'type' => $earning->earningType->name,
                                'code' => $earning->earningType->code,
                                'amount' => $earning->getMonthlyAmount(),
                                'taxable' => $earning->earningType->is_taxable,
                                'statutory' => $earning->earningType->is_statutory
                            ];
                        })->toArray(),
                        'adjustments' => $adjustmentDetails,
                        'adjustment_total' => $adjustmentTotal,
                        'adjustment_taxable' => $adjustmentTaxable,
                        'adjustment_non_taxable' => $adjustmentNonTaxable
                    ]),
                    'payment_status' => 'pending'
                ]);
                
                // Update totals
                $totalGross += $grossSalary;
                $totalTaxableEarnings += $taxableEarnings;
                $totalNonTaxableEarnings += $nonTaxableEarnings;
                $totalPaye += $payeTax;
                $totalNapsaEmployee += $napsaEmployee;
                $totalNapsaEmployer += $napsaEmployer;
                $totalNhimaEmployee += $nhimaEmployee;
                $totalNhimaEmployer += $nhimaEmployer;
                $totalOtherDeductions += $otherDeductions;
                $totalNetPay += $netPay;
            }
            
            // Update payroll run with totals
            $payrollRun->update([
                'total_gross_salary' => $totalGross,
                'total_paye_tax' => $totalPaye,
                'total_napsa_employee' => $totalNapsaEmployee,
                'total_napsa_employer' => $totalNapsaEmployer,
                'total_nhima_employee' => $totalNhimaEmployee,
                'total_nhima_employer' => $totalNhimaEmployer,
                'total_other_deductions' => $totalOtherDeductions,
                'total_net_pay' => $totalNetPay,
                'status' => 'calculated'
            ]);
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.processing.review', $payrollRun->id)
                ->with('success', 'Payroll calculated successfully. Please review before approval.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to calculate payroll: ' . $e->getMessage())
                ->withInput();
        }
    }

    // Update employee show method to include payroll entries
    public function employeesShow($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employee = Employee::with(['activeEarnings.earningType', 'company'])
            ->where('company_id', $company->id)
            ->findOrFail($id);
        
        // Get payroll entries for this employee
        $payrollEntries = PayrollEntry::where('employee_id', $id)
            ->with(['payrollRun', 'payrollPeriod'])
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();
        
        $totalEarnings = $employee->calculateMonthlyEarnings();
        $taxableEarnings = $employee->calculateTaxableEarnings();
        $nonTaxableEarnings = $employee->calculateNonTaxableEarnings();
        $earningsBreakdown = $employee->getEarningsBreakdown();
        
        return view('accounting.payroll.employees.show', compact(
            'company',
            'employee',
            'payrollEntries',
            'totalEarnings',
            'taxableEarnings',
            'nonTaxableEarnings',
            'earningsBreakdown'
        ));
    }

    private function calculatePAYETax($monthlySalary, $taxBands)
    {
        $totalTax = 0;
        
        foreach ($taxBands as $band) {
            if ($monthlySalary >= $band->monthly_from) {
                $taxableAmount = min($monthlySalary, $band->monthly_to) - $band->monthly_from;
                $bandTax = ($taxableAmount > 0) ? ($taxableAmount * $band->tax_rate) / 100 : 0;
                $totalTax += $bandTax;
            }
        }
        
        return $totalTax;
    }

    public function processingReview($runId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $payrollRun = PayrollRun::where('company_id', $company->id)
            ->with(['payrollPeriod', 'payrollEntries.employee'])
            ->findOrFail($runId);
        
        return view('accounting.payroll.processing.review', compact(
            'company',
            'payrollRun'
        ));
    }

    public function processingApprove(Request $request, $runId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $payrollRun = PayrollRun::where('company_id', $company->id)
            ->findOrFail($runId);
        
        if (!$payrollRun->canApprove()) {
            return redirect()->back()
                ->with('error', 'Payroll cannot be approved in its current status.');
        }
        
        try {
            $payrollRun->update([
                'status' => 'approved',
                'approved_by' => auth()->id(),
                'approved_at' => now()
            ]);
            
            return redirect()->route('admin.accounting.payroll.processing.review', $payrollRun->id)
                ->with('success', 'Payroll approved successfully. Ready for payment.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to approve payroll: ' . $e->getMessage());
        }
    }

  
     /**
     * Process payroll payment with journal entries
     */
    public function processingPay(Request $request, $runId)
    {
        $request->validate([
            'payment_date' => 'required|date',
            'payment_reference' => 'nullable|string|max:100',
            'bank_account_id' => 'required|exists:cash_bank_accounts,id'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $payrollRun = PayrollRun::where('company_id', $company->id)
            ->with('payrollEntries.employee')
            ->findOrFail($runId);
        
        if (!$payrollRun->canPay()) {
            return redirect()->back()
                ->with('error', 'Payroll cannot be paid in its current status.');
        }
        
        DB::beginTransaction();
        
        try {
            // Update payroll run
            $payrollRun->update([
                'status' => 'paid',
                'paid_by' => auth()->id(),
                'paid_at' => now()
            ]);
            
            // Update all payroll entries
            foreach ($payrollRun->payrollEntries as $entry) {
                $entry->update([
                    'payment_status' => 'paid',
                    'payment_date' => $request->payment_date,
                    'payment_reference' => $request->payment_reference
                ]);
            }
            
            // Create journal entries for payroll using business process mappings
            $this->createPayrollJournalEntries($payrollRun, $request->bank_account_id);
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.processing.review', $payrollRun->id)
                ->with('success', 'Payroll marked as paid and journal entries created successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to process payment: ' . $e->getMessage());
        }
    }



      /**
     * Create payroll journal entries using business process mappings
     */
    private function createPayrollJournalEntries($payrollRun, $bankAccountId)
    {
        $company = $payrollRun->company;
        
        // Get business process mappings for payroll
        $mappings = BusinessProcessMapping::getPayrollMappings();
        
        // Get bank account
        $bankAccount = \App\Models\CashBankAccount::find($bankAccountId);
        if (!$bankAccount) {
            throw new \Exception('Bank account not found.');
        }
        
        // Create payroll journal entry
        $journal = JournalEntry::create([
            'entry_date' => $payrollRun->paid_at ?: now(),
            'description' => 'Payroll payment for ' . $payrollRun->payrollPeriod->period_name,
            'reference_number' => 'PR-' . $payrollRun->run_number,
            'status' => 'draft',
            'created_by' => auth()->id(),
            'company_id' => $company->id,
            'branch_id' => $company->default_branch_id ?? 1,
        ]);
        
        // Get account IDs from mappings
        $grossSalaryAccountId = $mappings->get('Payroll - Gross Salary')->accountid ?? null;
        $payeAccountId = $mappings->get('Payroll - PAYE')->accountid ?? null;
        $napsaEmployeeAccountId = $mappings->get('Payroll - NAPSA Employee')->accountid ?? null;
        $napsaEmployerAccountId = $mappings->get('Payroll - NAPSA Employer')->accountid ?? null;
        $nhimaEmployeeAccountId = $mappings->get('Payroll - NHIMA Employee')->accountid ?? null;
        $nhimaEmployerAccountId = $mappings->get('Payroll - NHIMA Employer')->accountid ?? null;
        $netPayAccountId = $mappings->get('Payroll - Net Pay')->accountid ?? null;
        
        // Validate all required accounts are mapped
        $requiredAccounts = [
            'Gross Salary' => $grossSalaryAccountId,
            'PAYE' => $payeAccountId,
            'NAPSA Employee' => $napsaEmployeeAccountId,
            'NAPSA Employer' => $napsaEmployerAccountId,
            'NHIMA Employee' => $nhimaEmployeeAccountId,
            'NHIMA Employer' => $nhimaEmployerAccountId,
            'Net Pay' => $netPayAccountId,
        ];
        
        $missingAccounts = [];
        foreach ($requiredAccounts as $name => $accountId) {
            if (!$accountId) {
                $missingAccounts[] = $name;
            }
        }
        
        if (!empty($missingAccounts)) {
            throw new \Exception('Missing business process mappings for: ' . implode(', ', $missingAccounts));
        }
        
        // Calculate totals
        $totalGross = $payrollRun->total_gross_salary;
        $totalPaye = $payrollRun->total_paye_tax;
        $totalNapsaEmployee = $payrollRun->total_napsa_employee;
        $totalNapsaEmployer = $payrollRun->total_napsa_employer;
        $totalNhimaEmployee = $payrollRun->total_nhima_employee;
        $totalNhimaEmployer = $payrollRun->total_nhima_employer;
        $totalNetPay = $payrollRun->total_net_pay;
        
        // Create journal entries
        
        // 1. Gross Salary Expense
        $journal->items()->create([
            'account_id' => $grossSalaryAccountId,
            'debit' => $totalGross,
            'credit' => 0,
            'description' => 'Gross salary expense'
        ]);
        
        // 2. PAYE Payable
        $journal->items()->create([
            'account_id' => $payeAccountId,
            'debit' => 0,
            'credit' => $totalPaye,
            'description' => 'PAYE tax payable'
        ]);
        
        // 3. NAPSA Employee Payable
        $journal->items()->create([
            'account_id' => $napsaEmployeeAccountId,
            'debit' => 0,
            'credit' => $totalNapsaEmployee,
            'description' => 'NAPSA employee contributions payable'
        ]);
        
        // 4. NAPSA Employer Expense
        $journal->items()->create([
            'account_id' => $napsaEmployerAccountId,
            'debit' => $totalNapsaEmployer,
            'credit' => 0,
            'description' => 'NAPSA employer contributions expense'
        ]);
        
        // 5. NHIMA Employee Payable
        $journal->items()->create([
            'account_id' => $nhimaEmployeeAccountId,
            'debit' => 0,
            'credit' => $totalNhimaEmployee,
            'description' => 'NHIMA employee contributions payable'
        ]);
        
        // 6. NHIMA Employer Expense
        $journal->items()->create([
            'account_id' => $nhimaEmployerAccountId,
            'debit' => $totalNhimaEmployer,
            'credit' => 0,
            'description' => 'NHIMA employer contributions expense'
        ]);
        
        // 7. Net Salary Payable
        $journal->items()->create([
            'account_id' => $netPayAccountId,
            'debit' => 0,
            'credit' => $totalNetPay,
            'description' => 'Net salary payable to employees'
        ]);
        
        // Post the journal entry
        $journal->post();
        
        // Create payment journal entry
        $paymentJournal = JournalEntry::create([
            'entry_date' => $payrollRun->paid_at ?: now(),
            'description' => 'Payment of payroll - ' . $payrollRun->run_number,
            'reference_number' => 'PP-' . $payrollRun->run_number,
            'status' => 'draft',
            'created_by' => auth()->id(),
            'company_id' => $company->id,
            'branch_id' => $company->default_branch_id ?? 1,
        ]);
        
        // Debit: Net Pay Payable (clearing the liability)
        $paymentJournal->items()->create([
            'account_id' => $netPayAccountId,
            'debit' => $totalNetPay,
            'credit' => 0,
            'description' => 'Payment of net salary'
        ]);
        
        // Credit: Bank Account
        $paymentJournal->items()->create([
            'account_id' => $bankAccount->account_id,
            'debit' => 0,
            'credit' => $totalNetPay,
            'description' => 'Payment from bank account'
        ]);
        
        $paymentJournal->post();
        
        // Update payroll run with journal entry IDs
        $payrollRun->update([
            'salary_journal_entry_id' => $journal->id,
            'payment_journal_entry_id' => $paymentJournal->id
        ]);
        
        return $journal;
    }
    
    /**
     * Process statutory payments with journal entries
     */
    public function processStatutoryPayments(Request $request)
    {
        $request->validate([
            'payment_date' => 'required|date',
            'paye_amount' => 'required|numeric|min:0',
            'napsa_amount' => 'required|numeric|min:0',
            'nhima_amount' => 'required|numeric|min:0',
            'bank_account_id' => 'required|exists:cash_bank_accounts,id',
            'payment_reference' => 'nullable|string|max:100'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        // Get business process mappings
        $mappings = BusinessProcessMapping::getPayrollMappings();
        
        // Get bank account
        $bankAccount = \App\Models\CashBankAccount::find($request->bank_account_id);
        if (!$bankAccount) {
            return redirect()->back()->with('error', 'Bank account not found.');
        }
        
        DB::beginTransaction();
        
        try {
            // Create statutory payment journal entry
            $journal = JournalEntry::create([
                'entry_date' => $request->payment_date,
                'description' => 'Statutory payments to authorities',
                'reference_number' => 'SP-' . date('Ymd') . '-' . str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT),
                'status' => 'draft',
                'created_by' => auth()->id(),
                'company_id' => $company->id,
                'branch_id' => $company->default_branch_id ?? 1,
            ]);
            
            $totalPayment = $request->paye_amount + $request->napsa_amount + $request->nhima_amount;
            
            // Process PAYE payment
            if ($request->paye_amount > 0) {
                $payePayableAccountId = $mappings->get('Statutory Payable - PAYE')->accountid ?? null;
                $payePaymentAccountId = $mappings->get('Statutory Payment - PAYE')->accountid ?? null;
                
                if (!$payePayableAccountId || !$payePaymentAccountId) {
                    throw new \Exception('PAYE accounts not configured in business process mappings.');
                }
                
                // Debit: PAYE Payable (clearing liability)
                $journal->items()->create([
                    'account_id' => $payePayableAccountId,
                    'debit' => $request->paye_amount,
                    'credit' => 0,
                    'description' => 'PAYE payment to ZRA'
                ]);
                
                // Credit: Bank Account
                $journal->items()->create([
                    'account_id' => $payePaymentAccountId,
                    'debit' => 0,
                    'credit' => $request->paye_amount,
                    'description' => 'PAYE payment'
                ]);
            }
            
            // Process NAPSA payment
            if ($request->napsa_amount > 0) {
                $napsaPayableAccountId = $mappings->get('Statutory Payable - NAPSA')->accountid ?? null;
                $napsaPaymentAccountId = $mappings->get('Statutory Payment - NAPSA')->accountid ?? null;
                
                if (!$napsaPayableAccountId || !$napsaPaymentAccountId) {
                    throw new \Exception('NAPSA accounts not configured in business process mappings.');
                }
                
                // Debit: NAPSA Payable (clearing liability)
                $journal->items()->create([
                    'account_id' => $napsaPayableAccountId,
                    'debit' => $request->napsa_amount,
                    'credit' => 0,
                    'description' => 'NAPSA payment'
                ]);
                
                // Credit: Bank Account
                $journal->items()->create([
                    'account_id' => $napsaPaymentAccountId,
                    'debit' => 0,
                    'credit' => $request->napsa_amount,
                    'description' => 'NAPSA payment'
                ]);
            }
            
            // Process NHIMA payment
            if ($request->nhima_amount > 0) {
                $nhimaPayableAccountId = $mappings->get('Statutory Payable - NHIMA')->accountid ?? null;
                $nhimaPaymentAccountId = $mappings->get('Statutory Payment - NHIMA')->accountid ?? null;
                
                if (!$nhimaPayableAccountId || !$nhimaPaymentAccountId) {
                    throw new \Exception('NHIMA accounts not configured in business process mappings.');
                }
                
                // Debit: NHIMA Payable (clearing liability)
                $journal->items()->create([
                    'account_id' => $nhimaPayableAccountId,
                    'debit' => $request->nhima_amount,
                    'credit' => 0,
                    'description' => 'NHIMA payment'
                ]);
                
                // Credit: Bank Account
                $journal->items()->create([
                    'account_id' => $nhimaPaymentAccountId,
                    'debit' => 0,
                    'credit' => $request->nhima_amount,
                    'description' => 'NHIMA payment'
                ]);
            }
            
            // Final bank entry
            $journal->items()->create([
                'account_id' => $bankAccount->account_id,
                'debit' => $totalPayment,
                'credit' => 0,
                'description' => 'Payment from bank account'
            ]);
            
            $journal->post();
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.index')
                ->with('success', 'Statutory payments recorded successfully. Journal entry #' . $journal->reference_number . ' created.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to process statutory payments: ' . $e->getMessage())
                ->withInput();
        }
    }


    public function processingExport($runId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $payrollRun = PayrollRun::where('company_id', $company->id)
            ->with(['payrollEntries.employee', 'payrollPeriod'])
            ->findOrFail($runId);
        
        // Export to Excel or PDF
        // You can use Laravel Excel package or DomPDF
        
        return response()->streamDownload(function () use ($payrollRun) {
            echo $this->generatePayrollReport($payrollRun);
        }, 'payroll-run-' . $payrollRun->run_number . '.pdf');
    }
    
    /**
     * Process leave accrual for payroll period
     */
    public function processLeaveAccrual(Request $request, $periodId)
    {
        $request->validate([
            'accrue_for_all' => 'required|boolean',
            'employee_ids' => 'required_if:accrue_for_all,false|array',
            'employee_ids.*' => 'exists:employees,id',
            'create_journal_entries' => 'required|boolean'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        // Get employees to process
        if ($request->accrue_for_all) {
            $employees = Employee::where('company_id', $company->id)
                ->where('is_active', true)
                ->where('employment_type', 'permanent') // Typically only permanent employees get leave
                ->get();
        } else {
            $employees = Employee::where('company_id', $company->id)
                ->whereIn('id', $request->employee_ids)
                ->get();
        }
        
        DB::beginTransaction();
        
        try {
            $processedCount = 0;
            $totalAccruedDays = 0;
            
            foreach ($employees as $employee) {
                // Calculate leave accrual
                $accrualResult = $employee->calculateLeaveAccrual($period->end_date);
                
                if ($accrualResult && $accrualResult['accrued_days'] > 0) {
                    // Apply leave accrual
                    $employee->applyLeaveAccrual(
                        $accrualResult['accrued_days'],
                        $period->end_date,
                        $request->create_journal_entries
                    );
                    
                    $processedCount++;
                    $totalAccruedDays += $accrualResult['accrued_days'];
                }
            }
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.periods.index')
                ->with('success', 'Leave accrual processed for ' . $processedCount . ' employees. Total ' . $totalAccruedDays . ' days accrued.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to process leave accrual: ' . $e->getMessage());
        }
    }

    /**
     * Get payroll account mappings status
     */
    public function getPayrollAccountMappings()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $mappings = BusinessProcessMapping::getPayrollMappings();
        
        $mappingStatus = [];
        $requiredMappings = [
            'Payroll - Gross Salary' => 'Gross Salary Expense',
            'Payroll - PAYE' => 'PAYE Deduction',
            'Payroll - NAPSA Employer' => 'NAPSA Employer Contribution',
            'Payroll - NAPSA Employee' => 'NAPSA Employee Deduction',
            'Payroll - NHIMA Employer' => 'NHIMA Employer Contribution',
            'Payroll - NHIMA Employee' => 'NHIMA Employee Deduction',
            'Payroll - Net Pay' => 'Net Pay Payable',
            'Statutory Payable - PAYE' => 'PAYE Payable',
            'Statutory Payable - NAPSA' => 'NAPSA Payable',
            'Statutory Payable - NHIMA' => 'NHIMA Payable',
            'Leave Accrual' => 'Leave Accrual Expense',
            'Leave Liability' => 'Leave Liability'
        ];
        
        foreach ($requiredMappings as $process => $description) {
            $mapping = $mappings->get($process);
            $mappingStatus[] = [
                'process' => $process,
                'description' => $description,
                'is_configured' => !is_null($mapping),
                'account_code' => $mapping->account->code ?? null,
                'account_name' => $mapping->account->name ?? null,
                'account_id' => $mapping->accountid ?? null
            ];
        }
        
        return response()->json([
            'success' => true,
            'mappings' => $mappingStatus,
            'company' => $company->name
        ]);
    }

    public function processingExportBank($runId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $payrollRun = PayrollRun::where('company_id', $company->id)
            ->with(['payrollEntries.employee'])
            ->findOrFail($runId);
        
        // Generate bank file (CSV format for bank upload)
        $filename = 'bank-payroll-' . $payrollRun->run_number . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function () use ($payrollRun) {
            $file = fopen('php://output', 'w');
            
            // Bank file header (adjust based on your bank's requirements)
            fputcsv($file, ['Account Number', 'Account Name', 'Amount', 'Reference']);
            
            foreach ($payrollRun->payrollEntries as $entry) {
                if ($entry->employee->payment_method === 'bank' && $entry->employee->bank_account_number) {
                    fputcsv($file, [
                        $entry->employee->bank_account_number,
                        $entry->employee->bank_account_name ?: $entry->employee->full_name,
                        number_format($entry->net_pay, 2, '.', ''),
                        'SALARY-' . $payrollRun->run_number
                    ]);
                }
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }

    // Earnings Management Methods
    public function earningTypesIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $earningTypes = EarningType::where('company_id', $company->id)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->paginate(20);
        
        return view('accounting.payroll.earnings.types.index', compact(
            'company',
            'earningTypes'
        ));
    }

    public function earningTypesCreate()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        return view('accounting.payroll.earnings.types.create', compact('company'));
    }

    public function earningTypesStore(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:basic,gratuity,allowance,bonus,commission,overtime,other',
            'code' => 'required|string|max:50|unique:earning_types,code',
            'description' => 'nullable|string',
            'is_taxable' => 'required|boolean',
            'is_statutory' => 'required|boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        // Custom validation: Statutory deductions must be taxable
        if ($request->is_statutory && !$request->is_taxable) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Statutory deductions must be taxable. Please check "Taxable Earnings" if you want to mark this as statutory.');
        }

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        try {
            EarningType::create([
                'company_id' => $company->id,
                'name' => $request->name,
                'type' => $request->type,
                'code' => strtoupper($request->code),
                'description' => $request->description,
                'is_taxable' => $request->is_taxable,
                'is_statutory' => $request->is_statutory,
                'sort_order' => $request->sort_order ?? 0,
                'is_active' => true,
                'created_by' => auth()->id()
            ]);

            return redirect()->route('admin.accounting.payroll.earnings.types.index')
                ->with('success', 'Earning type created successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create earning type: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function earningTypesUpdate(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:basic,gratuity,allowance,bonus,commission,overtime,other',
            'code' => 'required|string|max:50|unique:earning_types,code,' . $id,
            'description' => 'nullable|string',
            'is_taxable' => 'required|boolean',
            'is_statutory' => 'required|boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        // Custom validation: Statutory deductions must be taxable
        if ($request->is_statutory && !$request->is_taxable) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Statutory deductions must be taxable. Please check "Taxable Earnings" if you want to mark this as statutory.');
        }

        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();

        $earningType = EarningType::where('company_id', $company->id)
            ->findOrFail($id);

        try {
            $earningType->update([
                'name' => $request->name,
                'type' => $request->type,
                'code' => strtoupper($request->code),
                'description' => $request->description,
                'is_taxable' => $request->is_taxable,
                'is_statutory' => $request->is_statutory,
                'sort_order' => $request->sort_order ?? 0,
                'updated_by' => auth()->id()
            ]);

            return redirect()->route('admin.accounting.payroll.earnings.types.index')
                ->with('success', 'Earning type updated successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update earning type: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function employeeEarningsIndex($employeeId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->findOrFail($employeeId);
        
        $earnings = EmployeeEarning::where('employee_id', $employee->id)
            ->where('is_active', true)
            ->with('earningType')
            ->orderBy('effective_date', 'desc')
            ->paginate(20);
        
        $earningTypes = EarningType::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();
        
        return view('accounting.payroll.earnings.employee.index', compact(
            'company',
            'employee',
            'earnings',
            'earningTypes'
        ));
    }

    public function employeeEarningsStore(Request $request, $employeeId)
    {
        $request->validate([
            'earning_type_id' => 'required|exists:earning_types,id',
            'amount' => 'required|numeric|min:0',
            'effective_date' => 'required|date',
            'end_date' => 'nullable|date|after:effective_date',
            'is_recurring' => 'required|boolean',
            'frequency' => 'required|in:monthly,quarterly,annually,one-time',
            'notes' => 'nullable|string',
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $employee = Employee::where('company_id', $company->id)
            ->findOrFail($employeeId);
        
        // Check if earning type belongs to same company
        $earningType = EarningType::where('id', $request->earning_type_id)
            ->where('company_id', $company->id)
            ->firstOrFail();
        
        try {
            EmployeeEarning::create([
                'employee_id' => $employee->id,
                'earning_type_id' => $earningType->id,
                'amount' => $request->amount,
                'currency' => $company->reporting_currency ?? 'ZMW',
                'effective_date' => $request->effective_date,
                'end_date' => $request->end_date,
                'is_recurring' => $request->is_recurring,
                'frequency' => $request->frequency,
                'notes' => $request->notes,
                'is_active' => true,
                'created_by' => auth()->id()
            ]);
            
            return redirect()->route('admin.accounting.payroll.earnings.employee.index', $employee->id)
                ->with('success', 'Employee earning added successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to add employee earning: ' . $e->getMessage())
                ->withInput();
        }
    }

    // Get earning types for API (used in employee edit form)
    public function getEarningTypesApi()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $earningTypes = EarningType::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get(['id', 'name', 'type', 'code', 'is_taxable'])
            ->map(function($type) {
                return [
                    'id' => $type->id,
                    'name' => $type->name,
                    'type' => $type->type,
                    'type_label' => $type->type_label,
                    'type_color' => $type->type_color,
                    'code' => $type->code,
                    'is_taxable' => $type->is_taxable,
                    'taxable_badge' => $type->taxable_badge
                ];
            });
        
        return response()->json($earningTypes);
    }

    // ============================================
    // PAYROLL PREPARATION METHODS - UPDATED SECTION
    // ============================================

    public function preparationIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $currentFinancialYear = $company->currentFinancialYear;
        
        // Show ALL periods (not just open ones)
        $payrollPeriods = PayrollPeriod::where('company_id', $company->id)
            ->where('financial_year_id', $currentFinancialYear->id ?? null)
            ->orderBy('start_date', 'desc')
            ->paginate(20);
        
        // Count adjustments by period
        $periodAdjustmentsCount = [];
        foreach ($payrollPeriods as $period) {
            $periodAdjustmentsCount[$period->id] = PayrollAdjustment::where('payroll_period_id', $period->id)
                ->where('company_id', $company->id)
                ->count();
        }
        
        return view('accounting.payroll.preparation.index', compact(
            'company',
            'currentFinancialYear',
            'payrollPeriods',
            'periodAdjustmentsCount'
        ));
    }

    public function preparationCreate($periodId, $employeeId = null)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        // Get active employees for this period
        $employees = Employee::where('company_id', $company->id)
            ->where('is_active', true)
            ->orderBy('first_name')
            ->get();
        
        // Get earning types (for one-time adjustments like overtime, bonus, etc.)
        $earningTypes = EarningType::where('company_id', $company->id)
            ->where('is_active', true)
            ->whereIn('type', ['overtime', 'bonus', 'commission', 'allowance', 'other'])
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();
        
        // Get existing adjustments for this period
        $adjustments = PayrollAdjustment::where('company_id', $company->id)
            ->where('payroll_period_id', $periodId)
            ->with(['employee', 'earningType', 'approver'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Get selected employee if provided via route parameter or query parameter
        $selectedEmployee = null;
        if (!$employeeId) {
            // Check query parameter as fallback
            $employeeId = request()->query('employee_id') ?? request()->query('employeeId');
        }
        
        if ($employeeId) {
            $selectedEmployee = Employee::where('company_id', $company->id)
                ->find($employeeId);
        }
        
        // Group adjustments by employee for easier display
        $employeeAdjustments = collect([]);
        
        if ($adjustments->count() > 0) {
            $employeeAdjustments = $adjustments->groupBy('employee_id');
        }
        
        // Load recurring earnings (this will be used by the view)
        $recurringEarnings = collect([]);
        if ($selectedEmployee) {
            // Load recurring earnings for selected employee
            $recurringEarnings = EmployeeEarning::where('employee_id', $selectedEmployee->id)
                ->where('is_active', true)
                ->where('is_recurring', true)
                ->where('effective_date', '<=', $period->end_date)
                ->where(function($q) use ($period) {
                    $q->whereNull('end_date')
                      ->orWhere('end_date', '>=', $period->start_date);
                })
                ->with('earningType')
                ->get();
        } else {
            // Load all recurring earnings if no employee selected
            $recurringEarnings = EmployeeEarning::whereHas('employee', function($q) use ($company) {
                    $q->where('company_id', $company->id)
                      ->where('is_active', true);
                })
                ->where('is_active', true)
                ->where('is_recurring', true)
                ->where('effective_date', '<=', $period->end_date)
                ->where(function($q) use ($period) {
                    $q->whereNull('end_date')
                      ->orWhere('end_date', '>=', $period->start_date);
                })
                ->with(['employee', 'earningType'])
                ->get();
        }
        
        return view('accounting.payroll.preparation.create', compact(
            'company',
            'period',
            'employees',
            'earningTypes',
            'adjustments',
            'employeeAdjustments',
            'selectedEmployee',
            'recurringEarnings'
        ));
    }

    public function preparationStore(Request $request, $periodId)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'earning_type_id' => 'required|exists:earning_types,id',
            'description' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'is_taxable' => 'required|boolean',
            'is_one_time' => 'required|boolean',
            'notes' => 'nullable|string'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        // Verify period belongs to company
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        // Verify employee belongs to company
        $employee = Employee::where('company_id', $company->id)
            ->findOrFail($request->employee_id);
        
        // Verify earning type belongs to company
        $earningType = EarningType::where('company_id', $company->id)
            ->findOrFail($request->earning_type_id);
        
        try {
            $adjustment = PayrollAdjustment::create([
                'company_id' => $company->id,
                'payroll_period_id' => $periodId,
                'employee_id' => $request->employee_id,
                'earning_type_id' => $request->earning_type_id,
                'description' => $request->description,
                'amount' => $request->amount,
                'is_taxable' => $request->is_taxable,
                'is_one_time' => $request->is_one_time,
                'notes' => $request->notes,
                'created_by' => auth()->id()
            ]);
            
            return redirect()->route('admin.accounting.payroll.preparation.create', $periodId)
                ->with('success', 'Payroll adjustment added successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to add payroll adjustment: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function preparationUpdate(Request $request, $adjustmentId)
    {
        $request->validate([
            'description' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'is_taxable' => 'required|boolean',
            'is_one_time' => 'required|boolean',
            'notes' => 'nullable|string'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $adjustment = PayrollAdjustment::where('company_id', $company->id)
            ->findOrFail($adjustmentId);
        
        // Check if adjustment is already approved
        if ($adjustment->isApproved()) {
            return redirect()->back()
                ->with('error', 'Cannot modify an approved adjustment. Please unapprove it first.')
                ->withInput();
        }
        
        try {
            $adjustment->update([
                'description' => $request->description,
                'amount' => $request->amount,
                'is_taxable' => $request->is_taxable,
                'is_one_time' => $request->is_one_time,
                'notes' => $request->notes,
                'updated_by' => auth()->id()
            ]);
            
            return redirect()->route('admin.accounting.payroll.preparation.create', $adjustment->payroll_period_id)
                ->with('success', 'Payroll adjustment updated successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update payroll adjustment: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function preparationDestroy($adjustmentId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $adjustment = PayrollAdjustment::where('company_id', $company->id)
            ->findOrFail($adjustmentId);
        
        // Check if adjustment is already approved
        if ($adjustment->isApproved()) {
            return redirect()->back()
                ->with('error', 'Cannot delete an approved adjustment. Please unapprove it first.');
        }
        
        try {
            $periodId = $adjustment->payroll_period_id;
            $adjustment->delete();
            
            return redirect()->route('admin.accounting.payroll.preparation.create', $periodId)
                ->with('success', 'Payroll adjustment deleted successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete payroll adjustment: ' . $e->getMessage());
        }
    }

    public function preparationApprove($adjustmentId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $adjustment = PayrollAdjustment::where('company_id', $company->id)
            ->findOrFail($adjustmentId);
        
        try {
            $adjustment->approve(auth()->id());
            
            return redirect()->route('admin.accounting.payroll.preparation.create', $adjustment->payroll_period_id)
                ->with('success', 'Payroll adjustment approved successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to approve adjustment: ' . $e->getMessage());
        }
    }

    public function preparationUnapprove($adjustmentId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $adjustment = PayrollAdjustment::where('company_id', $company->id)
            ->findOrFail($adjustmentId);
        
        try {
            $adjustment->update([
                'approved_by' => null,
                'approved_at' => null,
                'updated_by' => auth()->id()
            ]);
            
            return redirect()->route('admin.accounting.payroll.preparation.create', $adjustment->payroll_period_id)
                ->with('success', 'Payroll adjustment unapproved successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to unapprove adjustment: ' . $e->getMessage());
        }
    }

    public function preparationBulkApprove(Request $request, $periodId)
    {
        $request->validate([
            'adjustment_ids' => 'required|array',
            'adjustment_ids.*' => 'exists:payroll_adjustments,id'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        DB::beginTransaction();
        
        try {
            foreach ($request->adjustment_ids as $adjustmentId) {
                $adjustment = PayrollAdjustment::where('company_id', $company->id)
                    ->where('payroll_period_id', $periodId)
                    ->find($adjustmentId);
                
                if ($adjustment && !$adjustment->isApproved()) {
                    $adjustment->approve(auth()->id());
                }
            }
            
            DB::commit();
            
            return redirect()->route('admin.accounting.payroll.preparation.create', $periodId)
                ->with('success', 'Selected adjustments approved successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to bulk approve adjustments: ' . $e->getMessage());
        }
    }

    // ============================================
    // NEW PAYROLL CALCULATION METHODS
    // ============================================

    /**
     * Calculate payroll for a specific period
     */
    public function calculatePayroll(Request $request, $periodId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        // Check if payroll already calculated for this period
        $existingEntries = PayrollEntry::where('company_id', $company->id)
            ->where('payroll_period_id', $periodId)
            ->where('calculation_status', '!=', 'draft')
            ->count();
        
        if ($existingEntries > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Payroll has already been calculated for this period.'
            ]);
        }
        
        DB::beginTransaction();
        
        try {
            // Get all active employees
            $employees = Employee::where('company_id', $company->id)
                ->where('is_active', true)
                ->with(['activeEarnings.earningType'])
                ->get();
            
            // Get payroll settings
            $currentFinancialYear = $company->currentFinancialYear;
            $payeSettings = PayrollSetting::where('company_id', $company->id)
                ->where('financial_year_id', $currentFinancialYear->id ?? null)
                ->where('setting_type', 'paye')
                ->first();
            
            $napsaSettings = PayrollSetting::where('company_id', $company->id)
                ->where('financial_year_id', $currentFinancialYear->id ?? null)
                ->where('setting_type', 'napsa')
                ->first();
            
            $nhimaSettings = PayrollSetting::where('company_id', $company->id)
                ->where('financial_year_id', $currentFinancialYear->id ?? null)
                ->where('setting_type', 'nhima')
                ->first();
            
            if (!$payeSettings || !$napsaSettings || !$nhimaSettings) {
                throw new \Exception('Please configure PAYE, NAPSA, and NHIMA settings before calculating payroll.');
            }
            
            // Get PAYE tax bands
            $taxBands = PayeTaxBand::where('payroll_setting_id', $payeSettings->id)
                ->where('is_active', true)
                ->orderBy('band_number')
                ->get();
            
            // Process each employee
            foreach ($employees as $employee) {
                // Get recurring earnings (from EmployeeEarning)
                $recurringEarnings = EmployeeEarning::where('employee_id', $employee->id)
                    ->where('is_active', true)
                    ->where('is_recurring', true)
                    ->where('effective_date', '<=', $period->end_date)
                    ->where(function($q) use ($period) {
                        $q->whereNull('end_date')
                          ->orWhere('end_date', '>=', $period->start_date);
                    })
                    ->with('earningType')
                    ->get();
                
                // Get non-recurring adjustments (from PayrollAdjustment)
                $nonRecurringAdjustments = PayrollAdjustment::where('company_id', $company->id)
                    ->where('payroll_period_id', $periodId)
                    ->where('employee_id', $employee->id)
                    ->whereNotNull('approved_by') // Only approved adjustments
                    ->with('earningType')
                    ->get();
                
                // Prepare earnings breakdown
                $earningsBreakdown = [
                    'recurring' => [],
                    'non_recurring' => []
                ];
                
                $totalRecurring = 0;
                $totalNonRecurring = 0;
                $totalTaxableEarnings = 0;
                $totalNonTaxableEarnings = 0;
                $basicSalary = 0; // For NAPSA calculation
                
                // Process recurring earnings
                foreach ($recurringEarnings as $earning) {
                    $monthlyAmount = $earning->getMonthlyAmount();
                    $isTaxable = $earning->earningType->is_taxable;
                    
                    $earningsBreakdown['recurring'][] = [
                        'earning_type_id' => $earning->earning_type_id,
                        'name' => $earning->earningType->name,
                        'amount' => $monthlyAmount,
                        'is_taxable' => $isTaxable,
                        'type' => $earning->earningType->type
                    ];
                    
                    $totalRecurring += $monthlyAmount;
                    
                    if ($isTaxable) {
                        $totalTaxableEarnings += $monthlyAmount;
                    } else {
                        $totalNonTaxableEarnings += $monthlyAmount;
                    }
                    
                    // Get basic salary for NAPSA calculation
                    if ($earning->earningType->type === 'basic') {
                        $basicSalary += $monthlyAmount;
                    }
                }
                
                // Process non-recurring adjustments
                foreach ($nonRecurringAdjustments as $adjustment) {
                    $earningsBreakdown['non_recurring'][] = [
                        'earning_type_id' => $adjustment->earning_type_id,
                        'name' => $adjustment->earningType->name,
                        'amount' => $adjustment->amount,
                        'is_taxable' => $adjustment->is_taxable,
                        'type' => $adjustment->earningType->type,
                        'description' => $adjustment->description
                    ];
                    
                    $totalNonRecurring += $adjustment->amount;
                    
                    if ($adjustment->is_taxable) {
                        $totalTaxableEarnings += $adjustment->amount;
                    } else {
                        $totalNonTaxableEarnings += $adjustment->amount;
                    }
                    
                    // Add to basic salary if it's a basic type earning (though unlikely for non-recurring)
                    if ($adjustment->earningType->type === 'basic') {
                        $basicSalary += $adjustment->amount;
                    }
                }
                
                $grossEarnings = $totalRecurring + $totalNonRecurring;
                
                // Calculate PAYE on taxable earnings
                $payeTax = $this->calculatePayeTaxNew($totalTaxableEarnings, $taxBands);
                
                // ============================================
                // UPDATED NAPSA CALCULATION - FOR ALL EMPLOYEES
                // ============================================
                $napsaEmployee = 0;
                $napsaEmployer = 0;
                
                // Calculate NAPSA for ALL employees (removed napsa_number check)
                if ($napsaSettings && $basicSalary > 0) {
                    // Employee contribution: 5% of basic salary, capped at monthly ceiling
                    $napsaEmployee = min($basicSalary * ($napsaSettings->monthly_rate / 100), $napsaSettings->monthly_ceiling);
                    
                    // Employer contribution: 5% of basic salary up to ceiling
                    $napsaEmployer = min($basicSalary * ($napsaSettings->monthly_rate / 100), $napsaSettings->monthly_ceiling);
                }
                
                // ============================================
                // UPDATED NHIMA CALCULATION - FOR ALL EMPLOYEES
                // ============================================
                $nhimaEmployee = 0;
                $nhimaEmployer = 0;
                
                // Calculate NHIMA for ALL employees (removed nhima_number check)
                if ($nhimaSettings && $grossEarnings > 0) {
                    // Employee contribution: 1% of gross earnings
                    $nhimaEmployee = $grossEarnings * ($nhimaSettings->monthly_rate / 100);
                    
                    // Employer contribution: 1% of gross earnings
                    $nhimaEmployer = $grossEarnings * ($nhimaSettings->monthly_rate / 100);
                }
                
                $totalDeductions = $payeTax + $napsaEmployee + $nhimaEmployee;
                $netPay = $grossEarnings - $totalDeductions;
                
                // Create payroll entry
                PayrollEntry::create([
                    'payroll_period_id' => $periodId,
                    'company_id' => $company->id,
                    'employee_id' => $employee->id,
                    'period_start_date' => $period->start_date,
                    'period_end_date' => $period->end_date,
                    'entry_date' => now()->toDateString(),
                    'earnings_breakdown' => json_encode($earningsBreakdown),
                    'total_recurring' => $totalRecurring,
                    'total_non_recurring' => $totalNonRecurring,
                    'total_taxable_earnings' => $totalTaxableEarnings,
                    'total_non_taxable_earnings' => $totalNonTaxableEarnings,
                    'gross_earnings' => $grossEarnings,
                    'basic_salary_for_napsa' => $basicSalary,
                    'paye_tax' => $payeTax,
                    'napsa_employee' => $napsaEmployee,
                    'napsa_employer' => $napsaEmployer,
                    'nhima_employee' => $nhimaEmployee,
                    'nhima_employer' => $nhimaEmployer,
                    'total_deductions' => $totalDeductions,
                    'net_pay' => $netPay,
                    'calculation_status' => 'calculated',
                    'calculated_by' => auth()->id(),
                    'calculated_at' => now()
                ]);
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Payroll calculated successfully for ' . $employees->count() . ' employees.',
                'employees_count' => $employees->count()
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to calculate payroll: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get earnings breakdown for a payroll entry
     */
    public function getEarningsBreakdown($entryId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $entry = PayrollEntry::where('company_id', $company->id)
            ->with('employee')
            ->findOrFail($entryId);
        
        $breakdown = $entry->getEarningsBreakdownFormatted();
        
        $html = view('accounting.payroll.partials.earnings-breakdown', [
            'breakdown' => $breakdown,
            'entry' => $entry,
            'company' => $company
        ])->render();
        
        return response()->json([
            'success' => true,
            'html' => $html,
            'employee_name' => $entry->employee->full_name
        ]);
    }

    /**
     * Approve a single payroll entry
     */
    public function approvePayrollEntry($entryId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $entry = PayrollEntry::where('company_id', $company->id)
            ->findOrFail($entryId);
        
        if (!$entry->canApprove()) {
            return response()->json([
                'success' => false,
                'message' => 'Entry cannot be approved in its current status.'
            ]);
        }
        
        try {
            $entry->markAsApproved(auth()->id());
            
            return response()->json([
                'success' => true,
                'message' => 'Payroll entry approved successfully.'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to approve entry: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk approve payroll entries for a period
     */
    public function bulkApprovePayroll($periodId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $entries = PayrollEntry::where('company_id', $company->id)
            ->where('payroll_period_id', $periodId)
            ->where('calculation_status', 'calculated')
            ->get();
        
        if ($entries->count() === 0) {
            return response()->json([
                'success' => false,
                'message' => 'No entries to approve.'
            ]);
        }
        
        DB::beginTransaction();
        
        try {
            $approvedCount = 0;
            foreach ($entries as $entry) {
                if ($entry->canApprove()) {
                    $entry->markAsApproved(auth()->id());
                    $approvedCount++;
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => $approvedCount . ' payroll entries approved successfully.',
                'approved_count' => $approvedCount
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to approve entries: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export payroll for a period
     */
    public function exportPayroll($periodId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        $entries = PayrollEntry::where('company_id', $company->id)
            ->where('payroll_period_id', $periodId)
            ->with('employee')
            ->get();
        
        // Generate CSV
        $filename = 'payroll-export-' . $period->period_name . '-' . date('Y-m-d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function () use ($entries, $company) {
            $file = fopen('php://output', 'w');
            
            // CSV header
            fputcsv($file, [
                'Employee Number',
                'Employee Name',
                'Gross Earnings',
                'Taxable Earnings',
                'Non-Taxable Earnings',
                'PAYE Tax',
                'NAPSA (Employee)',
                'NHIMA (Employee)',
                'Total Deductions',
                'Net Pay',
                'Basic Salary (NAPSA Base)',
                'Status'
            ]);
            
            foreach ($entries as $entry) {
                fputcsv($file, [
                    $entry->employee->employee_number,
                    $entry->employee->full_name,
                    number_format($entry->gross_earnings, 2, '.', ''),
                    number_format($entry->total_taxable_earnings, 2, '.', ''),
                    number_format($entry->total_non_taxable_earnings, 2, '.', ''),
                    number_format($entry->paye_tax, 2, '.', ''),
                    number_format($entry->napsa_employee, 2, '.', ''),
                    number_format($entry->nhima_employee, 2, '.', ''),
                    number_format($entry->total_deductions, 2, '.', ''),
                    number_format($entry->net_pay, 2, '.', ''),
                    number_format($entry->basic_salary_for_napsa, 2, '.', ''),
                    $entry->calculation_status
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }

    /**
     * New PAYE calculation method
     */
    private function calculatePayeTaxNew($monthlySalary, $taxBands)
    {
        $totalTax = 0;
        
        foreach ($taxBands as $band) {
            if ($monthlySalary >= $band->monthly_from) {
                $taxableAmount = min($monthlySalary, $band->monthly_to) - $band->monthly_from;
                if ($taxableAmount > 0) {
                    $bandTax = ($taxableAmount * $band->tax_rate) / 100;
                    $totalTax += $bandTax;
                }
            }
        }
        
        return $totalTax;
    }

    // ============================================
    // NEW METHOD: GET RECURRING EARNINGS VIA AJAX
    // ============================================

    public function getRecurringEarnings(Request $request, $periodId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        // Get recurring earnings query
        $query = EmployeeEarning::whereHas('employee', function($q) use ($company) {
                $q->where('company_id', $company->id)
                  ->where('is_active', true);
            })
            ->where('is_active', true)
            ->where('is_recurring', true)
            ->where('effective_date', '<=', $period->end_date)
            ->where(function($q) use ($period) {
                $q->whereNull('end_date')
                  ->orWhere('end_date', '>=', $period->start_date);
            })
            ->with(['employee', 'earningType'])
            ->orderBy('employee_id')
            ->orderBy('effective_date', 'desc');
        
        // Filter by employee if requested
        if ($request->has('employee_id') && $request->employee_id !== 'all') {
            $query->where('employee_id', $request->employee_id);
        }
        
        $recurringEarnings = $query->get();
        
        // Prepare HTML response
        $html = view('accounting.payroll.preparation.partials.recurring-earnings-table', [
            'recurringEarnings' => $recurringEarnings,
            'company' => $company,
            'period' => $period,
            'filteredEmployeeId' => $request->employee_id !== 'all' ? $request->employee_id : null
        ])->render();
        
        return response()->json([
            'html' => $html,
            'count' => $recurringEarnings->count()
        ]);
    }

    // ============================================
    // ADDITIONAL METHODS FOR COMPLETENESS
    // ============================================

    public function getEmployeeAdjustments($periodId, $employeeId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        $employee = Employee::where('company_id', $company->id)
            ->where('id', $employeeId)
            ->firstOrFail();
        
        // Get adjustments for this employee and period
        $adjustments = PayrollAdjustment::where('company_id', $company->id)
            ->where('payroll_period_id', $periodId)
            ->where('employee_id', $employeeId)
            ->with(['employee', 'earningType', 'approver'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Count pending adjustments
        $pendingCount = $adjustments->whereNull('approved_by')->count();
        
        // Prepare HTML response
        $html = '';
        
        if ($adjustments->count() > 0) {
            $html = view('accounting.payroll.preparation.partials.employee-adjustments-table', [
                'adjustments' => $adjustments,
                'company' => $company,
                'employee' => $employee
            ])->render();
        } else {
            $html = '
                <div class="text-center py-8 text-gray-500">
                    <i class="fas fa-file-invoice-dollar text-3xl mb-3 text-gray-300"></i>
                    <p>No adjustments for this employee</p>
                    <p class="text-sm mt-1">Use the form above to add payroll adjustments</p>
                </div>
            ';
        }
        
        return response()->json([
            'html' => $html,
            'count' => $adjustments->count(),
            'pending_count' => $pendingCount
        ]);
    }

    // Preliminary Summary method
    public function preliminarySummary($periodId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        // Get all active employees
        $employees = Employee::where('company_id', $company->id)
            ->where('is_active', true)
            ->with(['activeEarnings.earningType'])
            ->orderBy('first_name')
            ->get();
        
        // Calculate totals for each employee
        $employeeSummaries = [];
        foreach ($employees as $employee) {
            // Calculate recurring earnings (from EmployeeEarning model)
            $recurringEarnings = EmployeeEarning::where('employee_id', $employee->id)
                ->where('is_active', true)
                ->where('is_recurring', true)
                ->where('effective_date', '<=', $period->end_date)
                ->where(function($q) use ($period) {
                    $q->whereNull('end_date')
                      ->orWhere('end_date', '>=', $period->start_date);
                })
                ->with('earningType')
                ->get();
            
            $totalRecurring = $recurringEarnings->sum(function($earning) {
                return $earning->getMonthlyAmount();
            });
            
            // Calculate non-recurring adjustments (from PayrollAdjustment model)
            $nonRecurringAdjustments = PayrollAdjustment::where('company_id', $company->id)
                ->where('payroll_period_id', $periodId)
                ->where('employee_id', $employee->id)
                ->where('is_one_time', true)
                ->get();
            
            $totalNonRecurring = $nonRecurringAdjustments->sum('amount');
            
            // Calculate basic salary (earnings with type 'basic')
            $basicSalary = $employee->getEarningsByType('basic');
            
            // Calculate allowances (earnings with type 'allowance')
            $allowances = $employee->getEarningsByType('allowance');
            
            // Calculate gross total (recurring + non-recurring)
            $grossTotal = $totalRecurring + $totalNonRecurring;
            
            // Check if employee has any pending adjustments
            $pendingAdjustments = PayrollAdjustment::where('company_id', $company->id)
                ->where('payroll_period_id', $periodId)
                ->where('employee_id', $employee->id)
                ->whereNull('approved_by')
                ->count();
            
            $employeeSummaries[] = [
                'employee' => $employee,
                'total_recurring' => $totalRecurring,
                'total_non_recurring' => $totalNonRecurring,
                'basic_salary' => $basicSalary,
                'allowances' => $allowances,
                'gross_total' => $grossTotal,
                'has_pending_adjustments' => $pendingAdjustments > 0,
                'adjustment_count' => $nonRecurringAdjustments->count(),
                'pending_adjustment_count' => $pendingAdjustments
            ];
        }
        
        // Calculate totals
        $summaryTotals = [
            'total_recurring' => collect($employeeSummaries)->sum('total_recurring'),
            'total_non_recurring' => collect($employeeSummaries)->sum('total_non_recurring'),
            'total_basic_salary' => collect($employeeSummaries)->sum('basic_salary'),
            'total_allowances' => collect($employeeSummaries)->sum('allowances'),
            'total_gross' => collect($employeeSummaries)->sum('gross_total'),
            'total_employees' => count($employeeSummaries),
            'total_pending_adjustments' => collect($employeeSummaries)->sum('pending_adjustment_count')
        ];
        
        // Check if payroll has been calculated for this period
        $payrollEntries = PayrollEntry::where('payroll_period_id', $period->id)
            ->where('company_id', $company->id)
            ->with('employee')
            ->get();
        
        return view('accounting.payroll.preparation.preliminary-summary', compact(
            'company',
            'period',
            'employeeSummaries',
            'summaryTotals',
            'payrollEntries'
        ));
    }

    // Adjustments Modal method
    public function adjustmentsModal($periodId, $employeeId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $period = PayrollPeriod::where('company_id', $company->id)
            ->findOrFail($periodId);
        
        $employee = Employee::where('company_id', $company->id)
            ->findOrFail($employeeId);
        
        // Get earning types for adjustments
        $earningTypes = EarningType::where('company_id', $company->id)
            ->where('is_active', true)
            ->whereIn('type', ['overtime', 'bonus', 'commission', 'allowance', 'other'])
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();
        
        // Get existing adjustments for this employee and period
        $adjustments = PayrollAdjustment::where('company_id', $company->id)
            ->where('payroll_period_id', $periodId)
            ->where('employee_id', $employeeId)
            ->with(['earningType', 'approver'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // If it's an AJAX request, return just the modal content
        if (request()->ajax()) {
            return view('accounting.payroll.preparation.adjustments-modal', compact(
                'company',
                'period',
                'employee',
                'earningTypes',
                'adjustments'
            ));
        }
        
        // Otherwise return full page
        return view('accounting.payroll.preparation.adjustments-modal-full', compact(
            'company',
            'period',
            'employee',
            'earningTypes',
            'adjustments'
        ));
    }
    
    // Placeholder methods that were referenced in routes but not implemented
    public function employeesImport(Request $request)
    {
        // Import employees from CSV/Excel
        return redirect()->back()->with('error', 'Import functionality not implemented yet.');
    }

    public function employeesTemplateDownload()
    {
        // Download import template
        return redirect()->back()->with('error', 'Template download not implemented yet.');
    }

    public function earningTypesEdit($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $earningType = EarningType::where('company_id', $company->id)->findOrFail($id);
        
        return view('accounting.payroll.earnings.types.edit', compact('company', 'earningType'));
    }

    public function earningTypesDestroy($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $earningType = EarningType::where('company_id', $company->id)->findOrFail($id);
        
        try {
            $earningType->delete();
            return redirect()->route('admin.accounting.payroll.earnings.types.index')
                ->with('success', 'Earning type deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to delete earning type: ' . $e->getMessage());
        }
    }

    public function earningTypesToggleStatus($id)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $earningType = EarningType::where('company_id', $company->id)->findOrFail($id);
        
        try {
            $earningType->update(['is_active' => !$earningType->is_active]);
            return redirect()->back()->with('success', 'Earning type status updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to update status: ' . $e->getMessage());
        }
    }

    public function employeeEarningsEdit($employeeId, $earningId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $employee = Employee::where('company_id', $company->id)->findOrFail($employeeId);
        $earning = EmployeeEarning::where('employee_id', $employeeId)->findOrFail($earningId);
        $earningTypes = EarningType::where('company_id', $company->id)->where('is_active', true)->get();
        
        return view('accounting.payroll.earnings.employee.edit', compact('company', 'employee', 'earning', 'earningTypes'));
    }

    public function employeeEarningsUpdate(Request $request, $employeeId, $earningId)
    {
        $request->validate([
            'earning_type_id' => 'required|exists:earning_types,id',
            'amount' => 'required|numeric|min:0',
            'effective_date' => 'required|date',
            'end_date' => 'nullable|date|after:effective_date',
            'is_recurring' => 'required|boolean',
            'frequency' => 'required|in:monthly,quarterly,annually,one-time',
            'notes' => 'nullable|string',
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $earning = EmployeeEarning::where('employee_id', $employeeId)->findOrFail($earningId);
        
        try {
            $earning->update([
                'earning_type_id' => $request->earning_type_id,
                'amount' => $request->amount,
                'effective_date' => $request->effective_date,
                'end_date' => $request->end_date,
                'is_recurring' => $request->is_recurring,
                'frequency' => $request->frequency,
                'notes' => $request->notes,
                'updated_by' => auth()->id()
            ]);
            
            return redirect()->route('admin.accounting.payroll.earnings.employee.index', $employeeId)
                ->with('success', 'Employee earning updated successfully.');
                
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to update earning: ' . $e->getMessage())->withInput();
        }
    }

    public function employeeEarningsDestroy($employeeId, $earningId)
    {
        $earning = EmployeeEarning::where('employee_id', $employeeId)->findOrFail($earningId);
        
        try {
            $earning->delete();
            return redirect()->route('admin.accounting.payroll.earnings.employee.index', $employeeId)
                ->with('success', 'Employee earning deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to delete earning: ' . $e->getMessage());
        }
    }

    public function employeeEarningsToggleStatus($employeeId, $earningId)
    {
        $earning = EmployeeEarning::where('employee_id', $employeeId)->findOrFail($earningId);
        
        try {
            $earning->update(['is_active' => !$earning->is_active]);
            return redirect()->back()->with('success', 'Earning status updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to update status: ' . $e->getMessage());
        }
    }

    // Placeholder methods for bulk operations
    public function earningsBulkImport()
    {
        return view('accounting.payroll.earnings.bulk.import');
    }

    public function earningsBulkImportStore(Request $request)
    {
        return redirect()->back()->with('error', 'Bulk import not implemented yet.');
    }

    public function earningsTemplateDownload()
    {
        return redirect()->back()->with('error', 'Template download not implemented yet.');
    }

    public function earningsBulkAssign(Request $request)
    {
        return redirect()->back()->with('error', 'Bulk assign not implemented yet.');
    }

    public function earningsReportsIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        return view('accounting.payroll.earnings.reports.index', compact('company'));
    }

    public function earningsSummaryReport()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        return view('accounting.payroll.earnings.reports.summary', compact('company'));
    }

    public function taxableEarningsReport()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        return view('accounting.payroll.earnings.reports.taxable', compact('company'));
    }

    // Placeholder for payroll processing methods
    public function processingReverse($runId)
    {
        return redirect()->back()->with('error', 'Reverse functionality not implemented yet.');
    }

    public function processingShow($runId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $payrollRun = PayrollRun::where('company_id', $company->id)->findOrFail($runId);
        
        return view('accounting.payroll.processing.show', compact('company', 'payrollRun'));
    }

    public function processingEntryShow($entryId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $entry = PayrollEntry::where('company_id', $company->id)->findOrFail($entryId);
        
        return view('accounting.payroll.processing.entry-show', compact('company', 'entry'));
    }

    // Placeholder for report generation
    private function generatePayrollReport($payrollRun)
    {
        // Generate PDF/Excel report
        return "Payroll Report for " . $payrollRun->run_number;
    }

    // Placeholder for report methods
    public function reportsIndex()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        return view('accounting.payroll.reports.index', compact('company'));
    }

    public function payrollSummary()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        return view('accounting.payroll.reports.payroll-summary', compact('company'));
    }

    public function payeReport()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        return view('accounting.payroll.reports.paye-report', compact('company'));
    }

    public function napsaReport()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        return view('accounting.payroll.reports.napsa-report', compact('company'));
    }

    public function nhimaReport()
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        return view('accounting.payroll.reports.nhima-report', compact('company'));
    }

    public function employeePayslip($entryId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        $entry = PayrollEntry::where('company_id', $company->id)->findOrFail($entryId);
        
        return view('accounting.payroll.reports.employee-payslip', compact('company', 'entry'));
    }
    
    /**
     * Get simple payroll counts for sidebar menu
     */
    public static function getPayrollMenuCounts()
    {
        if (!auth()->check()) {
            return [
                'pending_review_count' => 0,
                'approved_pending_count' => 0,
                'paid_recent_count' => 0
            ];
        }
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        // 1. Payroll Pending Review (calculated but not approved)
        $pendingReviewCount = PayrollEntry::where('company_id', $company->id)
            ->where('calculation_status', 'calculated')
            ->where('payment_status', 'pending')
            ->count();
        
        // 2. Approved Payroll (approved but not paid)
        $approvedPendingCount = PayrollEntry::where('company_id', $company->id)
            ->where('calculation_status', 'approved')
            ->where('payment_status', 'pending')
            ->count();
        
        // 3. Paid Payroll (recently paid - last 30 days)
        $paidRecentCount = PayrollEntry::where('company_id', $company->id)
            ->where('calculation_status', 'approved')
            ->where('payment_status', 'paid')
            ->where('updated_at', '>=', now()->subDays(30))
            ->count();
        
        return [
            'pending_review_count' => $pendingReviewCount,
            'approved_pending_count' => $approvedPendingCount,
            'paid_recent_count' => $paidRecentCount
        ];
    }
    
    // ============================================
    // MISSING METHODS ADDED HERE
    // ============================================
    
    /**
     * Get entry details via AJAX
     */
    public function getEntryDetails($entryId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $entry = PayrollEntry::where('company_id', $company->id)
            ->with(['employee', 'payrollPeriod', 'payrollRun'])
            ->findOrFail($entryId);
        
        // Create a simple HTML response for details
        $html = '<div class="space-y-4">';
        
        // Employee Information
        $html .= '<div class="bg-gray50 p-4 rounded-lg">
            <h4 class="font-semibold text-gray900 mb-2">Employee Information</h4>
            <div class="grid grid-cols-2 gap-3 text-sm">
                <div><span class="text-gray600">Name:</span> <span class="font-medium">' . ($entry->employee->full_name ?? 'N/A') . '</span></div>
                <div><span class="text-gray600">Employee No:</span> <span class="font-medium">' . ($entry->employee->employee_number ?? 'N/A') . '</span></div>
                <div><span class="text-gray600">Email:</span> <span class="font-medium">' . ($entry->employee->email ?? 'N/A') . '</span></div>
                <div><span class="text-gray600">Payment Method:</span> <span class="font-medium">' . ucfirst($entry->employee->payment_method ?? 'N/A') . '</span></div>
            </div>
        </div>';
        
        // Period Information
        $html .= '<div class="bg-gray50 p-4 rounded-lg">
            <h4 class="font-semibold text-gray900 mb-2">Payroll Period</h4>
            <div class="grid grid-cols-2 gap-3 text-sm">
                <div><span class="text-gray600">Period:</span> <span class="font-medium">' . ($entry->payrollPeriod->period_name ?? 'N/A') . '</span></div>
                <div><span class="text-gray600">Date Range:</span> <span class="font-medium">' . ($entry->payrollPeriod->formatted_date_range ?? 'N/A') . '</span></div>
                <div><span class="text-gray600">Entry Date:</span> <span class="font-medium">' . ($entry->entry_date ? Carbon::parse($entry->entry_date)->format('M d, Y') : 'N/A') . '</span></div>
                <div><span class="text-gray600">Status:</span> <span class="font-medium">' . ucfirst($entry->payrollPeriod->status ?? 'N/A') . '</span></div>
            </div>
        </div>';
        
        // Earnings Summary
        $html .= '<div class="bg-gray50 p-4 rounded-lg">
            <h4 class="font-semibold text-gray900 mb-2">Earnings Summary</h4>
            <div class="grid grid-cols-2 gap-3 text-sm">
                <div><span class="text-gray600">Gross Earnings:</span> <span class="font-medium">' . ($company->currency_symbol ?? 'K') . number_format($entry->gross_earnings, 2) . '</span></div>
                <div><span class="text-gray600">Taxable Earnings:</span> <span class="font-medium">' . ($company->currency_symbol ?? 'K') . number_format($entry->total_taxable_earnings, 2) . '</span></div>
                <div><span class="text-gray600">Non-Taxable Earnings:</span> <span class="font-medium">' . ($company->currency_symbol ?? 'K') . number_format($entry->total_non_taxable_earnings, 2) . '</span></div>
                <div><span class="text-gray600">Basic Salary (NAPSA):</span> <span class="font-medium">' . ($company->currency_symbol ?? 'K') . number_format($entry->basic_salary_for_napsa, 2) . '</span></div>
            </div>
        </div>';
        
        // Deductions Summary
        $html .= '<div class="bg-gray50 p-4 rounded-lg">
            <h4 class="font-semibold text-gray900 mb-2">Deductions</h4>
            <div class="grid grid-cols-2 gap-3 text-sm">
                <div><span class="text-gray600">PAYE Tax:</span> <span class="font-medium text-danger">' . ($company->currency_symbol ?? 'K') . number_format($entry->paye_tax, 2) . '</span></div>
                <div><span class="text-gray600">NAPSA (Employee):</span> <span class="font-medium text-danger">' . ($company->currency_symbol ?? 'K') . number_format($entry->napsa_employee, 2) . '</span></div>
                <div><span class="text-gray600">NHIMA (Employee):</span> <span class="font-medium text-danger">' . ($company->currency_symbol ?? 'K') . number_format($entry->nhima_employee, 2) . '</span></div>
                <div><span class="text-gray600">Total Deductions:</span> <span class="font-medium text-danger">' . ($company->currency_symbol ?? 'K') . number_format($entry->total_deductions, 2) . '</span></div>
            </div>
        </div>';
        
        // Net Pay
        $html .= '<div class="bg-green-50 p-4 rounded-lg border border-green-200">
            <h4 class="font-semibold text-green-900 mb-2">Net Pay</h4>
            <div class="text-center">
                <div class="text-2xl font-bold text-success">' . ($company->currency_symbol ?? 'K') . number_format($entry->net_pay, 2) . '</div>
                <p class="text-sm text-green-700 mt-1">Amount to be paid to employee</p>
            </div>
        </div>';
        
        // Status Information
        $html .= '<div class="bg-gray50 p-4 rounded-lg">
            <h4 class="font-semibold text-gray900 mb-2">Status Information</h4>
            <div class="grid grid-cols-2 gap-3 text-sm">';
        
        // Calculation Status Badge
        $calcBadgeClass = $entry->calculation_status == 'approved' ? 'bg-green-100 text-green-800' : 'bg-blue-100 text-blue-800';
        $html .= '<div><span class="text-gray600">Calculation Status:</span> 
            <span class="quickbooks-badge ' . $calcBadgeClass . '">
                ' . ucfirst($entry->calculation_status) . '
            </span>
        </div>';
        
        // Payment Status Badge
        $paymentBadgeClass = $entry->payment_status == 'paid' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800';
        $html .= '<div><span class="text-gray600">Payment Status:</span> 
            <span class="quickbooks-badge ' . $paymentBadgeClass . '">
                ' . ucfirst($entry->payment_status) . '
            </span>
        </div>';
        
        if ($entry->approved_by) {
            $approver = User::find($entry->approved_by);
            $html .= '<div><span class="text-gray600">Approved By:</span> <span class="font-medium">' . ($approver->name ?? 'User #' . $entry->approved_by) . '</span></div>';
        }
        
        if ($entry->approved_at) {
            $html .= '<div><span class="text-gray600">Approved At:</span> <span class="font-medium">' . Carbon::parse($entry->approved_at)->format('M d, Y H:i') . '</span></div>';
        }
        
        $html .= '</div></div></div>';
        
        return response()->json([
            'success' => true,
            'html' => $html,
            'employee_name' => $entry->employee->full_name ?? 'N/A',
            'period_name' => $entry->payrollPeriod->period_name ?? 'N/A',
            'entry_date' => $entry->entry_date ? Carbon::parse($entry->entry_date)->format('M d, Y') : 'N/A'
        ]);
    }
    
    /**
     * Approve a single payroll entry
     */
    public function approveEntry(Request $request, $entryId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $entry = PayrollEntry::where('company_id', $company->id)
            ->findOrFail($entryId);
        
        // Check if entry can be approved
        if ($entry->calculation_status !== 'calculated') {
            return response()->json([
                'success' => false,
                'message' => 'Entry cannot be approved. Current status: ' . $entry->calculation_status
            ], 400);
        }
        
        DB::beginTransaction();
        
        try {
            $entry->update([
                'calculation_status' => 'approved',
                'approved_by' => auth()->id(),
                'approved_at' => now(),
                'updated_by' => auth()->id()
            ]);
            
            // If this entry belongs to a payroll run, check if all entries are approved
            if ($entry->payroll_run_id) {
                $run = PayrollRun::find($entry->payroll_run_id);
                if ($run) {
                    $unapprovedEntries = PayrollEntry::where('payroll_run_id', $run->id)
                        ->where('calculation_status', '!=', 'approved')
                        ->count();
                    
                    // If all entries in the run are approved, update run status
                    if ($unapprovedEntries === 0) {
                        $run->update([
                            'status' => 'approved',
                            'approved_by' => auth()->id(),
                            'approved_at' => now()
                        ]);
                    }
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Payroll entry approved successfully.',
                'entry_id' => $entry->id,
                'new_status' => 'approved'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to approve entry: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Process payment for a single entry
     */
    public function processEntryPayment(Request $request, $entryId)
    {
        $request->validate([
            'payment_date' => 'required|date',
            'payment_reference' => 'nullable|string|max:100'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $entry = PayrollEntry::where('company_id', $company->id)
            ->findOrFail($entryId);
        
        // Check if entry can be paid
        if ($entry->calculation_status !== 'approved' || $entry->payment_status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Entry cannot be paid. Current status: ' . $entry->calculation_status . '/' . $entry->payment_status
            ], 400);
        }
        
        DB::beginTransaction();
        
        try {
            $entry->update([
                'payment_status' => 'paid',
                'payment_date' => $request->payment_date,
                'payment_reference' => $request->payment_reference,
                'updated_by' => auth()->id()
            ]);
            
            // If this entry belongs to a payroll run, check if all entries are paid
            if ($entry->payroll_run_id) {
                $run = PayrollRun::find($entry->payroll_run_id);
                if ($run) {
                    $unpaidEntries = PayrollEntry::where('payroll_run_id', $run->id)
                        ->where('payment_status', '!=', 'paid')
                        ->count();
                    
                    if ($unpaidEntries === 0) {
                        $run->update([
                            'status' => 'paid',
                            'paid_by' => auth()->id(),
                            'paid_at' => now()
                        ]);
                    }
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Payroll entry marked as paid successfully.',
                'entry_id' => $entry->id,
                'payment_date' => $request->payment_date
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to process payment: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Export single entry
     */
    public function exportEntry($entryId)
    {
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        $entry = PayrollEntry::where('company_id', $company->id)
            ->with(['employee', 'payrollPeriod'])
            ->findOrFail($entryId);
        
        // Generate CSV for single entry
        $filename = 'payroll-entry-' . ($entry->employee->employee_number ?? 'entry') . '-' . date('Y-m-d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function () use ($entry, $company) {
            $file = fopen('php://output', 'w');
            
            // CSV header
            fputcsv($file, [
                'Employee Number', 'Employee Name', 'Period', 
                'Gross Earnings', 'Taxable Earnings', 'Non-Taxable Earnings',
                'PAYE Tax', 'NAPSA (Employee)', 'NHIMA (Employee)', 
                'Total Deductions', 'Net Pay', 'Status'
            ]);
            
            fputcsv($file, [
                $entry->employee->employee_number ?? 'N/A',
                $entry->employee->full_name,
                $entry->payrollPeriod->period_name ?? 'N/A',
                number_format($entry->gross_earnings, 2, '.', ''),
                number_format($entry->total_taxable_earnings, 2, '.', ''),
                number_format($entry->total_non_taxable_earnings, 2, '.', ''),
                number_format($entry->paye_tax, 2, '.', ''),
                number_format($entry->napsa_employee, 2, '.', ''),
                number_format($entry->nhima_employee, 2, '.', ''),
                number_format($entry->total_deductions, 2, '.', ''),
                number_format($entry->net_pay, 2, '.', ''),
                $entry->calculation_status . '/' . $entry->payment_status
            ]);
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    /**
     * Bulk approve entries
     */
    public function bulkApproveEntries(Request $request)
    {
        $request->validate([
            'entry_ids' => 'required|array',
            'entry_ids.*' => 'exists:payroll_entries,id'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        DB::beginTransaction();
        
        try {
            $approvedCount = 0;
            foreach ($request->entry_ids as $entryId) {
                $entry = PayrollEntry::where('company_id', $company->id)
                    ->find($entryId);
                
                if ($entry && $entry->calculation_status === 'calculated') {
                    $entry->update([
                        'calculation_status' => 'approved',
                        'approved_by' => auth()->id(),
                        'approved_at' => now(),
                        'updated_by' => auth()->id()
                    ]);
                    $approvedCount++;
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => $approvedCount . ' payroll entries approved successfully.',
                'approved_count' => $approvedCount
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to approve entries: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Bulk pay entries
     */
    public function bulkPayEntries(Request $request)
    {
        $request->validate([
            'entry_ids' => 'required|array',
            'entry_ids.*' => 'exists:payroll_entries,id',
            'payment_date' => 'required|date',
            'payment_reference' => 'nullable|string|max:100'
        ]);
        
        $company = Company::where('id', auth()->user()->companyid)->firstOrFail();
        
        DB::beginTransaction();
        
        try {
            $paidCount = 0;
            foreach ($request->entry_ids as $entryId) {
                $entry = PayrollEntry::where('company_id', $company->id)
                    ->find($entryId);
                
                if ($entry && $entry->calculation_status === 'approved' && $entry->payment_status === 'pending') {
                    $entry->update([
                        'payment_status' => 'paid',
                        'payment_date' => $request->payment_date,
                        'payment_reference' => $request->payment_reference,
                        'updated_by' => auth()->id()
                    ]);
                    $paidCount++;
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => $paidCount . ' payroll entries marked as paid successfully.',
                'paid_count' => $paidCount
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to process payments: ' . $e->getMessage()
            ], 500);
        }
    }
}