<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PrimaryCategory;
use App\Models\ReportType;

class PrimaryCategoryController extends Controller
{
    public function index()
    {
        $primaryCategories = PrimaryCategory::with(['reportType'])
            ->orderBy('sequence')
            ->get();
            
        $reportTypes = ReportType::where('is_active', true)->get();
        
        return view('accounting.settings.primary-categories.index', compact('primaryCategories', 'reportTypes'));
    }

    public function create()
    {
        $reportTypes = ReportType::where('is_active', true)->get();
        
        return view('accounting.settings.primary-categories.create', compact('reportTypes'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required',
            // 'code' => 'required|unique:primary_categories,code', // REMOVED
            'report_type_id' => 'required|exists:report_types,id',
            'description' => 'nullable',
            'sequence' => 'required|integer|min:0',
            'is_active' => 'boolean',
        ]);

        // Add authenticated user as creator
        $validated['created_by'] = auth()->id();
        $validated['updated_by'] = auth()->id();

        PrimaryCategory::create($validated);

        return redirect()->route('admin.accounting.settings.primary-categories.index')
            ->with('success', 'Primary category created successfully.');
    }

    public function show($id)
    {
        $primaryCategory = PrimaryCategory::with(['reportType', 'accountGroups'])->findOrFail($id);
        
        return view('accounting.settings.primary-categories.show', compact('primaryCategory'));
    }

    public function edit($id)
    {
        $primaryCategory = PrimaryCategory::findOrFail($id);
        $reportTypes = ReportType::where('is_active', true)->get();
        
        return view('accounting.settings.primary-categories.edit', compact('primaryCategory', 'reportTypes'));
    }

    public function update(Request $request, $id)
    {
        $primaryCategory = PrimaryCategory::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required',
            // 'code' => 'required|unique:primary_categories,code,' . $id, // REMOVED
            'report_type_id' => 'required|exists:report_types,id',
            'description' => 'nullable',
            'sequence' => 'required|integer|min:0',
            'is_active' => 'boolean',
        ]);

        // Add authenticated user as updater
        $validated['updated_by'] = auth()->id();

        $primaryCategory->update($validated);

        return redirect()->route('admin.accounting.settings.primary-categories.index')
            ->with('success', 'Primary category updated successfully.');
    }

    public function destroy($id)
    {
        $primaryCategory = PrimaryCategory::findOrFail($id);
        
        // Check if category has account groups
        if ($primaryCategory->accountGroups()->count() > 0) {
            return redirect()->route('admin.accounting.settings.primary-categories.index')
                ->with('error', 'Cannot delete primary category that has account groups assigned.');
        }

        $primaryCategory->delete();

        return redirect()->route('admin.accounting.settings.primary-categories.index')
            ->with('success', 'Primary category deleted successfully.');
    }
}