<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\LoanType;
use App\Models\ApplicationRequirement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class ProductController extends Controller
{
    /**
     * Display a listing of loan products with statistics.
     */
    public function index()
    {
        $loanTypes = LoanType::orderBy('product')->paginate(20);
        
        $stats = [
            'total' => LoanType::count(),
            'active' => LoanType::where('status', 'Active')->count(),
            'inactive' => LoanType::where('status', 'Inactive')->count(),
            // FIXED: Using correct table name 'applications' based on your provided schema
            'totalApplications' => DB::table('applications')->count(), 
        ];
        
        return view('accounting.products.index', compact('loanTypes', 'stats'));
    }

    /**
     * Show form to create a new product.
     */
    public function create()
    {
        return view('accounting.products.create');
    }

    /**
     * Store a newly created loan product.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product' => 'required|string|max:255',
            'details' => 'required|string|max:255',
            'status' => 'required|in:Active,Inactive',
            'interest_method' => 'required|in:simple_interest,reducing_balance',
            'dailyrate' => 'nullable|numeric|min:0',
            'weeklyrate' => 'nullable|numeric|min:0',
            'monthlyrate' => 'nullable|numeric|min:0',
            'processing_fee' => 'nullable|numeric|min:0',
            'processing_fee_basis' => 'required_if:processing_fee,>,0|in:initial_amount,outstanding_balance',
            'penalty_rate' => 'nullable|numeric|min:0',
            'penalty_basis' => 'required_if:penalty_rate,>,0|in:fixed,percentage_of_due,percentage_of_outstanding',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            $monthly = $request->monthlyrate ?? 0;
            // Computed Logic: Yearly Rate = Monthly Rate * 12
            $computedYearly = $monthly * 12;

            LoanType::create([
                'product' => $request->product,
                'details' => $request->details,
                'status' => $request->status,
                'interest_method' => $request->interest_method, // NEW
                'dailyrate' => $request->dailyrate ?? 0,
                'weeklyrate' => $request->weeklyrate ?? 0,
                'monthlyrate' => $monthly,
                'yearlyrate' => $computedYearly, // Auto-computed
                'processing_fee' => $request->processing_fee ?? 0,
                'processing_fee_basis' => $request->processing_fee_basis ?? 'initial_amount',
                'penalty_rate' => $request->penalty_rate ?? 0,
                'penalty_basis' => $request->penalty_basis ?? 'fixed',
                'userid' => auth()->id(),
                'branchid' => auth()->user()->branch_id ?? 1,
                'companyid' => auth()->user()->company_id ?? 1,
                'prate' => $request->prate ?? 0,
                'adminfee' => $request->adminfee ?? 0,
                'insurancefee' => $request->insurancefee ?? 0,
                'mouquestion' => $request->mouquestion,
            ]);

            DB::commit();

            return redirect()->route('admin.accounting.products.index')
                ->with('success', 'Product created successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Product Creation Error: " . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Error creating product: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Show specific product details.
     */
    public function show($id)
    {
        $loanType = LoanType::findOrFail($id);
        return view('accounting.products.show', compact('loanType'));
    }

    /**
     * Show form to edit an existing product.
     */
    public function edit($id)
    {
        $loanType = LoanType::findOrFail($id);
        return view('accounting.products.edit', compact('loanType'));
    }

    /**
     * Update existing loan product.
     */
    public function update(Request $request, $id)
    {
        $loanType = LoanType::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'product' => 'required|string|max:255',
            'details' => 'required|string|max:255',
            'status' => 'required|in:Active,Inactive',
            'interest_method' => 'required|in:simple_interest,reducing_balance',
            'monthlyrate' => 'nullable|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            $data = $request->all();
            
            // Re-compute Yearly Rate if Monthly Rate changed
            if (isset($data['monthlyrate'])) {
                $data['yearlyrate'] = $data['monthlyrate'] * 12;
            }

            $loanType->update($data);

            DB::commit();
            return redirect()->route('admin.accounting.products.index')
                ->with('success', 'Product updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error updating product: " . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Update Error: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the product.
     */
    public function destroy($id)
    {
        $loanType = LoanType::findOrFail($id);
        $loanType->delete();
        
        return redirect()->route('admin.accounting.products.index')
            ->with('success', 'Product deleted successfully!');
    }

    /**
     * Toggle product status via AJAX.
     */
    public function toggleStatus($id)
    {
        $loanType = LoanType::findOrFail($id);
        $loanType->update([
            'status' => $loanType->status === 'Active' ? 'Inactive' : 'Active'
        ]);

        return response()->json(['success' => true]);
    }

    /**
     * Show requirement assignment view.
     */
    public function assignRequirements($id)
    {
        $loanType = LoanType::findOrFail($id);
        $allRequirements = ApplicationRequirement::orderBy('documentname')->get();
        $assignedRequirements = $loanType->requirements()->pluck('requirement_id')->toArray();
        
        // Define categories for grouping requirements
        $categories = [
            'Identification' => ['passport', 'id', 'drivers', 'license', 'national', 'voters', 'card', 'alien'],
            'Proof of Income' => ['payslip', 'salary', 'income', 'bank statement', 'employment', 'contract', 'business', 'tax'],
            'Proof of Residence' => ['utility', 'bill', 'water', 'electricity', 'lease', 'rental', 'residence', 'address'],
            'Financial Documents' => ['bank', 'statement', 'account', 'transaction', 'credit', 'loan', 'debt'],
            'Business Documents' => ['business', 'registration', 'certificate', 'license', 'tax', 'vat', 'permit'],
            'Property Documents' => ['title', 'deed', 'property', 'land', 'house', 'valuation', 'survey'],
            'Vehicle Documents' => ['logbook', 'vehicle', 'car', 'registration', 'insurance'],
            'Employment Documents' => ['employment', 'letter', 'appointment', 'confirmation', 'reference'],
            'Other Documents' => [] // Default category
        ];
        
        return view('accounting.products.assign-requirements', [
            'loanType' => $loanType,
            'allRequirements' => $allRequirements,
            'assignedRequirements' => $assignedRequirements,
            'categories' => $categories, // Now includes categories
        ]);
    }

    /**
     * Save updated requirements.
     */
    public function updateRequirements(Request $request, $id)
    {
        $loanType = LoanType::findOrFail($id);
        
        $request->validate([
            'requirements' => 'nullable|array',
            'requirements.*' => 'exists:applicationrequirements,id',
            'mandatory' => 'nullable|array',
            'mandatory.*' => 'exists:applicationrequirements,id',
        ]);
        
        $requirements = $request->input('requirements', []);
        $mandatory = $request->input('mandatory', []);
        
        $syncData = [];
        foreach ($requirements as $requirementId) {
            $syncData[$requirementId] = [
                'is_mandatory' => in_array($requirementId, $mandatory)
            ];
        }
        
        $loanType->requirements()->sync($syncData);
        
        return redirect()
            ->route('admin.accounting.products.assign-requirements', $loanType->id)
            ->with('success', 'Requirements updated successfully.');
    }
}