<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ReportType;

class ReportTypeController extends Controller
{
    public function index()
    {
        $reportTypes = ReportType::orderBy('name')->get();
        
        return view('accounting.settings.report-types.index', compact('reportTypes'));
    }

    public function create()
    {
        return view('accounting.settings.report-types.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required',
            'description' => 'nullable',
            'frequency' => 'required|in:daily,weekly,monthly,quarterly,annually',
            'is_active' => 'boolean',
        ]);

        // Add default values for required fields
        $validated['company_id'] = 1; // Set default company ID
        $validated['branch_id'] = 1;  // Set default branch ID
        $validated['created_by'] = auth()->id();
        $validated['updated_by'] = auth()->id();

        ReportType::create($validated);

        return redirect()->route('admin.accounting.settings.report-types.index')
            ->with('success', 'Report type created successfully.');
    }

    public function show($id)
    {
        $reportType = ReportType::with(['primaryCategories'])->findOrFail($id);
        
        return view('accounting.settings.report-types.show', compact('reportType'));
    }

    public function edit($id)
    {
        $reportType = ReportType::findOrFail($id);
        
        return view('accounting.settings.report-types.edit', compact('reportType'));
    }

    public function update(Request $request, $id)
    {
        $reportType = ReportType::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required',
            'description' => 'nullable',
            'frequency' => 'required|in:daily,weekly,monthly,quarterly,annually',
            'is_active' => 'boolean',
        ]);

        // Add authenticated user as updater
        $validated['updated_by'] = auth()->id();

        $reportType->update($validated);

        return redirect()->route('admin.accounting.settings.report-types.index')
            ->with('success', 'Report type updated successfully.');
    }

    public function destroy($id)
    {
        $reportType = ReportType::findOrFail($id);
        
        // Check if report type has primary categories
        if ($reportType->primaryCategories()->count() > 0) {
            return redirect()->route('admin.accounting.settings.report-types.index')
                ->with('error', 'Cannot delete report type that has primary categories assigned.');
        }

        $reportType->delete();

        return redirect()->route('admin.accounting.settings.report-types.index')
            ->with('success', 'Report type deleted successfully.');
    }
}
