<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Accounting\SupplierInvoice;
use App\Models\Accounting\Supplier;
use App\Models\Accounting\ChartOfAccount;

class SupplierInvoiceController extends Controller
{
    public function index()
    {
        $invoices = SupplierInvoice::with(['supplier'])->orderBy('invoice_date', 'desc')->get();
        $suppliers = Supplier::where('is_active', true)->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.payables.supplier-invoices.index', compact('invoices', 'suppliers', 'accounts'));
    }

    public function create()
    {
        $suppliers = Supplier::where('is_active', true)->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.payables.supplier-invoices.create', compact('suppliers', 'accounts'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'invoice_number' => 'required|unique:supplier_invoices,invoice_number',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date',
            'total_amount' => 'required|numeric|min:0',
            'tax_amount' => 'numeric|min:0',
            'description' => 'nullable',
            'status' => 'required|in:draft,unpaid,paid,overdue,cancelled',
            'account_id' => 'required|exists:chart_of_accounts,id',
        ]);

        SupplierInvoice::create($validated);

        return redirect()->route('admin.accounting.payables.supplier-invoices.index')
            ->with('success', 'Supplier invoice created successfully.');
    }

    public function show($id)
    {
        $invoice = SupplierInvoice::with(['supplier', 'account'])->findOrFail($id);
        return view('accounting.payables.supplier-invoices.show', compact('invoice'));
    }

    public function edit($id)
    {
        $invoice = SupplierInvoice::findOrFail($id);
        $suppliers = Supplier::where('is_active', true)->get();
        $accounts = ChartOfAccount::where('is_active', true)->get();
        
        return view('accounting.payables.supplier-invoices.edit', compact('invoice', 'suppliers', 'accounts'));
    }

    public function update(Request $request, $id)
    {
        $invoice = SupplierInvoice::findOrFail($id);
        
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'invoice_number' => 'required|unique:supplier_invoices,invoice_number,' . $id,
            'invoice_date' => 'required|date',
            'due_date' => 'required|date',
            'total_amount' => 'required|numeric|min:0',
            'tax_amount' => 'numeric|min:0',
            'description' => 'nullable',
            'status' => 'required|in:draft,unpaid,paid,overdue,cancelled',
            'account_id' => 'required|exists:chart_of_accounts,id',
        ]);

        $invoice->update($validated);

        return redirect()->route('admin.accounting.payables.supplier-invoices.index')
            ->with('success', 'Supplier invoice updated successfully.');
    }

    public function destroy($id)
    {
        $invoice = SupplierInvoice::findOrFail($id);
        $invoice->delete();

        return redirect()->route('admin.accounting.payables.supplier-invoices.index')
            ->with('success', 'Supplier invoice deleted successfully.');
    }
}