<?php

namespace App\Http\Controllers\Accounting;

use App\Http\Controllers\Controller;
use App\Models\Vendor;
use App\Models\VendorContact;
use App\Models\VendorStatement;
use App\Models\VendorNote;
use App\Models\ChartOfAccount;
use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class VendorController extends Controller
{
    /**
     * Display a listing of the vendors.
     */
    public function index(Request $request)
    {
        $query = Vendor::with(['chartOfAccount', 'currencyDetail']);
        
        // Apply search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('display_name', 'like', "%{$search}%")
                  ->orWhere('company_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('mobile_phone', 'like', "%{$search}%")
                  ->orWhere('account_number', 'like', "%{$search}%");
            });
        }
        
        // Apply status filter
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->where('active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('active', false);
            }
        }
        
        // Apply balance filter
        if ($request->filled('balance')) {
            if ($request->balance === 'with_balance') {
                $query->where('balance', '>', 0);
            } elseif ($request->balance === 'no_balance') {
                $query->where('balance', '<=', 0);
            }
        }
        
        // Apply 1099 filter
        if ($request->filled('vendor_1099')) {
            if ($request->vendor_1099 === '1099') {
                $query->where('vendor_1099', true);
            } elseif ($request->vendor_1099 === 'non_1099') {
                $query->where('vendor_1099', false);
            }
        }
        
        // Apply sorting
        $sortField = $request->get('sort', 'display_name');
        $sortDirection = $request->get('direction', 'asc');
        $query->orderBy($sortField, $sortDirection);
        
        // Get paginated results
        $vendors = $query->paginate($request->get('per_page', 20));
        
        // Get statistics for dashboard
        $stats = [
            'total' => Vendor::count(),
            'active' => Vendor::where('active', true)->count(),
            'inactive' => Vendor::where('active', false)->count(),
            'with_balance' => Vendor::where('balance', '>', 0)->count(),
            'overdue' => Vendor::where('overdue', true)->count(),
            'vendor_1099' => Vendor::where('vendor_1099', true)->count(),
        ];
        
        if ($request->wantsJson() || $request->ajax()) {
            return response()->json([
                'vendors' => $vendors,
                'stats' => $stats,
                'filters' => $request->all(),
            ]);
        }
        
        return view('accounting.expenses.vendors.index', compact('vendors', 'stats'));
    }

    /**
     * Show the form for creating a new vendor.
     */
    public function create()
    {
        // Get chart of accounts for dropdown
        $accounts = ChartOfAccount::where('is_active', true)
            ->orderBy('code')
            ->get(['id', 'code', 'name']);
            
        // Get currencies from database
        $currencies = Currency::where('is_active', true)
            ->orderBy('code')
            ->get(['id', 'code', 'name', 'symbol']);
        
        return view('accounting.expenses.vendors.create', compact('accounts', 'currencies'));
    }

    /**
     * Store a newly created vendor in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'display_name' => 'required|string|max:255|unique:vendors',
            'company_name' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'mobile_phone' => 'nullable|string|max:50',
            'account_number' => 'nullable|string|max:100|unique:vendors,account_number',
            'chart_of_account_id' => 'nullable|exists:chart_of_accounts,id',
            'currency_id' => 'required|exists:currencies,id',
            'tax_id' => 'nullable|string|max:50',
            'balance' => 'nullable|numeric|min:0',
            'vendor_1099' => 'boolean',
            'active' => 'boolean',
        ], [
            'display_name.required' => 'Vendor display name is required',
            'display_name.unique' => 'A vendor with this display name already exists',
            'account_number.unique' => 'This account number is already in use',
            'chart_of_account_id.exists' => 'Selected account does not exist',
            'currency_id.required' => 'Currency is required',
            'currency_id.exists' => 'Selected currency does not exist',
        ]);
        
        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        DB::beginTransaction();
        try {
            $vendorData = $request->only([
                'display_name',
                'company_name',
                'first_name',
                'middle_name',
                'last_name',
                'suffix',
                'email',
                'mobile_phone',
                'other_phone',
                'fax',
                'website',
                'check_printing_name',
                'billing_street1',
                'billing_street2',
                'billing_city',
                'billing_state',
                'billing_zip',
                'billing_country',
                'shipping_street1',
                'shipping_street2',
                'shipping_city',
                'shipping_state',
                'shipping_zip',
                'shipping_country',
                'account_number',
                'chart_of_account_id',
                'currency_id',
                'payment_terms',
                'vendor_1099',
                'tax_id',
                'balance',
                'active',
            ]);
            
            // Set default values
            $vendorData['balance'] = $vendorData['balance'] ?? 0;
            $vendorData['active'] = $vendorData['active'] ?? true;
            $vendorData['vendor_1099'] = $vendorData['vendor_1099'] ?? false;
            
            // Get currency code from currency_id for backward compatibility
            if (isset($vendorData['currency_id'])) {
                $currency = Currency::find($vendorData['currency_id']);
                if ($currency) {
                    $vendorData['currency'] = $currency->code;
                } else {
                    $vendorData['currency'] = 'USD';
                }
            }
            
            $vendor = Vendor::create($vendorData);
            
            // Create primary contact if contact info provided
            if ($request->filled('email') || $request->filled('mobile_phone')) {
                $vendor->contacts()->create([
                    'contact_type' => 'primary',
                    'name' => $request->full_name,
                    'email' => $request->email,
                    'phone' => $request->mobile_phone,
                    'title' => $request->contact_title,
                    'is_primary' => true,
                ]);
            }
            
            DB::commit();
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Vendor created successfully',
                    'vendor' => $vendor,
                    'redirect' => route('admin.accounting.expenses.vendors.index')
                ]);
            }
            
            return redirect()->route('admin.accounting.expenses.vendors.index')
                ->with('success', 'Vendor created successfully');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create vendor: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->with('error', 'Failed to create vendor: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified vendor.
     */
    public function show($id)
    {
    // Eager load the relationships
    $vendor = Vendor::with(['contacts', 'statements', 'notes', 'chartOfAccount', 'currencyDetail'])->findOrFail($id);
    
    if (request()->ajax()) {
        return response()->json([
            'vendor' => $vendor,
            'contacts' => $vendor->contacts,
            'statements' => $vendor->statements,
            'notes' => $vendor->notes,
        ]);
    }
    
    return view('accounting.expenses.vendors.show', compact('vendor'));
    }

    /**
     * Show the form for editing the specified vendor.
     */
    public function edit($id)
    {
        $vendor = Vendor::with('contacts')->findOrFail($id);
        
        // Get chart of accounts for dropdown
        $accounts = ChartOfAccount::where('is_active', true)
            ->orderBy('code')
            ->get(['id', 'code', 'name']);
            
        // Get currencies from database
        $currencies = Currency::where('is_active', true)
            ->orderBy('code')
            ->get(['id', 'code', 'name', 'symbol']);
        
        if (request()->ajax()) {
            return response()->json([
                'vendor' => $vendor,
                'contacts' => $vendor->contacts,
                'accounts' => $accounts,
                'currencies' => $currencies,
            ]);
        }
        
        return view('accounting.expenses.vendors.edit', compact('vendor', 'accounts', 'currencies'));
    }

    /**
     * Update the specified vendor in storage.
     */
    public function update(Request $request, $id)
    {
        $vendor = Vendor::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'display_name' => 'required|string|max:255|unique:vendors,display_name,' . $vendor->id,
            'company_name' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'mobile_phone' => 'nullable|string|max:50',
            'account_number' => 'nullable|string|max:100|unique:vendors,account_number,' . $vendor->id,
            'chart_of_account_id' => 'nullable|exists:chart_of_accounts,id',
            'currency_id' => 'required|exists:currencies,id',
            'tax_id' => 'nullable|string|max:50',
            'balance' => 'nullable|numeric|min:0',
            'vendor_1099' => 'boolean',
            'active' => 'boolean',
        ], [
            'display_name.required' => 'Vendor display name is required',
            'display_name.unique' => 'A vendor with this display name already exists',
            'account_number.unique' => 'This account number is already in use',
            'chart_of_account_id.exists' => 'Selected account does not exist',
            'currency_id.required' => 'Currency is required',
            'currency_id.exists' => 'Selected currency does not exist',
        ]);
        
        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        DB::beginTransaction();
        try {
            $vendorData = $request->only([
                'display_name',
                'company_name',
                'first_name',
                'middle_name',
                'last_name',
                'suffix',
                'email',
                'mobile_phone',
                'other_phone',
                'fax',
                'website',
                'check_printing_name',
                'billing_street1',
                'billing_street2',
                'billing_city',
                'billing_state',
                'billing_zip',
                'billing_country',
                'shipping_street1',
                'shipping_street2',
                'shipping_city',
                'shipping_state',
                'shipping_zip',
                'shipping_country',
                'account_number',
                'chart_of_account_id',
                'currency_id',
                'payment_terms',
                'vendor_1099',
                'tax_id',
                'balance',
                'active',
            ]);
            
            // Get currency code from currency_id for backward compatibility
            if (isset($vendorData['currency_id'])) {
                $currency = Currency::find($vendorData['currency_id']);
                if ($currency) {
                    $vendorData['currency'] = $currency->code;
                } else {
                    $vendorData['currency'] = $vendor->currency;
                }
            }
            
            $vendor->update($vendorData);
            
            // Update or create primary contact
            if ($request->filled('email') || $request->filled('mobile_phone')) {
                $primaryContact = $vendor->primaryContact();
                
                if ($primaryContact) {
                    $primaryContact->update([
                        'name' => $request->full_name,
                        'email' => $request->email,
                        'phone' => $request->mobile_phone,
                        'title' => $request->contact_title,
                    ]);
                } else {
                    $vendor->contacts()->create([
                        'contact_type' => 'primary',
                        'name' => $request->full_name,
                        'email' => $request->email,
                        'phone' => $request->mobile_phone,
                        'title' => $request->contact_title,
                        'is_primary' => true,
                    ]);
                }
            }
            
            DB::commit();
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Vendor updated successfully',
                    'vendor' => $vendor,
                ]);
            }
            
            return redirect()->route('admin.accounting.expenses.vendors.index')
                ->with('success', 'Vendor updated successfully');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update vendor: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->with('error', 'Failed to update vendor: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified vendor from storage.
     */
    public function destroy($id)
    {
        $vendor = Vendor::findOrFail($id);
        
        try {
            $vendor->delete();
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Vendor deleted successfully'
                ]);
            }
            
            return redirect()->route('admin.accounting.expenses.vendors.index')
                ->with('success', 'Vendor deleted successfully');
                
        } catch (\Exception $e) {
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete vendor: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->route('admin.accounting.expenses.vendors.index')
                ->with('error', 'Failed to delete vendor: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete vendors.
     */
    public function bulkDestroy(Request $request)
    {
        $request->validate([
            'vendor_ids' => 'required|array',
            'vendor_ids.*' => 'exists:vendors,id'
        ]);
        
        try {
            $count = Vendor::whereIn('id', $request->vendor_ids)->delete();
            
            return response()->json([
                'success' => true,
                'message' => "Successfully deleted {$count} vendors",
                'count' => $count
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete vendors: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export vendors to CSV.
     */
    public function export(Request $request)
    {
        $query = Vendor::with(['chartOfAccount', 'currencyDetail']);
        
        if ($request->filled('vendor_ids')) {
            $query->whereIn('id', $request->vendor_ids);
        }
        
        $vendors = $query->get();
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="vendors_' . date('Y-m-d') . '.csv"',
        ];
        
        $callback = function() use ($vendors) {
            $file = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($file, [
                'ID',
                'Display Name',
                'Company Name',
                'Email',
                'Phone',
                'Account',
                'Currency',
                'Balance',
                'Payment Terms',
                '1099 Eligible',
                'Tax ID',
                'Status',
                'Created Date'
            ]);
            
            // Data rows
            foreach ($vendors as $vendor) {
                fputcsv($file, [
                    $vendor->id,
                    $vendor->display_name,
                    $vendor->company_name ?? '',
                    $vendor->email ?? '',
                    $vendor->primary_phone ?? '',
                    $vendor->account_name,
                    $vendor->currency_with_symbol,
                    number_format($vendor->balance, 2),
                    $vendor->payment_terms ?? '',
                    $vendor->vendor_1099 ? 'Yes' : 'No',
                    $vendor->tax_id ?? '',
                    $vendor->active ? 'Active' : 'Inactive',
                    $vendor->created_at->format('Y-m-d')
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }

    /**
     * Get vendor statistics.
     */
    public function statistics()
    {
        $stats = [
            'total' => Vendor::count(),
            'active' => Vendor::where('active', true)->count(),
            'inactive' => Vendor::where('active', false)->count(),
            'with_balance' => Vendor::where('balance', '>', 0)->count(),
            'overdue' => Vendor::where('overdue', true)->count(),
            'vendor_1099' => Vendor::where('vendor_1099', true)->count(),
            'total_balance' => Vendor::sum('balance'),
            'average_balance' => Vendor::avg('balance'),
        ];
        
        return response()->json([
            'success' => true,
            'statistics' => $stats
        ]);
    }

    /**
     * Toggle vendor active status.
     */
    public function toggleStatus($id)
    {
        try {
            $vendor = Vendor::findOrFail($id);
            $vendor->active = !$vendor->active;
            $vendor->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Vendor status updated successfully',
                'vendor' => $vendor
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update vendor status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate statement for a vendor.
     */
    public function generateStatement(Request $request, $id)
    {
        $vendor = Vendor::findOrFail($id);
        
        $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
        ]);
        
        try {
            // In a real app, you would generate a PDF statement here
            // For now, we'll create a record in the database
            
            $statement = $vendor->statements()->create([
                'statement_date' => now()->format('Y-m-d'),
                'period_start' => $request->period_start,
                'period_end' => $request->period_end,
                'opening_balance' => $vendor->balance,
                'closing_balance' => $vendor->balance,
                'total_charges' => 0,
                'total_payments' => 0,
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Statement generated successfully',
                'statement' => $statement,
                'download_url' => route('admin.accounting.expenses.vendors.statement.download', $statement->id)
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate statement: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get chart of accounts for AJAX dropdown
     */
    public function getChartOfAccounts(Request $request)
    {
        $accounts = ChartOfAccount::where('is_active', true)
            ->when($request->filled('search'), function($query) use ($request) {
                $search = $request->search;
                return $query->where(function($q) use ($search) {
                    $q->where('code', 'like', "%{$search}%")
                      ->orWhere('name', 'like', "%{$search}%");
                });
            })
            ->orderBy('code')
            ->get(['id', 'code', 'name']);
            
        return response()->json([
            'success' => true,
            'accounts' => $accounts
        ]);
    }
}