<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\CustomerCreditScore;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class CustomerController extends Controller
{
    /**
     * Display a listing of the customers.
     */
       public function index(Request $request)
{
    // 1. Check Permissions
    if (!auth()->user()->hasAnyPermission(['customers.view', 'customers.create', 'customers.edit', 'poweruser'])) {
        abort(403, 'You do not have permission to view customers.');
    }

    // 2. Capture dynamic inputs
    $search = $request->input('search');
    $perPage = $request->input('per_page', 20); // Defaults to 20 if not in URL

    // 3. Build the query
    $customers = Customer::query()
        ->with(['currentCreditScore', 'activeDisbursedLoans'])
        ->when($search, function ($query, $search) {
            return $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('surname', 'like', "%{$search}%")
                  ->orWhere('middle_name', 'like', "%{$search}%")
                  ->orWhere('customer_number', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('number', 'like', "%{$search}%");
            });
        })
        ->orderBy('id', 'desc')
        ->paginate($perPage) // Dynamic pagination length
        ->withQueryString(); // Keeps 'search' and 'per_page' in the pagination links

    return view('admin.customers.index', compact('customers'));
}


    /**
     * Show the form for creating a new customer.
     */
    public function create()
    {
        if (!auth()->user()->hasPermission('customers.create')) {
            abort(403, 'You do not have permission to create customers.');
        }

        return view('admin.customers.create');
    }

    /**
     * Store a newly created customer in storage.
     */
    public function store(Request $request)
    {
        if (!auth()->user()->hasPermission('customers.create')) {
            abort(403, 'You do not have permission to create customers.');
        }

        $validated = $request->validate([
            'title' => 'nullable|string|max:10',
            'surname' => 'required|string|max:100',
            'first_name' => 'required|string|max:100',
            'middle_name' => 'nullable|string|max:100',
            'email' => 'required|email|unique:customer,email',
            'phone' => 'required|string|max:20',
            'office_phone' => 'nullable|string|max:20',
            'town' => 'required|string|max:100',
            'address' => 'required|string',
            'province' => 'nullable|string|max:100',
            'occupation' => 'nullable|string|max:100',
            'income' => 'nullable|numeric|min:0',
            'company' => 'nullable|string|max:100',
            'dob' => 'nullable|date',
            'idtype' => 'nullable|string|max:50',
            'number' => 'nullable|string|max:50',
            'status' => 'required|string|in:Active,Inactive,Suspended,Pending',
        ]);

        try {
            DB::transaction(function () use ($validated) {
                $customer = Customer::create($validated);

                \Log::info('New customer created', [
                    'customer_id' => $customer->id,
                    'customer_number' => $customer->customer_number,
                    'created_by' => auth()->id()
                ]);
            });

            return redirect()->route('admin.customers.index')
                ->with('success', 'Customer created successfully.');

        } catch (\Exception $e) {
            \Log::error('Failed to create customer', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to create customer: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified customer.
     */
    public function show(Customer $customer)
    {
        if (!auth()->user()->hasAnyPermission(['customers.view', 'customers.edit', 'poweruser'])) {
            abort(403, 'You do not have permission to view customer details.');
        }

        $customer->load([
            'currentCreditScore',
            'creditScoreHistory' => function($query) {
                $query->orderBy('score_date', 'desc')->limit(10);
            },
            'activeDisbursedLoans',
            'applications' => function($query) {
                $query->orderBy('created_at', 'desc')->limit(5);
            },
            'disbursedLoans' => function($query) {
                $query->orderBy('disburseddate', 'desc')->limit(5);
            },
            'repayments' => function($query) {
                $query->orderBy('rdate', 'desc')->limit(10);
            }
        ]);

        $financialOverview = $customer->getFinancialOverview();

        return view('admin.customers.show', compact('customer', 'financialOverview'));
    }

    /**
     * Show the form for editing the specified customer.
     */
    public function edit(Customer $customer)
    {
        if (!auth()->user()->hasPermission('customers.edit')) {
            abort(403, 'You do not have permission to edit customers.');
        }

        return view('admin.customers.edit', compact('customer'));
    }

    /**
     * Update the specified customer in storage.
     */
    public function update(Request $request, Customer $customer)
    {
        if (!auth()->user()->hasPermission('customers.edit')) {
            abort(403, 'You do not have permission to edit customers.');
        }

        $validated = $request->validate([
            'title' => 'nullable|string|max:10',
            'surname' => 'required|string|max:100',
            'first_name' => 'required|string|max:100',
            'middle_name' => 'nullable|string|max:100',
            'email' => 'required|email|unique:customer,email,' . $customer->id,
            'phone' => 'required|string|max:20',
            'office_phone' => 'nullable|string|max:20',
            'town' => 'required|string|max:100',
            'address' => 'required|string',
            'province' => 'nullable|string|max:100',
            'occupation' => 'nullable|string|max:100',
            'income' => 'nullable|numeric|min:0',
            'company' => 'nullable|string|max:100',
            'dob' => 'nullable|date',
            'idtype' => 'nullable|string|max:50',
            'number' => 'nullable|string|max:50',
            'status' => 'required|string|in:Active,Inactive,Suspended,Pending',
            'verification_status' => 'nullable|string|in:verified,failed,skipped',
        ]);

        try {
            $customer->update($validated);

            \Log::info('Customer updated', [
                'customer_id' => $customer->id,
                'updated_by' => auth()->id()
            ]);

            return redirect()->route('admin.customers.show', $customer->id)
                ->with('success', 'Customer updated successfully.');

        } catch (\Exception $e) {
            \Log::error('Failed to update customer', [
                'customer_id' => $customer->id,
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to update customer: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified customer from storage.
     */
    public function destroy(Customer $customer)
    {
        if (!auth()->user()->hasPermission('customers.delete')) {
            abort(403, 'You do not have permission to delete customers.');
        }

        if ($customer->activeDisbursedLoans()->exists()) {
            return redirect()->back()
                ->with('error', 'Cannot delete customer with active loans. Please close all loans first.');
        }

        if ($customer->applications()->whereIn('status', ['submitted', 'under_review', 'approved'])->exists()) {
            return redirect()->back()
                ->with('error', 'Cannot delete customer with pending or approved applications.');
        }

        try {
            $customerId = $customer->id;
            $customerNumber = $customer->customer_number;
            
            $customer->delete();

            \Log::info('Customer deleted', [
                'customer_id' => $customerId,
                'customer_number' => $customerNumber,
                'deleted_by' => auth()->id()
            ]);

            return redirect()->route('admin.customers.index')
                ->with('success', 'Customer deleted successfully.');

        } catch (\Exception $e) {
            \Log::error('Failed to delete customer', [
                'customer_id' => $customer->id,
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to delete customer: ' . $e->getMessage());
        }
    }

    /**
     * Update customer verification status
     */
    public function verify(Customer $customer)
    {
        if (!auth()->user()->hasPermission('customers.edit')) {
            abort(403, 'You do not have permission to verify customers.');
        }

        try {
            $customer->update(['verification_status' => 'verified']);

            return redirect()->back()
                ->with('success', 'Customer marked as verified successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to verify customer: ' . $e->getMessage());
        }
    }
}