<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Application;
use App\Models\Loan;
use App\Models\Customer;
use App\Models\Repayment;

class DashboardController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        
        // Initialize empty arrays and collections
        $stats = [];
        $recentApplications = collect();
        $recentLoans = collect();
        $applicationProgress = collect();

        // Applications Module Statistics
        if ($user->hasAnyPermission(['applications.view', 'applications.approve', 'poweruser'])) {
            $stats['total_applications'] = Application::count();
            $stats['pending_applications'] = Application::where('status', 'submitted')->count();
            $stats['under_review_applications'] = Application::where('status', 'under_review')->count();
            $stats['approved_applications'] = Application::where('status', 'approved')->count();
            $stats['rejected_applications'] = Application::where('status', 'rejected')->count();
            
            // Recent applications for review
            $recentApplications = Application::with(['customer', 'product'])
                ->whereIn('status', ['submitted', 'under_review'])
                ->orderBy('created_at', 'desc')
                ->take(5)
                ->get();

            // Application progress data
            $applicationProgress = $this->getApplicationProgress();
        }

        // Loans Module Statistics
        if ($user->hasAnyPermission(['loans.view', 'poweruser'])) {
            $stats['total_loans'] = Loan::count();
            $stats['active_loans'] = Loan::where('status', 'active')->count();
            $stats['closed_loans'] = Loan::where('status', 'closed')->count();
            $stats['non_performing_loans'] = Loan::where('status', 'default')->count();
            $stats['total_loan_amount'] = Loan::sum('amount');
            $stats['total_outstanding'] = $this->calculateTotalOutstanding();
            $stats['total_collected'] = $this->calculateTotalCollected();
            
            // Recent loan disbursements
            $recentLoans = Loan::with(['customer'])
                ->whereNotNull('disburseddate')
                ->orderBy('disburseddate', 'desc')
                ->take(5)
                ->get();

            // Loan performance stats
            $stats['overdue_loans_count'] = $this->getOverdueLoansCount();
            $stats['collection_rate'] = $this->calculateCollectionRate();
        }

        // Collections Module Statistics
        if ($user->hasAnyPermission(['collections.view', 'poweruser'])) {
            $stats['total_collections'] = Repayment::count();
            $stats['today_collections'] = Repayment::whereDate('rdate', today())->count();
            $stats['month_collections'] = Repayment::whereMonth('rdate', now()->month)
                ->whereYear('rdate', now()->year)
                ->count();
            $stats['total_collection_amount'] = Repayment::sum('rtotal');
        }

        // Customer Statistics (visible to all authenticated users with any permission)
        if ($user->permissions->count() > 0) {
            $stats['total_customers'] = Customer::count();
            $stats['active_customers'] = Customer::where('status', 'Active')->count();
            $stats['new_customers_this_month'] = Customer::whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->count();
        }

        // System-wide statistics (only for power users)
        if ($user->hasPermission('poweruser')) {
            $stats['system_health'] = $this->calculateSystemHealth();
            $stats['portfolio_at_risk'] = $this->calculatePortfolioAtRisk();
        }

        return view('admin.dashboard', compact('stats', 'recentApplications', 'recentLoans', 'applicationProgress', 'user'));
    }

    private function calculateTotalOutstanding()
    {
        // Simplified calculation - in real scenario, you'd calculate based on repayments
        $activeLoans = Loan::where('status', 'active')->get();
        $totalOutstanding = 0;
        
        foreach ($activeLoans as $loan) {
            $totalRepaid = Repayment::where('rloanid', $loan->loanid)->sum('rprincipal');
            $outstanding = $loan->amount - $totalRepaid;
            $totalOutstanding += max(0, $outstanding);
        }
        
        return $totalOutstanding;
    }

    private function calculateTotalCollected()
    {
        return Repayment::sum('rtotal');
    }

    private function getOverdueLoansCount()
    {
        // Simplified - you'd need to implement your overdue logic based on your business rules
        return Loan::where('status', 'active')
            ->where('maturitydate', '<', now())
            ->count();
    }

    private function calculateCollectionRate()
    {
        $totalExpected = Loan::where('status', 'active')->sum('amount');
        $totalCollected = $this->calculateTotalCollected();
        
        if ($totalExpected > 0) {
            return min(100, ($totalCollected / $totalExpected) * 100);
        }
        
        return 0;
    }

    private function getApplicationProgress()
    {
        $applications = Application::with(['customer', 'product'])
            ->whereIn('status', ['submitted', 'under_review', 'approved'])
            ->orderBy('created_at', 'desc')
            ->take(3)
            ->get();

        $progressData = collect();

        foreach ($applications as $application) {
            $progress = $this->calculateApplicationProgress($application);
            $progressData->put($application->id, [
                'application' => $application,
                'progress' => $progress,
                'status_text' => ucfirst(str_replace('_', ' ', $application->status)),
                'status_color' => $this->getStatusColor($application->status)
            ]);
        }

        return $progressData;
    }

    private function calculateApplicationProgress($application)
    {
        $statusProgress = [
            'draft' => 25,
            'submitted' => 50,
            'under_review' => 75,
            'approved' => 100,
            'rejected' => 100,
            'disbursed' => 100
        ];

        return $statusProgress[$application->status] ?? 0;
    }

    private function getStatusColor($status)
    {
        $colors = [
            'draft' => 'gray',
            'submitted' => 'blue',
            'under_review' => 'yellow',
            'approved' => 'green',
            'rejected' => 'red',
            'disbursed' => 'purple'
        ];

        return $colors[$status] ?? 'gray';
    }

    private function calculateSystemHealth()
    {
        // Simplified system health calculation
        $totalLoans = Loan::count();
        $nonPerformingLoans = Loan::where('status', 'default')->count();
        
        if ($totalLoans > 0) {
            $health = (($totalLoans - $nonPerformingLoans) / $totalLoans) * 100;
            return min(100, max(0, $health));
        }
        
        return 100;
    }

    private function calculatePortfolioAtRisk()
    {
        // Simplified PAR calculation
        $totalPortfolio = Loan::sum('amount');
        $nonPerformingPortfolio = Loan::where('status', 'default')->sum('amount');
        
        if ($totalPortfolio > 0) {
            return ($nonPerformingPortfolio / $totalPortfolio) * 100;
        }
        
        return 0;
    }
}