<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\CustomerMessage;
use App\Services\SmsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MessageController extends Controller
{
    protected $smsService;

    public function __construct(SmsService $smsService)
    {
        $this->smsService = $smsService;
    }

    /**
     * Store the message and send via SMS Service.
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customer,id',
            'category' => 'required|string|in:Promotions,Follow Up,Payment Reminder,Document Requests,Others',
            'message' => 'required|string|max:480',
        ]);

        try {
            $customer = Customer::findOrFail($request->customer_id);

            if (!$customer->phone) {
                return response()->json([
                    'success' => false, 
                    'message' => 'This customer does not have a registered phone number.'
                ], 422);
            }

            // 1. Attempt to send SMS via Zamtel API
            // Note: sendSms returns true/false based on your service logic
            $sent = $this->smsService->sendSms($customer->phone, $request->message);

            // 2. Log the transaction to your customer_messages table
            $messageLog = CustomerMessage::create([
                'customer_id' => $customer->id,
                'category'    => $request->category,
                'message'     => $request->message,
                'phone_number'=> $customer->phone,
                'status'      => $sent ? 'sent' : 'failed',
                'user_id'     => auth()->id(),
            ]);

            if ($sent) {
                return response()->json([
                    'success' => true,
                    'message' => 'SMS sent successfully!',
                    'data'    => $messageLog
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'The SMS provider rejected the request. Please check your balance or logs.',
            ], 500);

        } catch (\Exception $e) {
            Log::error("SMS Store Error: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'A server error occurred while sending the message.'
            ], 500);
        }
    }

    /**
     * Fetch message history for a specific customer (AJAX).
     */
    public function history($customerId)
    {
        try {
            // Verify customer existence
            $customer = Customer::find($customerId);
            if (!$customer) {
                return response()->json(['error' => 'Customer not found'], 404);
            }

            // Fetch messages ordered from oldest to newest for chat flow
            $messages = CustomerMessage::where('customer_id', $customerId)
                ->orderBy('created_at', 'asc')
                ->get();

            // Format the data for the JavaScript Chat UI
            $formattedMessages = $messages->map(function ($msg) {
                return [
                    'id'          => $msg->id,
                    'message'     => $msg->message,
                    'category'    => $msg->category,
                    'status'      => $msg->status, // sent or failed
                    'time'        => $msg->created_at ? $msg->created_at->format('h:i A') : 'N/A',
                    'date'        => $msg->created_at ? $msg->created_at->format('d M, Y') : '',
                    'is_outbound' => true 
                ];
            });

            return response()->json($formattedMessages);

        } catch (\Exception $e) {
            // Log the actual error to storage/logs/laravel.log to debug the 500 error
            Log::error("Chat History 500 Error: " . $e->getMessage(), [
                'customer_id' => $customerId,
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Internal Server Error. Check Laravel logs.'
            ], 500);
        }
    }
}