<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\LoanService;
use App\Models\LoanType; 
use App\Models\LoanQuotation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use PDF; // Facade for elibyy/tcpdf-laravel

class QuotationController extends Controller
{
    protected $loanService;

    public function __construct(LoanService $loanService)
    {
        $this->loanService = $loanService;
    }
    
    /**
     * Display a listing of the quotations.
     */
    public function index()
    {
        $quotations = LoanQuotation::orderBy('created_at', 'desc')->paginate(10);
        return view('admin.quotations.index', compact('quotations'));
    }

    /**
     * Display the quotation builder form.
     */
    public function create()
    {
        $loanProducts = LoanType::where('status', 'active')->get();
        return view('admin.quotations.create', compact('loanProducts'));
    }
    
    /**
     * Show the form for editing the specified quotation.
     */
    public function edit($id)
    {
        $quotation = LoanQuotation::findOrFail($id);
        $loanProducts = LoanType::where('status', 'active')->get();
    
        return view('admin.quotations.create', compact('loanProducts', 'quotation'));
    }

    /**
     * Logic to Save, Update, and Calculate the Amortization Schedule.
     * Updated to handle 'draft', 'active', and 'complete' statuses.
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_name'             => 'required|string|max:255',
            'principal_amount'          => 'required|numeric|min:0',
            'tenure'                    => 'required|integer|min:1',
            'interest_rate'             => 'required|numeric',
            'effective_processing_rate' => 'required|numeric',
            'status'                    => 'required|in:draft,active,complete', // UPDATED: Included 'complete'
            'tenure_type'               => 'required|in:daily,weekly,monthly',
            'interest_method'           => 'required|string'
        ]);

        try {
            return DB::transaction(function () use ($request) {
                
                $upfront = [
                    'admin_fee'     => (float) (($request->admin_rate / 100) * $request->principal_amount),
                    'insurance_fee' => (float) (($request->insurance_rate / 100) * $request->principal_amount),
                ];

                $calcData = $this->loanService->generateSchedule(
                    (float) $request->principal_amount,
                    (float) $request->interest_rate,
                    (float) $request->effective_processing_rate,
                    (int) $request->tenure,
                    $request->processing_basis ?? 'outstanding_balance',
                    $upfront
                );

                $quotation = LoanQuotation::updateOrCreate(
                    ['id' => $request->id], 
                    [
                        'quotation_number'          => $request->id ? $request->quotation_number : 'QUO-' . strtoupper(uniqid()),
                        'customer_name'             => $request->customer_name,
                        'product_id'                => $request->product_id,
                        'principal_amount'          => $request->principal_amount,
                        'tenure'                    => $request->tenure,
                        'tenure_type'               => $request->tenure_type,
                        'interest_method'           => $request->interest_method,
                        'base_interest_rate'        => $request->base_interest_rate,
                        'interest_discount'         => $request->interest_discount,
                        'effective_interest_rate'   => $request->interest_rate,
                        'base_processing_rate'      => $request->base_processing_rate,
                        'processing_discount'       => $request->processing_discount,
                        'effective_processing_rate' => $request->effective_processing_rate,
                        'admin_rate'                => $request->admin_rate,
                        'insurance_rate'            => $request->insurance_rate,
                        'monthly_installment'       => $calcData['summary']['periodic_installment'],
                        'total_interest'            => $calcData['summary']['total_interest'],
                        'total_repayable'           => $calcData['summary']['total_repayable'],
                        'net_disbursement'          => $calcData['summary']['net_disbursement'],
                        'status'                    => $request->status,
                        'created_by'                => Auth::id() ?? 1,
                    ]
                );

                return response()->json([
                    'success'      => true,
                    'quotation_id' => $quotation->id,
                    'status'       => $quotation->status,
                    'message'      => 'Quotation saved successfully'
                ]);
            });

        } catch (\Exception $e) {
            \Log::error("Quotation Error: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the finalized quotation with dynamic month-end amortization dates.
     */
    public function show($id)
    {
        $quotation = LoanQuotation::findOrFail($id);
        $prepared = $this->prepareScheduleData($quotation);

        return view('admin.quotations.show', [
            'quotation'    => $quotation,
            'scheduleData' => $prepared['schedule'],
            'summary'      => $prepared['summary']
        ]);
    }

    /**
     * Generate and download the PDF Amortization Schedule using TCPDF.
     */
    public function downloadPdf($id)
    {
        $quotation = LoanQuotation::findOrFail($id);
        $prepared = $this->prepareScheduleData($quotation);

        // Configure TCPDF via Facade
        PDF::SetTitle('Loan Quotation - ' . $quotation->quotation_number);
        PDF::SetMargins(15, 20, 15);
        PDF::SetFont('helvetica', '', 10);
        PDF::AddPage();

        // Render the PDF template blade to HTML
        $html = view('admin.quotations.pdf_template', [
            'quotation'    => $quotation,
            'scheduleData' => $prepared['schedule'],
            'summary'      => $prepared['summary']
        ])->render();

        PDF::writeHTML($html, true, false, true, false, '');

        return PDF::Output($quotation->quotation_number . '.pdf', 'D');
    }

    /**
     * Private helper to synchronize math and Month-End date logic.
     */
    private function prepareScheduleData($quotation)
    {
        $upfront = [
            'admin_fee' => (float) (($quotation->admin_rate / 100) * $quotation->principal_amount),
            'insurance_fee' => (float) (($quotation->insurance_rate / 100) * $quotation->principal_amount),
        ];

        $calcData = $this->loanService->generateSchedule(
            (float) $quotation->principal_amount,
            (float) $quotation->effective_interest_rate,
            (float) $quotation->effective_processing_rate,
            (int) $quotation->tenure,
            'outstanding_balance',
            $upfront
        );

        $scheduleData = $calcData['schedule'];
        $startDate = Carbon::now();

        foreach ($scheduleData as $index => &$row) {
            $nextDate = clone $startDate;
            
            if ($quotation->tenure_type === 'monthly') {
                // FIXED: Force to last day of month to handle varying month lengths (28, 30, 31)
                $row['date'] = $nextDate->addMonths($index + 1)->endOfMonth()->format('d M, Y');
            } elseif ($quotation->tenure_type === 'weekly') {
                $row['date'] = $nextDate->addWeeks($index + 1)->format('d M, Y');
            } else {
                $row['date'] = $nextDate->addDays($index + 1)->format('d M, Y');
            }
        }

        return [
            'schedule' => $scheduleData,
            'summary'  => $calcData['summary']
        ];
    }

    /**
     * Standalone calculation endpoint for AJAX previews.
     */
    public function calculateSchedule(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'interest_rate' => 'required|numeric',
            'processing_rate' => 'required|numeric',
            'processing_basis' => 'required|in:initial_amount,outstanding_balance',
            'tenure' => 'required|integer|min:1',
        ]);

        $upfront = [
            'admin_fee' => $request->admin_fee ?? 0,
            'insurance_fee' => $request->insurance_fee ?? 0,
        ];

        $data = $this->loanService->generateSchedule(
            (float) $request->amount,
            (float) $request->interest_rate,
            (float) $request->processing_rate,
            (int) $request->tenure,
            $request->processing_basis,
            $upfront
        );

        return response()->json($data);
    }
    
    public function downloadSchedulePdf(Request $request)
{
    $monthlyRatePercent = $request->get('rate', 10);
    $monthlyRate = $monthlyRatePercent / 100;
    
    $amounts = [2000, 3000, 5000, 8000, 10000, 12000, 15000, 18000, 20000, 25000, 30000, 40000, 50000, 60000, 80000, 100000];
    $tenures = [2, 3, 6, 12, 18, 24, 30, 36, 48, 60];

    // Configure PDF for Landscape to fit columns
    PDF::SetPageOrientation('L'); 
    PDF::SetTitle('Loan Schedule Matrix - ' . $monthlyRatePercent . '%');
    PDF::SetMargins(10, 10, 10);
    PDF::SetFont('helvetica', '', 8);
    PDF::AddPage();

    // Custom HTML for the wide table
    $html = view('admin.quotations.schedule_pdf_table', compact('amounts', 'tenures', 'monthlyRate', 'monthlyRatePercent'))->render();

    PDF::writeHTML($html, true, false, true, false, '');
    return PDF::Output('Loan_Schedule_Matrix_'.$monthlyRatePercent.'pc.pdf', 'D');
}
}