<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Branch;
use App\Models\Permission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use App\Mail\UserCreatedMail;
use Illuminate\Validation\Rules;

class UserController extends Controller
{
    public function index()
    {
        $users = User::with(['branch', 'permissions', 'creator'])
            ->orderBy('created_at', 'desc')
            ->get();

        $branches = Branch::active()->get();
        $permissions = Permission::all();

        return view('admin.users.index', compact('users', 'branches', 'permissions'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'firstname' => 'required|string|max:255',
            'middlename' => 'nullable|string|max:255',
            'lastname' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'branchid' => 'required|exists:branches,id',
            // Password removed from validation since we'll generate it
        ]);

        // Generate a strong random password
        $generatedPassword = Str::password(12); // 12 characters with letters, numbers, and symbols

        $user = User::create([
            'firstname' => $request->firstname,
            'middlename' => $request->middlename,
            'lastname' => $request->lastname,
            'email' => $request->email,
            'password' => Hash::make($generatedPassword),
            'branchid' => $request->branchid,
            'created_by' => auth()->id(),
        ]);

        try {
            // Send email with generated password
            Mail::to($user->email)->send(new UserCreatedMail($user, $generatedPassword));
            
            $emailStatus = 'Password has been sent to the user via email.';
            $emailSuccess = true;
        } catch (\Exception $e) {
            \Log::error('Failed to send user creation email: ' . $e->getMessage());
            $emailStatus = 'User created but failed to send email: ' . $e->getMessage();
            $emailSuccess = false;
        }

        return response()->json([
            'success' => true,
            'message' => 'User created successfully. ' . $emailStatus,
            'user' => $user->load(['branch']),
            'email_sent' => $emailSuccess,
            'generated_password' => $emailSuccess ? null : $generatedPassword // Only return password if email failed
        ]);
    }

    public function show(User $user)
    {
        return response()->json([
            'user' => $user->load(['branch', 'permissions', 'creator'])
        ]);
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'firstname' => 'required|string|max:255',
            'middlename' => 'nullable|string|max:255',
            'lastname' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'branchid' => 'required|exists:branches,id',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id'
        ]);

        $user->update([
            'firstname' => $request->firstname,
            'middlename' => $request->middlename,
            'lastname' => $request->lastname,
            'email' => $request->email,
            'branchid' => $request->branchid,
        ]);

        // Sync permissions only during update
        if ($request->has('permissions')) {
            $user->permissions()->sync($request->permissions);
        } else {
            $user->permissions()->detach();
        }

        return response()->json([
            'success' => true,
            'message' => 'User updated successfully',
            'user' => $user->load(['branch', 'permissions'])
        ]);
    }

    public function updatePassword(Request $request, User $user)
    {
        $request->validate([
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password updated successfully'
        ]);
    }

    public function toggleStatus(User $user)
    {
        $user->update([
            'status' => !$user->status
        ]);

        return response()->json([
            'success' => true,
            'message' => 'User status updated successfully',
            'user' => $user->fresh()
        ]);
    }

    public function destroy(User $user)
    {
        // Prevent users from deleting themselves
        if ($user->id === auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'You cannot delete your own account'
            ], 422);
        }

        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully'
        ]);
    }

    // New method to resend password email
    public function resendPassword(User $user)
    {
        // Generate a new strong random password
        $newPassword = Str::password(12);
        
        $user->update([
            'password' => Hash::make($newPassword),
        ]);

        try {
            // Send email with new password
            Mail::to($user->email)->send(new UserCreatedMail($user, $newPassword));
            
            return response()->json([
                'success' => true,
                'message' => 'New password has been generated and sent to the user via email.'
            ]);
        } catch (\Exception $e) {
            \Log::error('Failed to resend password email: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to send email: ' . $e->getMessage(),
                'generated_password' => $newPassword // Return password so admin can manually provide it
            ], 422);
        }
    }
}