<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\LipilaPaymentService;
use App\Models\PaymentDisbursement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\DB; // Added DB facade
use Illuminate\Http\JsonResponse;

class PaymentApiController extends Controller
{
    public function __construct(protected LipilaPaymentService $paymentService) {}

    /**
     * RESTORED: Collect Method (Repayments)
     */
          public function collect(Request $request): JsonResponse
{
    // Log for debugging
    Log::info('Incoming Collection Request Data:', $request->all());

    // 1. Validate based on Android incoming fields
    $validated = $request->validate([
        'amount' => 'required|numeric|min:1',
        'account_number' => 'required|string',
        'paymentmethod' => 'required|string|in:mtn,airtel,zamtel',
        'loan_number' => 'required|string',
        'customer_id' => 'required|integer',
        'company_id' => 'required|integer',
        'user_id' => 'required|integer',
        'narration' => 'nullable|string|max:255',
    ]);

    // 2. Lookup the Loan in disbursedloans table
    $loan = DB::table('disbursedloans')
        ->where('loannumber', $validated['loan_number'])
        ->first();

    if (!$loan) {
        return response()->json(['success' => false, 'message' => 'Loan record not found.'], 404);
    }

    try {
        $webhookUrl = URL::to('/api/webhooks/lipila/collection');
        
        $lipilaData = [
            'amount' => $validated['amount'],
            'account_number' => $validated['account_number'],
            'paymentmethod' => $validated['paymentmethod'],
            'narration' => $validated['narration'] ?? "Loan Repayment: " . $validated['loan_number'],
            'currency' => 'ZMW',
            'callback_url' => $webhookUrl,
        ];

        $response = $this->paymentService->collect($lipilaData, $validated['company_id'], $validated['user_id']);

        if ($response && $response->successful()) {
            
            $responseData = $response->json();
            // --- CRITICAL LOGGING HERE ---
            Log::info('Lipila Gateway Raw Response:', $responseData);

            $finalResponse = [
                'success' => true,
                'message' => 'Collection initiated.',
                'referenceId' => $responseData['referenceId'] ?? null, // Check if gateway uses camelCase
                'loan_id' => (int)$loan->loanid, // The app will now receive '9' (or similar) instead of '0'
                'status' => $responseData['status'] ?? 'Pending'
            ];

            Log::info('Final JSON sent to Android:', $finalResponse);
            
            // 3. Insert into payment_collections with aligned fields
            // FIX: Use $responseData instead of undefined $data variable
            DB::table('payment_collections')->insert([
                'company_id'         => $validated['company_id'],
                'created_by'         => $validated['user_id'], // Maps to the user performing the action
                'loan_id'            => $loan->loanid,         // Internal ID from disbursedloans
                'loan_customer_id'   => $validated['customer_id'],
                'loan_company_id'    => $validated['company_id'],
                'loan_user_id'       => $validated['user_id'],
                'reference_id'       => $responseData['referenceId'] ?? null, // FIX: Changed from $data to $responseData
                'identifier'         => $responseData['identifier'] ?? ($responseData['referenceId'] ?? null), // FIX: Changed from $data to $responseData
                'amount'             => $validated['amount'], // FIX: Use validated amount, or $responseData['amount'] if gateway returns it
                'currency'           => 'ZMW',
                'account_number'     => $validated['account_number'], // FIX: Use validated account_number, or $responseData['accountNumber'] if gateway returns it
                'status'             => $responseData['status'] ?? 'Pending', // FIX: Changed from $data to $responseData
                'payment_type'       => $validated['paymentmethod'],
                'narration'          => $lipilaData['narration'],
                'ip_address'         => $request->ip(),
                'api_created_at'     => now(),
                'webhook_url'        => $webhookUrl,
                'created_at'         => now(),
                'updated_at'         => now(),
            ]);
            
            return response()->json($finalResponse, 201);

           
        }
        
        Log::error('Gateway Error response:', [$response->body()]);
        return response()->json(['success' => false, 'message' => 'Gateway Error'], 400);
        
    } catch (\Exception $e) {
        Log::error('Collection Failed: ' . $e->getMessage());
        return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
    }
}
    /**
     * Step 1: Initiate Disbursement Gateway (Payouts)
     */
    public function disburse(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:1',
            'account_number' => 'required|string',
            'paymentmethod' => 'required|string|in:mtn,airtel,zamtel',
            'loan_id' => 'required|integer', 
            'customer_id' => 'required|integer|exists:customer,id',
            'company_id' => 'required|integer|exists:companies,id',
            'user_id' => 'required|integer|exists:users,id',
            'narration' => 'nullable|string|max:255',
        ]);

        try {
            $webhookUrl = URL::to('/api/webhooks/lipila/disbursement');
            $lipilaData = [
                'amount' => $validated['amount'],
                'account_number' => $validated['account_number'],
                'paymentmethod' => $validated['paymentmethod'],
                'narration' => $validated['narration'] ?? "Salary Advance Payout",
                'currency' => 'ZMW',
                'callback_url' => $webhookUrl,
            ];

            $response = $this->paymentService->disburse($lipilaData, $validated['company_id'], $validated['user_id']);

            if ($response && $response->successful()) {
                $data = $response->json();
                $disbursement = PaymentDisbursement::create([
                    'company_id'     => $validated['company_id'],
                    'customer_id'    => $validated['customer_id'],
                    'created_by'     => $validated['user_id'],
                    'reference_id'   => $data['referenceId'],
                    'identifier'     => $data['identifier'] ?? $data['referenceId'],
                    'amount'         => $data['amount'],
                    'account_number' => $data['accountNumber'],
                    'status'         => $data['status'] ?? 'Pending', 
                    'payment_type'   => $validated['paymentmethod'],
                    'loan_id'        => $validated['loan_id'] > 0 ? $validated['loan_id'] : null, 
                    'narration'      => $lipilaData['narration'],
                    'ip_address'     => $request->ip(),
                    'api_created_at' => now(),
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Payout initiated via gateway',
                    'data' => [
                        'reference_id' => $data['referenceId'],
                        'status' => $data['status']
                    ]
                ], 201);
            }
            return response()->json(['success' => false, 'message' => 'Gateway Error'], 400);
        } catch (\Exception $e) {
            Log::error('Disbursement Initiation Failed: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Polled by Android app to check status.
     * LOGGING UPDATED for deep debugging.
     */
    public function checkPaymentStatusPublic($referenceId): JsonResponse
    {
        Log::info("--- POLLING REQUEST --- Ref: $referenceId");

        $payment = PaymentDisbursement::where('reference_id', $referenceId)->first();

        // Fallback to collections if not in disbursements
        if (!$payment) {
            $payment = DB::table('payment_collections')->where('reference_id', $referenceId)->first();
        }

        if (!$payment) {
            Log::warning("POLL FAILED: Ref $referenceId not found in DB.");
            return response()->json(['success' => false, 'message' => 'Record not found'], 404);
        }

        // Logic check: What are we actually sending to Android?
        $statusToSend = $payment->status;
        
        Log::info("POLL SUCCESS: Ref: $referenceId | DB Status: $statusToSend | Amount: $payment->amount");

        return response()->json([
            'success' => true,
            'data' => [
                'reference_id' => $payment->reference_id,
                'status' => $statusToSend, // This MUST match 'Successful' or 'SUCCESSFUL'
                'amount' => (float)$payment->amount
            ]
        ]);
    }
}