<?php

namespace App\Http\Controllers;


use Illuminate\Support\Facades\Mail; // You were missing this import
use Illuminate\Support\Str; // Needed for verifyLoanOtp random string
use App\Models\Application;
use App\Models\Customer;
use App\Models\DisbursedLoan;
use App\Models\LoanType;
use App\Models\PaymentSchedule;
use App\Models\PaymentDisbursement;
use App\Models\CashBankAccount;
use App\Models\CashBook;
use App\Services\SmsService;
use App\Services\CreditScoreService;
use App\Services\LipilaPaymentService;
use App\Services\PaymentScheduleService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log; // <--- ADD THIS LINE
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use App\Mail\LoanApplicationSubmitted;
use App\Mail\LoanApplicationApproved;
use App\Mail\LoanApplicationRejected;

class CustomerDashboardController extends Controller
{
    private $smsService;
    private $creditScoreService;
    private $lipilaPaymentService;
    private $paymentScheduleService;

    public function __construct(
        SmsService $smsService, 
        CreditScoreService $creditScoreService,
        LipilaPaymentService $lipilaPaymentService,
        PaymentScheduleService $paymentScheduleService
    ) {
        $this->smsService = $smsService;
        $this->creditScoreService = $creditScoreService;
        $this->lipilaPaymentService = $lipilaPaymentService;
        $this->paymentScheduleService = $paymentScheduleService;
    }

    private function getCustomer(Request $request): ?Customer
    {
        if ($user = $request->user()) {
            return $user;
        }
        $customerId = $request->query('customer_id') ?? $request->input('customer_id');
        if (!$customerId) return null;

        return Customer::where('id', $customerId)->where('status', 'Active')->first();
    }

    public function getDashboardData(Request $request): JsonResponse
    {
        $customer = $this->getCustomer($request);
        
        if (!$customer) {
            return response()->json(['success' => false, 'message' => 'Customer profile not found'], 404);
        }

        $creditScoreInfo = $this->getCreditScoreInfo($customer);
        $outstanding = $this->calculateCurrentOutstanding($customer->id);
        
        $loanOffer = null;
        
        // Removed the $outstanding <= 10.00 check to always provide offer details
        $score = DB::table('customer_credit_scores')
            ->where('customer_id', $customer->id)
            ->where('is_current', 1)
            ->value('credit_score') ?? 300;

        $tier = DB::table('credit_score_tiers')
            ->where('min_score', '<=', $score)
            ->where('max_score', '>=', $score)
            ->where('is_active', 1)
            ->first();

      // ... inside getDashboardData method ...

if ($tier) {
    $offer = DB::table('product_credit_score_tier')
        ->join('loantype', 'product_credit_score_tier.product_id', '=', 'loantype.id')
        ->where('product_credit_score_tier.product_id', 5) // Fast Loan Product ID
        ->where('product_credit_score_tier.credit_score_tier_id', $tier->id)
        ->where('product_credit_score_tier.is_active', 1)
        ->where('loantype.status', 'Active')
        ->select(
            'loantype.product as product_name',
            'loantype.id as product_id',
            'loantype.details',
            'loantype.interest_method',
            'product_credit_score_tier.max_amount',
            'product_credit_score_tier.min_amount',
            'product_credit_score_tier.daily_rate',
            'product_credit_score_tier.duration'
        )
        ->first();

    if ($offer) {
        // Fetch Requirements for the Fast Loan (Product 5)
        $requirements = DB::table('product_requirements')
            ->join('applicationrequirements', 'product_requirements.requirement_id', '=', 'applicationrequirements.id')
            ->where('product_requirements.product_id', 5)
            ->select(
                'applicationrequirements.id',
                'applicationrequirements.documentname as name',
                'applicationrequirements.description as desc',
                'product_requirements.is_mandatory'
            )
            ->get();

        $loanOffer = [
            'available' => true,
            'product_name' => $offer->product_name,
            'product_id' => (int)$offer->product_id,
            'max_amount' => (float)$offer->max_amount,
            'min_amount' => (float)$offer->min_amount,
            'interest_rate' => (float)$offer->daily_rate,
            'max_duration_days' => (int)$offer->duration,
            'rate_type' => 'daily',
            'tier_name' => $tier->name,
            'details' => $offer->details,
            'interest_method' => $offer->interest_method,
            // Include requirements directly here
            'requirements' => $requirements 
        ];
    }
}


        return response()->json([
            'success' => true,
            'data' => [
                'customer' => [
                    'id' => $customer->id,
                    'customer_number' => $customer->customer_number,
                    'name' => $customer->first_name . ' ' . $customer->surname,
                    'email' => $customer->email,
                    'phone' => $customer->phone,
                ],
                'loan_offer' => $loanOffer,
                'application_stats' => $this->getApplicationStatsData($customer),
                'loan_stats' => ['total_outstanding_balance' => (float)$outstanding],
                'active_applications' => $this->getActiveApplicationsData($customer),
                'active_loans' => $this->getActiveLoansData($customer),
                'upcoming_payments' => $this->getUpcomingPaymentsData($customer),
                'credit_score' => $creditScoreInfo,
            ]
        ]);
    }

    public function smallloanapplication(Request $request): JsonResponse
    {
        \Log::info("🏁 STEP 3: Loan Accounting Finalization Started", $request->all());

        $validator = Validator::make($request->all(), [
            'product_id'   => 'required|exists:loantype,id',
            'amount'       => 'required|numeric|min:1',
            'tenure'       => 'required|integer|min:1',
            'reference_id' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $payout = PaymentDisbursement::where('reference_id', $request->reference_id)->first();
        if (!$payout) {
            return response()->json(['success' => false, 'message' => 'Payment record not found'], 404);
        }

        $customer = Customer::find($payout->customer_id);
        if (!$customer || $customer->status !== 'Active') {
            return response()->json(['success' => false, 'message' => 'Active profile not found'], 401);
        }

        try {
            return DB::transaction(function () use ($request, $customer, $payout) {
                $productId = 5; // Enforce Fast Loan Product ID
                $loanProduct = LoanType::findOrFail($productId);
                $principal = floatval($request->amount);
                $tenureDays = intval($request->tenure);

                // Fetch Daily Rate based on Score Tier
                $score = DB::table('customer_credit_scores')
                    ->where('customer_id', $customer->id)
                    ->where('is_current', 1)
                    ->value('credit_score') ?? 300;

                $tierId = DB::table('credit_score_tiers')
                    ->where('min_score', '<=', $score)
                    ->where('max_score', '>=', $score)
                    ->value('id');

                $dailyRate = DB::table('product_credit_score_tier')
                    ->where('product_id', $productId)
                    ->where('credit_score_tier_id', $tierId)
                    ->where('is_active', 1)
                    ->value('daily_rate') ?? 2.0;

                $interestAmount = $principal * ($dailyRate / 100) * $tenureDays;
                $totalRepayment = $principal + $interestAmount;
                $maturityDate = now()->addDays($tenureDays);

                $application = Application::create([
                    'application_number' => 'S-' . time(),
                    'customer_id'        => $customer->id,
                    'product_id'         => $productId,
                    'loan_amount'        => $principal,
                    'loan_tenure'        => $tenureDays,
                    'interest_rate'      => $dailyRate,
                    'interest_method'    => 'simple_interest',
                    'status'             => 'disbursed',
                    'approved_at'        => now(),
                    'disbursed_at'       => now(),
                    'company_id'         => $customer->companyid ?? 1,
                    'branch_id'          => $customer->branchid ?? 1,
                    'source'             => 'Mobile App',
                ]);

                $disbursedLoan = DisbursedLoan::create([
                    'application_id'    => $application->id,
                    'loannumber'        => DisbursedLoan::generateLoanNumber($customer->companyid),
                    'customerid'        => $customer->id,
                    'loantypeid'        => $productId,
                    'amount'            => $principal,
                    'interestrate'      => $dailyRate,
                    'interest_method'   => 'simple_interest',
                    'loanterm'          => $tenureDays,
                    'paymentfrequency'  => 'one_off',
                    'installmentamount' => $totalRepayment,
                    'disburseddate'     => now(),
                    'firstpaymentdate'  => $maturityDate,
                    'maturitydate'      => $maturityDate,
                    'status'            => 'active',
                    'principalbalance'  => $principal,
                    'interestbalance'   =>0.00,
                    'totalbalance'      => $principal,
                    'disbursement_reference' => $payout->reference_id,
                    'companyid'         => $customer->companyid ?? 1,
                    'branchid'          => $customer->branchid ?? 1,
                    'payment_account_id'=> 8,
                    'disbursedby'       => 1,
                    'createdby'         => 1,
                    'updatedby'         => 1,
                    'interest_calculation_status' => 'pending'
                ]);

                $payout->update(['loan_id' => $disbursedLoan->loanid]);
                $this->postToFinancials($disbursedLoan, $principal, $totalRepayment);
                $this->paymentScheduleService->generatePaymentSchedule($disbursedLoan);

                return response()->json(['success' => true, 'data' => ['loan_number' => $disbursedLoan->loannumber]], 201);
            });
        } catch (\Exception $e) {
            \Log::error("❌ STEP 3 TRANSACTION FAILED: " . $e->getMessage());
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    private function postToFinancials($loan, $principal, $total)
    {
        $description = "Loan disbursement - {$loan->loannumber} for {$loan->customer->customer}";
        $entryDate = now()->format('Y-m-d');
        $receivableAccountId = 15; 
        $cashAccountId = 8;       

        try {
            DB::table('generalledgerentries')->insert([
                'entrydate'   => $entryDate,
                'description' => $description,
                'accountid'   => $receivableAccountId,
                'documentno'  => $loan->loannumber,
                'entrytype'   => 'debit',
                'amount'      => $total,
                'companyid'   => $loan->companyid,
                'branchid'    => $loan->branchid,
                'transtype'   => 'loan_disbursement_mobile',
                'loanid'      => $loan->loanid,
                'created_at'  => now()
            ]);

            DB::table('generalledgerentries')->insert([
                'entrydate'   => $entryDate,
                'description' => $description,
                'accountid'   => $cashAccountId,
                'documentno'  => $loan->loannumber,
                'entrytype'   => 'credit',
                'amount'      => $principal * -1,
                'companyid'   => $loan->companyid,
                'branchid'    => $loan->branchid,
                'transtype'   => 'loan_disbursement_mobile',
                'loanid'      => $loan->loanid,
                'created_at'  => now()
            ]);

            DB::table('chart_of_accounts')->where('id', $receivableAccountId)->increment('current_balance', $total);
            DB::table('chart_of_accounts')->where('id', $cashAccountId)->decrement('current_balance', $principal);

            $account = CashBankAccount::find($loan->payment_account_id);
            if ($account) {
                CashBook::create([
                    'companyid' => $loan->companyid,
                    'branchid' => $loan->branchid,
                    'account_id' => $loan->payment_account_id,
                    'transaction_date' => now(),
                    'transaction_type' => 'Loan Disbursement',
                    'reference_number' => $loan->loannumber,
                    'description' => $description,
                    'amount' => -$principal,
                    'balance_before' => $account->current_balance,
                    'balance_after' => $account->current_balance - $principal,
                    'status' => 'Posted',
                    'related_transaction_type' => 'Loan Disbursement',
                    'related_transaction_id' => $loan->loanid,
                    'created_by' => 1 
                ]);
                $account->decrement('current_balance', $principal);
            }
        } catch (\Exception $e) {
            \Log::error("❌ FINANCIAL POSTING FAILED: " . $e->getMessage());
        }
    }

    private function calculateCurrentOutstanding($customerId)
    {
        return DisbursedLoan::where('customerid', $customerId)
            ->whereIn('status', ['active', 'default', 'delinquent'])
            ->sum('totalbalance');
    }

    private function getCreditScoreInfo(Customer $customer): array
    {
        $score = DB::table('customer_credit_scores')
            ->where('customer_id', $customer->id)
            ->where('is_current', 1)
            ->first();

        return [
            'current_score' => $score->credit_score ?? 300,
            'score_band' => $score->score_band ?? 'Fair',
            'available_credit' => (float)($score->available_credit ?? 0),
            'credit_limit' => (float)($score->credit_limit ?? 0),
        ];
    }

    private function getApplicationStatsData(Customer $customer): array
    {
        return [
            'total' => $customer->applications()->count(),
            'pending' => $customer->applications()->whereIn('status', ['submitted', 'under_review'])->count()
        ];
    }

    private function getActiveApplicationsData(Customer $customer): array
    {
        return $customer->applications()->whereIn('status', ['submitted', 'under_review', 'approved'])
            ->with('product')->latest()->get()->map(fn($app) => [
            'id' => $app->id,
            'application_number' => $app->application_number,
            'loan_amount' => (float)$app->loan_amount,
            'status_text' => ucfirst($app->status),
            'product_name' => $app->product->product ?? 'N/A',
        ])->toArray();
    }

    private function getActiveLoansData(Customer $customer): array
    {
        return DisbursedLoan::where('customerid', $customer->id)
            ->whereIn('status', ['active', 'default', 'delinquent'])
            ->get()->map(fn($loan) => [
            'id' => $loan->loanid,
            'loan_number' => $loan->loannumber,
            'amount' => (float)$loan->amount,
            'total_balance' => (float)$loan->totalbalance,
            'status' => $loan->status
        ])->toArray();
    }

    private function getUpcomingPaymentsData(Customer $customer): array
    {
        return PaymentSchedule::query()
            ->join('disbursedloans', 'paymentschedule.loanid', '=', 'disbursedloans.loanid')
            ->join('loantype', 'disbursedloans.loantypeid', '=', 'loantype.id')
            ->where('disbursedloans.customerid', $customer->id)
            ->where('disbursedloans.status', 'active')
            ->whereIn('paymentschedule.status', ['scheduled', 'partial', 'overdue']) 
            ->orderBy('paymentschedule.paymentdate', 'asc')
            ->limit(5)
            ->select([
                'paymentschedule.loannumber',
                'paymentschedule.paymentdate as due_date',
                'paymentschedule.totalamount', 
                'paymentschedule.total_paid',  
                'loantype.product as product_name'
            ])
            ->get()
            ->map(fn($p) => [
                'loan_number' => $p->loannumber,
                'product_name' => $p->product_name,
                'due_date'    => $p->due_date,
                'amount'      => (float)($p->totalamount - $p->total_paid) 
            ])
            ->toArray();
    }
    
   public function getAvailableProducts(Request $request): JsonResponse
{
    // Fetch all active products EXCEPT Fast Loan (ID 5)
    $products = DB::table('loantype')
        ->where('status', 'Active')
        ->where('id', '!=', 5) 
        ->select(
            'id', 
            'product', 
            'details', 
            'monthlyrate', 
            'interest_method',
            'adminfee',              // Added
            'insurancefee',           // Added
            'processing_fee',         // Added
            'processing_fee_basis',   // Added
            'penalty_rate',           // Added
            'penalty_basis'           // Added
        )
        ->get();

    return response()->json([
        'success' => true,
        'data' => $products
    ]);
}
    /**
 * Stage 3: Fetch requirements linked to a specific product
 */
/**
 * Stage 3: Fetch requirements linked to a specific product
 * REMOVED: Customer check to prevent 401 Unauthorized errors
 */
public function getProductDetails(Request $request, $id): JsonResponse
{
    $product = DB::table('loantype')->where('id', $id)->first();

    if (!$product) {
        return response()->json(['success' => false, 'message' => 'Product not found'], 404);
    }

    $requirements = DB::table('product_requirements')
        ->join('applicationrequirements', 'product_requirements.requirement_id', '=', 'applicationrequirements.id')
        ->where('product_requirements.product_id', $id)
        ->select(
            'applicationrequirements.id',
            'applicationrequirements.documentname as name',
            'applicationrequirements.description as desc',
            'product_requirements.is_mandatory'
        )
        ->get();

    return response()->json([
        'success' => true,
        'data' => [
            'product' => [
                'id' => $product->id,
                'product' => $product->product,
                'details' => $product->details,
                'monthlyrate' => (float)$product->monthlyrate,
                'interest_method' => $product->interest_method,
                
                // --- New Financial Parameters ---
                'adminfee' => (float)($product->adminfee ?? 0),
                'insurancefee' => (float)($product->insurancefee ?? 0),
                'processing_fee' => (float)($product->processing_fee ?? 0),
                'processing_fee_basis' => $product->processing_fee_basis ?? 'initial_amount',
                'penalty_rate' => (float)($product->penalty_rate ?? 0),
                'penalty_basis' => $product->penalty_basis ?? 'fixed',
            ],
            'requirements' => $requirements
        ]
    ]);
}
/**
 * Stage 5: Check if the requested amount is within the customer's credit limit
 */
public function checkEligibility(Request $request): JsonResponse
{
    $customer = $this->getCustomer($request);
    if (!$customer) {
        return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
    }

    $productId = $request->input('product_id');
    $principal = (float)$request->input('amount');
    $months = (int)$request->input('months');
    $netSalary = (float)$request->input('net_salary');

    // 1. Fetch Product details from loantype table
    $product = DB::table('loantype')->where('id', $productId)->first();
    if (!$product) {
        return response()->json(['success' => false, 'message' => 'Invalid Product'], 400);
    }

    // Rates as decimals
    $monthlyRate = (float)$product->monthlyrate / 100;
    $pFeeRate = (float)($product->processing_fee ?? 0) / 100;

    // --- 2. INTEREST & PRINCIPAL CALCULATION ---
    // Switch based on 'interest_method' column
    if ($product->interest_method === 'reducing_balance') {
        // Reducing Balance Amortization Formula: [P * r * (1 + r)^n] / [(1 + r)^n - 1]
        if ($monthlyRate > 0 && $months > 0) {
            $powInt = pow(1 + $monthlyRate, $months);
            $monthlyInterestPrincipal = ($principal * $monthlyRate * $powInt) / ($powInt - 1);
        } else {
            $monthlyInterestPrincipal = $principal / ($months ?: 1);
        }
    } else {
        // Simple Interest Formula: (P + (P * r * n)) / n
        $totalInterest = $principal * $monthlyRate * $months;
        $monthlyInterestPrincipal = ($principal + $totalInterest) / ($months ?: 1);
    }

    // --- 3. PROCESSING FEE CALCULATION (Amortized Periodic Component) ---
    // Switch based on 'processing_fee_basis'
    if ($product->processing_fee_basis === 'outstanding_balance') {
        // Amortized like interest (reducing balance style)
        if ($pFeeRate > 0 && $months > 0) {
            $powFee = pow(1 + $pFeeRate, $months);
            // We subtract (principal/months) to isolate the fee portion of the payment if needed, 
            // but usually, it's an add-on. Here is the pure fee component:
            $monthlyProcessingFee = (($principal * $pFeeRate * $powFee) / ($powFee - 1)) - ($principal / $months);
        } else {
            $monthlyProcessingFee = 0;
        }
    } else {
        // 'initial_amount' - Flat fee split across months
        $monthlyProcessingFee = ($principal * $pFeeRate);
    }

    // Total Periodic Installment
    $totalMonthlyInstallment = $monthlyInterestPrincipal + $monthlyProcessingFee;

    // --- 4. UPFRONT DEDUCTIONS (Admin & Insurance) ---
    $adminFeeAmount = $principal * ((float)($product->adminfee ?? 0) / 100);
    $insuranceFeeAmount = $principal * ((float)($product->insurancefee ?? 0) / 100);
    $netDisbursement = $principal - ($adminFeeAmount + $insuranceFeeAmount);

    // --- 5. ELIGIBILITY CHECK (60% Ceiling) ---
    $criteria = DB::table('loaneligibilitycriteria')
        ->where('companyid', $customer->companyid ?? 1)
        ->first();
    
    $ceilingPercent = $criteria ? (int)$criteria->maximumceiling : 60;
    $maxAllowedInstallment = $netSalary * ($ceilingPercent / 100);

    // --- 6. RESPONSE ---
    if ($totalMonthlyInstallment > $maxAllowedInstallment) {
        return response()->json([
            'success' => true,
            'eligible' => false,
            'message' => "Declined: Monthly installment of ZMW " . number_format($totalMonthlyInstallment, 2) . 
                         " (Incl. Fees) exceeds " . $ceilingPercent . "% of your income."
        ]);
    }

    return response()->json([
        'success' => true,
        'eligible' => true,
        'message' => "Eligibility Verified! Repayment Type: " . ucwords(str_replace('_', ' ', $product->interest_method)),
        'installment_amount' => round($totalMonthlyInstallment, 2),
        'processing_fee' => round($monthlyProcessingFee, 2),
        'processing_fee_rate' => (float)$product->processing_fee,
        'admin_fee' => round($adminFeeAmount, 2),
        'admin_fee_rate' => (float)$product->adminfee,
        'insurance_fee' => round($insuranceFeeAmount, 2),
        'insurance_fee_rate' => (float)$product->insurancefee,
        'net_disbursement' => round($netDisbursement, 2)
    ]);
}

public function requestLoanOtp(Request $request): JsonResponse
{
    $customer = $this->getCustomer($request);
    if (!$customer) return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);

    // 1. Generate OTP
    $otp = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);

    // 2. Prepare the temporary application data from the request
    $tempAppData = [
        'customer_id' => $customer->id,
        'product_id'  => $request->product_id,
        'amount'      => $request->amount,
        'tenure'      => $request->tenure,
        'otp'         => $otp, // You can hash this for extra security
    ];

    // 3. Store in Cache for 10 minutes using a unique customer key
    $cacheKey = "loan_otp_{$customer->id}";
    Cache::put($cacheKey, $tempAppData, now()->addMinutes(10));

    // 4. Send SMS
    $message = "Your Immia Finance loan verification code is: {$otp}.";
    $this->smsService->sendSms($customer->phone, $message);

    return response()->json(['success' => true, 'message' => 'OTP sent successfully']);
}

/**
 * STEP 2: Verify OTP (Move from Cache to Database)
 */
/**
 * STEP 2: Verify OTP (Move from Cache to Database)
 * Optimized for Fast Loan (Product 5) Daily Rates and One-off Payments
 */
public function verifyLoanOtp(Request $request): JsonResponse
{
    \Log::info("🚀 [verifyLoanOtp] Submission started", [
        'has_data' => $request->has('data'),
        'has_signature' => $request->hasFile('signature')
    ]);

    // 1. Decode the multipart JSON data payload
    $data = json_decode($request->input('data'), true);
    
    if (!$data) {
        return response()->json(['success' => false, 'message' => 'Invalid data payload'], 400);
    }

    $customerId = $data['customerId'];
    $productId = (int)($data['productId'] ?? 0);
    $userInputOtp = $data['otp'] ?? '';
    $cacheKey = "loan_otp_{$customerId}";

    // 2. Validate OTP from Cache
    $cachedData = Cache::get($cacheKey);

    if (!$cachedData || $cachedData['otp'] !== $userInputOtp) {
        return response()->json(['success' => false, 'message' => 'Invalid or expired OTP'], 400);
    }

    DB::beginTransaction();
    try {
        $customer = Customer::findOrFail($customerId);

        // 3. SPECIAL HANDLING FOR PRODUCT 5 (Fast Loan)
        $interestRate = $data['interestRate'] ?? 0;
        $interestMethod = $data['interestMethod'] ?? 'reducing_balance';
        $paymentFrequency = 'monthly';

        if ($productId === 5) {
            // Get current Credit Score
            $score = DB::table('customer_credit_scores')
                ->where('customer_id', $customerId)
                ->where('is_current', 1)
                ->value('credit_score') ?? 300;

            // Get the Tier ID
            $tierId = DB::table('credit_score_tiers')
                ->where('min_score', '<=', $score)
                ->where('max_score', '>=', $score)
                ->value('id');

            // Force the Daily Rate from the Tiered table
            $interestRate = DB::table('product_credit_score_tier')
                ->where('product_id', 5)
                ->where('credit_score_tier_id', $tierId)
                ->where('is_active', 1)
                ->value('daily_rate') ?? 2.0;

            $interestMethod = 'simple_interest';
            $paymentFrequency = 'one_off';
        }

        // 4. Handle Digital Signature File
        $signaturePath = null;
        if ($request->hasFile('signature')) {
            $sigFile = $request->file('signature');
            $signaturePath = $sigFile->store("loan_signatures/cust_{$customerId}", 'public');
        }

        // 5. Create Application Record
        $application = Application::create([
            'application_number'   => 'APP-' . strtoupper(Str::random(8)),
            'application_date'     => now()->toDateString(),
            'customer_id'          => $customerId,
            'product_id'           => $productId,
            'loan_amount'          => $data['amount'],
            'loan_tenure'          => $data['tenure'],
            
            // Logic-driven parameters
            'interest_rate'        => $interestRate,
            'interest_method'      => $interestMethod,
            'payment_frequency'    => $paymentFrequency,

            // Financial Parameters from Payload
            'processing_fee'       => $data['processingFee'] ?? 0,
            'processing_fee_basis' => $data['processingFeeBasis'] ?? 'initial_amount',
            'penalty_rate'         => $data['penaltyRate'] ?? 0,
            'penalty_basis'        => $data['penaltyBasis'] ?? 'fixed',
            'applicationtype'      => ($productId === 5) ? 'Fast Loan' : ($data['applicationType'] ?? 'Normal'),

            // Financial Results
            'installment_amount'    => $data['installmentAmount'] ?? 0,
            'processing_fee_amount' => $data['processingFeeAmount'] ?? 0,
            'admin_fee_amount'      => $data['adminFeeAmount'] ?? 0,
            'insurance_fee_amount'  => $data['insurance_fee_amount'] ?? 0, // Note the underscore fix
            'net_disbursement'      => $data['netDisbursement'] ?? 0,

            // Employment Info
            'employer_name'             => $data['employerName'] ?? null,
            'job_title'                 => $data['jobTitle'] ?? null,
            'employee_number'           => $data['employeeNumber'] ?? null,
            'employment_type'           => strtolower($data['employmentType'] ?? 'permanent'),
            'employment_years'          => $data['employmentYears'] ?? null,
            'net_salary'                => $data['netSalary'] ?? 0,
            'employer_physical_address' => $data['employerPhysicalAddress'] ?? null,
            'employer_postal_address'   => $data['employerPostalAddress'] ?: null,
            'employer_town_province'    => ($data['employerTown'] ?? '') . ', ' . ($data['employerProvince'] ?? ''),
            'gross_salary'              => $data['grossSalary'] ?? null,
            'retirement_year'           => $data['retirementYear'] ?? null,

            // Next of Kin
            'kin_first_name'       => $data['kinFirstName'] ?? null,
            'kin_surname'          => $data['kinSurname'] ?? null,
            'kin_relationship'     => $data['kinRelationship'] ?? null,
            'kin_telephone'        => $data['kinTelephone'] ?? null,
            'kin_physical_address' => $data['kinPhysicalAddress'] ?? null,
            'kin_town'             => $data['kinTown'] ?? null,
            'kin_province'         => $data['kinProvince'] ?? null,

            // Payment / Disbursement Details
            'suggestedpaymentmethod' => $data['suggestedPaymentMethod'] ?? null,
            'bank_name'             => $data['bankName'] ?? null,
            'account_number'        => $data['accountNumber'] ?? null,
            'mobile_channel'        => $data['mobileChannel'] ?? null,
            'mobile_number'         => $data['mobileNumber'] ?: $customer->phone,
            
            // Signature & Metadata
            'signature_path'       => $signaturePath,
            'signed_at'            => $signaturePath ? now() : null,
            'status'               => 'submitted',
            'submitted_at'         => now(),
            'source'               => 'Mobile App',
            'branch_id'            => $customer->branchid ?? 1,
            'company_id'           => $customer->companyid ?? 1,
            'currency_id'          => 57 
        ]);

        // 6. Handle Requirement Document Uploads
        if ($request->hasFile('documents')) {
            $files = $request->file('documents');
            foreach ($files as $reqId => $file) {
                $path = $file->store("loan_documents/app_{$application->id}", 'public');
                DB::table('loanapplicationdocuments')->insert([
                    'application_id' => $application->id,
                    'requirement_id' => (int)$reqId,
                    'file_path'      => $path,
                    'file_name'      => $file->getClientOriginalName(),
                    'file_size'      => $file->getSize(),
                    'file_type'      => $file->getMimeType(),
                    'uploaded_by'    => $customerId,
                    'status'         => 'pending',
                    'uploaded_at'    => now(),
                    'created_at'     => now(),
                ]);
            }
        }

        Cache::forget($cacheKey);
        DB::commit();

        $this->sendRequestNotifications($application);

        return response()->json([
            'success' => true, 
            'message' => 'Application submitted successfully',
            'data' => [
                'app_id' => $application->id,
                'application_number' => $application->application_number,
                'monthly_installment' => (float)$application->installment_amount
            ]
        ], 201);

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error("🔥 [verifyLoanOtp] Critical Submission Failure: " . $e->getMessage());
        return response()->json(['success' => false, 'message' => 'Submission failed: ' . $e->getMessage()], 500);
    }
}

/**
 * Send application request notifications
 * This is the central dispatcher for the 'app_request' event.
 */
private function sendRequestNotifications($application)
{
    try {
        // 1. Ensure the customer relationship is loaded so we have the email/phone
        if (!$application->relationLoaded('customer')) {
            $application->load('customer');
        }
        
        $customer = $application->customer;
        if (!$customer) {
            Log::warning('Customer not found for application notification', [
                'application_id' => $application->id
            ]);
            return;
        }

        // 2. Define the event key (must match your notification_settings table)
        $eventKey = 'app_request';
        
        // 3. Trigger SMS (Internal method that checks settings and templates)
        $this->sendSmsNotification($eventKey, $customer, $application);
        
        // 4. Trigger Email (Internal method that checks settings and sends the Mailable)
        $this->sendEmailNotification($eventKey, $customer, $application);

        Log::info('Notification dispatcher finished for application: ' . $application->application_number);

    } catch (\Exception $e) {
        Log::error('❌ Exception in sendRequestNotifications dispatcher', [
            'application_id' => $application->id,
            'error' => $e->getMessage()
        ]);
    }
}
    
     /**
     * Send SMS notification with template
     */
    private function sendSmsNotification($eventKey, $customer, $application, $additionalVariables = [])
    {
        try {
            // Check if SMS is enabled
            $smsEnabled = $this->isNotificationEnabled($eventKey, 'sms');
            
            if (!$smsEnabled) {
                Log::info('SMS notification disabled', ['event_key' => $eventKey]);
                return false;
            }
            
            if (empty($customer->phone)) {
                Log::warning('Customer phone number not available', [
                    'customer_id' => $customer->id,
                    'event_key' => $eventKey
                ]);
                return false;
            }
            
            // Get SMS template from database
            $smsTemplate = $this->getSmsTemplate($eventKey);
            
            if (!$smsTemplate) {
                Log::warning('No SMS template found', ['event_key' => $eventKey]);
                return false;
            }
            
            // Prepare variables for template
            $variables = array_merge([
                'Name' => $customer->first_name,
                'AppNo' => $application->application_number,
                'Amount' => number_format($application->loan_amount, 2),
                'Product' => $application->product->product ?? 'Loan',
                'Date' => $application->submitted_at ? $application->submitted_at->format('d/m/Y') : date('d/m/Y')
            ], $additionalVariables);
            
            // Replace template variables
            $message = $this->replaceTemplateVariables($smsTemplate, $variables);
            
            Log::info('Attempting to send SMS', [
                'phone' => $customer->phone,
                'formatted_message' => $message,
                'event_key' => $eventKey
            ]);

            // Send SMS
            $smsSent = $this->smsService->sendSms($customer->phone, $message);
            
            if ($smsSent) {
                Log::info('✅ SMS sent successfully', [
                    'customer_id' => $customer->id,
                    'phone' => $customer->phone,
                    'event_key' => $eventKey
                ]);
            } else {
                Log::warning('❌ Failed to send SMS (SMS service returned false)', [
                    'customer_id' => $customer->id,
                    'phone' => $customer->phone,
                    'event_key' => $eventKey
                ]);
            }
            
            return $smsSent;
            
        } catch (\Exception $e) {
            Log::error('❌ Exception in sendSmsNotification', [
                'event_key' => $eventKey,
                'customer_id' => $customer->id ?? 'unknown',
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send email notification
     */
    private function sendEmailNotification($eventKey, $customer, $application, $additionalData = null)
    {
        try {
            // Check if email is enabled
            $emailEnabled = $this->isNotificationEnabled($eventKey, 'email');
            
            if (!$emailEnabled) {
                Log::info('Email notification disabled', ['event_key' => $eventKey]);
                return false;
            }
            
            if (empty($customer->email)) {
                Log::warning('Customer email not available', [
                    'customer_id' => $customer->id,
                    'event_key' => $eventKey
                ]);
                return false;
            }
            
            Log::info('Attempting to send email', [
                'email' => $customer->email,
                'event_key' => $eventKey
            ]);
            
            // Send email based on event type
            $emailSent = false;
            switch ($eventKey) {
                case 'app_request':
                    Mail::to($customer->email)->send(new LoanApplicationSubmitted($application));
                    $emailSent = true;
                    break;
                    
                case 'app_approve':
                    Mail::to($customer->email)->send(new LoanApplicationApproved($application));
                    $emailSent = true;
                    break;
                    
                case 'app_rejection':
                    Mail::to($customer->email)->send(new LoanApplicationRejected($application));
                    $emailSent = true;
                    break;
                    
                default:
                    Log::warning('No email template for event', ['event_key' => $eventKey]);
                    return false;
            }
            
            if ($emailSent) {
                Log::info('✅ Email sent successfully', [
                    'customer_id' => $customer->id,
                    'email' => $customer->email,
                    'event_key' => $eventKey
                ]);
            }
            
            return $emailSent;
            
        } catch (\Exception $e) {
            Log::error('❌ Exception in sendEmailNotification', [
                'event_key' => $eventKey,
                'customer_email' => $customer->email ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Check if a notification is enabled for a specific channel
     */
    private function isNotificationEnabled($eventKey, $channel)
    {
        try {
            $setting = DB::table('notification_settings')
                ->where('event_key', $eventKey)
                ->where('channel', $channel)
                ->first();
                
            return $setting && $setting->is_enabled == 1;
        } catch (\Exception $e) {
            Log::error('❌ Failed to check notification setting', [
                'event_key' => $eventKey,
                'channel' => $channel,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get SMS template for an event
     */
    private function getSmsTemplate($eventKey)
    {
        try {
            $event = DB::table('notification_events')
                ->where('event_key', $eventKey)
                ->first();
                
            return $event ? $event->sms_template : null;
        } catch (\Exception $e) {
            Log::error('❌ Failed to get SMS template', [
                'event_key' => $eventKey,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Replace variables in template
     */
    private function replaceTemplateVariables($template, $variables)
    {
        if (empty($template)) {
            return '';
        }
        
        foreach ($variables as $key => $value) {
            $template = str_replace("[$key]", $value, $template);
        }
        return $template;
    }
    
    public function getStatusByNumber($appNumber): JsonResponse
    {
    // Find the application by the string number instead of the auto-increment ID
    $application = Application::where('application_number', $appNumber)->first();

    if (!$application) {
        return response()->json([
            'success' => false,
            'message' => 'Application not found'
        ], 404);
    }

    return response()->json([
        'success' => true,
        'status' => $application->status,
        'application_number' => $application->application_number,
        'updated_at' => $application->updated_at->toDateTimeString()
    ]);
    }
     
    
}