<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\User;
use App\Models\Customer;
use App\Models\LoanType;
use App\Models\Currency;
use App\Models\DisbursedLoan;
use App\Models\LoanApplicationDocument;

class Application extends Model
{
    use HasFactory;

    protected $table = 'applications';

    /**
     * The attributes that are mass assignable.
     * Synchronized with SQL Schema and PDF Agreement requirements.
     */
    protected $fillable = [
        'application_number',
        'application_date',
        'customer_id',
        'product_id',
        'branch_id',
        'company_id',
        'currency_id',
        
        // --- Loan Parameters ---
        'loan_amount',
        'loan_tenure',
        'loan_purpose',
        'interest_rate',
        'interest_method',
        'applicationtype',
        'payment_frequency', // <--- ADDED: Now synchronized with Schema
        
        // --- Financial Breakdown (ZMW Amounts) ---
        'net_disbursement',
        'installment_amount',
        'processing_fee_amount',
        'admin_fee_amount',
        'insurance_fee_amount',
        
        // --- Rates and Bases ---
        'processing_fee',
        'processing_fee_basis',
        'penalty_rate',
        'penalty_basis',
        'adminfee',
        'insurancefee',

        // --- Employment Details ---
        'employer_name',
        'job_title',
        'employee_number',
        'employment_type',
        'employment_years',
        'retirement_year',
        'net_salary',
        'gross_salary',
        'contract_start_date', // <--- ADDED: Helpful for PDF Page 2
        'contract_expiry_date',
        'employer_physical_address',
        'employer_postal_address',
        'employer_town_province',

        // --- Next of Kin Details ---
        'kin_first_name',
        'kin_surname',
        'kin_relationship',
        'kin_telephone',
        'kin_email',
        'kin_physical_address',
        'kin_postal_address',
        'kin_town',
        'kin_province',
        'kin_country',

        // --- Disbursement / Payment Details ---
        'suggestedpaymentmethod',
        'bank_name',
        'bank_branch',
        'account_number',
        'account_holder',
        'swift_code',
        'mobile_channel',
        'mobile_number',

        // --- Digital Signature & Generated Contract ---
        'signature_path',
        'signed_at',
        'contract_file', 

        // --- Status & Tracking ---
        'status',
        'submitted_at',
        'reviewed_at',
        'approved_at',
        'disbursed_at',
        'rejection_reason',
        'admin_notes',
        'disbursement_notes',
        'source',
        'created_by', 
        'reviewed_by', 
        'disbursed_by',
        'loan_id',
        'remarks'
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'application_date' => 'date',
        'submitted_at' => 'datetime',
        'reviewed_at' => 'datetime',
        'approved_at' => 'datetime',
        'disbursed_at' => 'datetime',
        'signed_at' => 'datetime',
        'contract_start_date' => 'date', // <--- ADDED
        'contract_expiry_date' => 'date',
        
        'loan_amount' => 'decimal:2',
        'net_disbursement' => 'decimal:2',
        'installment_amount' => 'decimal:2',
        'processing_fee_amount' => 'decimal:2',
        'admin_fee_amount' => 'decimal:2',
        'insurance_fee_amount' => 'decimal:2',
        'net_salary' => 'decimal:2',
        'gross_salary' => 'decimal:2',
        'interest_rate' => 'decimal:2',
        'processing_fee' => 'decimal:2',
        'penalty_rate' => 'decimal:2',
        'adminfee' => 'decimal:2',
        'insurancefee' => 'decimal:2',
    ];

    // ================= RELATIONSHIPS =================
    
    public function customer() 
{ 
    return $this->belongsTo(Customer::class, 'customer_id')->withDefault([
        'first_name' => 'Unknown',
        'surname' => 'Customer',
        'customer_number' => 'N/A'
    ]); 
}
    
    public function product() 
    { 
        return $this->belongsTo(LoanType::class, 'product_id'); 
    }
    
    public function loan() 
    { 
        return $this->hasOne(DisbursedLoan::class, 'application_id', 'id'); 
    }
    
    public function currency() 
    { 
        return $this->belongsTo(Currency::class, 'currency_id'); 
    }
    
    public function documents(): HasMany
    {
        return $this->hasMany(LoanApplicationDocument::class, 'application_id');
    }

    public function createdBy() 
    { 
        return $this->belongsTo(User::class, 'created_by'); 
    }
    
    public function reviewedBy() 
    { 
        return $this->belongsTo(User::class, 'reviewed_by'); 
    }
    
    public function disbursedBy() 
    { 
        return $this->belongsTo(User::class, 'disbursed_by'); 
    }

    // ================= SCOPES =================

    public function scopeNotDisbursed($query)
    {
        return $query->whereNull('disbursed_at');
    }

    public function scopeApproved($query)
    {
        return $query->where('status', 'approved')->whereNotNull('approved_at');
    }

    public function scopeSubmitted($query)
    {
        return $query->where('status', 'submitted');
    }

    // ================= ADDED SCOPE =================
    
    public function scopeDisbursed($query)
    {
        return $query->where('status', 'disbursed')->whereNotNull('disbursed_at');
    }

    // ================= HELPER METHODS =================
    
    /**
     * Check if application is approved
     */
    public function isApproved(): bool
    {
        return $this->status === 'approved' && !is_null($this->approved_at);
    }
    
    /**
     * Check if application is disbursed
     */
    public function isDisbursed(): bool
    {
        return $this->status === 'disbursed' && !is_null($this->disbursed_at);
    }
    
    /**
     * Check if application is pending (submitted)
     */
    public function isPending(): bool
    {
        return $this->status === 'submitted';
    }
    
    /**
     * Check if application is rejected
     */
    public function isRejected(): bool
    {
        return $this->status === 'rejected';
    }
    
    /**
     * Check if application is draft
     */
    public function isDraft(): bool
    {
        return $this->status === 'draft';
    }
}