<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ChartOfAccount extends Model
{
    use HasFactory;

    protected $fillable = [
        'company_id',
        'branch_id',
        'account_type_id',
        'parent_id',
        'code',
        'name',
        'description',
        'account_type',
        'currency_id',
        'opening_balance',
        'current_balance',
        'is_system_account',
        'is_active',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'opening_balance' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'is_system_account' => 'boolean',
        'is_active' => 'boolean'
    ];

    // Relationships
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    public function accountType()
    {
        return $this->belongsTo(AccountType::class);
    }

    public function parent()
    {
        return $this->belongsTo(ChartOfAccount::class, 'parent_id');
    }

    public function children()
    {
        return $this->hasMany(ChartOfAccount::class, 'parent_id');
    }

    public function currency()
    {
        return $this->belongsTo(Currency::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function suppliers()
    {
        return $this->hasMany(Supplier::class, 'account_payable_id');
    }

    public function bankPaymentVouchers()
    {
        return $this->hasMany(PaymentVoucher::class, 'bank_account_id');
    }

    // NEW: General Ledger Entries relationship
    public function generalLedgerEntries()
    {
        return $this->hasMany(GeneralLedgerEntry::class, 'accountid');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeSystemAccounts($query)
    {
        return $query->where('is_system_account', true);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('account_type', $type);
    }

    // NEW: Helper methods for general ledger integration
    public function hasTransactions()
    {
        return $this->generalLedgerEntries()->count() > 0;
    }

    public function updateCurrentBalance()
    {
        $balance = GeneralLedgerEntry::calculateAccountBalance($this->id);
        $this->update(['current_balance' => $balance]);
        return $balance;
    }

    public function getFormattedBalanceAttribute()
    {
        return 'ZMW ' . number_format($this->current_balance, 2);
    }
}