<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CustomerCreditScore extends Model
{
    use HasFactory;

    protected $primaryKey = 'credit_score_id';
    protected $table = 'customer_credit_scores';

    protected $fillable = [
        'customer_id',
        'credit_score',
        'score_band',
        'previous_score',
        'score_change',
        'change_reason',
        'change_type',
        'days_delinquent',
        'total_loans',
        'active_loans',
        'closed_loans',
        'defaulted_loans',
        'total_borrowed',
        'current_outstanding',
        'total_repaid',
        'on_time_repayment_rate',
        'late_payments_count',
        'missed_payments_count',
        'credit_utilization_ratio',
        'available_credit',
        'credit_limit',
        'last_repayment_date',
        'last_repayment_amount',
        'last_loan_date',
        'last_credit_inquiry',
        'has_active_default',
        'has_recent_late_payment',
        'has_high_utilization',
        'has_multiple_active_loans',
        'score_date',
        'is_current',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'credit_score' => 'integer',
        'previous_score' => 'decimal:2',
        'score_change' => 'integer',
        'days_delinquent' => 'integer',
        'total_loans' => 'integer',
        'active_loans' => 'integer',
        'closed_loans' => 'integer',
        'defaulted_loans' => 'integer',
        'total_borrowed' => 'decimal:2',
        'current_outstanding' => 'decimal:2',
        'total_repaid' => 'decimal:2',
        'on_time_repayment_rate' => 'decimal:2',
        'late_payments_count' => 'integer',
        'missed_payments_count' => 'integer',
        'credit_utilization_ratio' => 'decimal:2',
        'available_credit' => 'decimal:2',
        'credit_limit' => 'decimal:2',
        'last_repayment_amount' => 'decimal:2',
        'has_active_default' => 'boolean',
        'has_recent_late_payment' => 'boolean',
        'has_high_utilization' => 'boolean',
        'has_multiple_active_loans' => 'boolean',
        'is_current' => 'boolean',
        'score_date' => 'date',
        'last_repayment_date' => 'date',
        'last_loan_date' => 'date',
        'last_credit_inquiry' => 'date',
    ];

    // Relationships
    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customer_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    // Scopes
    public function scopeCurrent($query)
    {
        return $query->where('is_current', true);
    }

    public function scopeByCustomer($query, $customerId)
    {
        return $query->where('customer_id', $customerId);
    }

    public function scopeByScoreBand($query, $band)
    {
        return $query->where('score_band', $band);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('score_date', [$startDate, $endDate]);
    }

    public function scopeWithScoreChange($query, $minChange = 0)
    {
        return $query->where('score_change', '>=', $minChange);
    }

    public function scopeWithNegativeChange($query, $maxChange = 0)
    {
        return $query->where('score_change', '<=', $maxChange);
    }

    public function scopeHasScore($query)
    {
        return $query->whereNotNull('credit_score');
    }

    public function scopeNoHistory($query)
    {
        return $query->whereNull('credit_score');
    }

    // Helper Methods
    public function getScoreBandClassAttribute()
    {
        $classes = [
            'Excellent' => 'bg-green-100 text-green-800',
            'Good' => 'bg-blue-100 text-blue-800',
            'Fair' => 'bg-yellow-100 text-yellow-800',
            'Poor' => 'bg-orange-100 text-orange-800',
            'Very Poor' => 'bg-red-100 text-red-800',
            'No History' => 'bg-gray-100 text-gray-800',
        ];

        return $classes[$this->score_band] ?? 'bg-gray-100 text-gray-800';
    }

    public function getFormattedScoreChangeAttribute()
    {
        if ($this->score_change > 0) {
            return "+{$this->score_change}";
        } elseif ($this->score_change < 0) {
            return "{$this->score_change}";
        }
        return "0";
    }

    public function getScoreChangeClassAttribute()
    {
        if ($this->score_change > 0) {
            return 'text-green-600';
        } elseif ($this->score_change < 0) {
            return 'text-red-600';
        }
        return 'text-gray-600';
    }

    public function getFormattedCreditUtilizationAttribute()
    {
        return number_format($this->credit_utilization_ratio, 1) . '%';
    }

    public function getFormattedRepaymentRateAttribute()
    {
        return number_format($this->on_time_repayment_rate, 1) . '%';
    }

    // Determine score band based on score
    public static function getScoreBand($score)
    {
        if ($score === null) return 'No History';
        if ($score >= 700) return 'Excellent';
        if ($score >= 600) return 'Good';
        if ($score >= 500) return 'Fair';
        if ($score >= 400) return 'Poor';
        return 'Very Poor';
    }

    // Calculate credit utilization ratio
    public static function calculateUtilizationRatio($outstanding, $creditLimit)
    {
        if ($creditLimit <= 0) return 0;
        return min(100, ($outstanding / $creditLimit) * 100);
    }

    // Mark previous scores as not current
    public function markPreviousScoresAsHistorical()
    {
        self::where('customer_id', $this->customer_id)
            ->where('credit_score_id', '!=', $this->credit_score_id)
            ->update(['is_current' => false]);
    }

    /**
     * Check if this record has a credit score
     */
    public function getHasScoreAttribute()
    {
        return $this->credit_score !== null;
    }

    /**
     * Check if this is a "No History" record
     */
    public function getIsNoHistoryAttribute()
    {
        return $this->credit_score === null;
    }
}