<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class EarningType extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'company_id',
        'name',
        'type',
        'code',
        'description',
        'is_taxable',
        'is_active',
        'sort_order',
        'created_by',
        'updated_by'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_taxable' => 'boolean',
        'is_active' => 'boolean',
        'sort_order' => 'integer'
    ];

    /**
     * Get the company that owns the earning type.
     */
    public function company()
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    /**
     * Get the employee earnings for this earning type.
     */
    public function employeeEarnings()
    {
        return $this->hasMany(EmployeeEarning::class, 'earning_type_id');
    }

    /**
     * Scope a query to only include active earning types.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include taxable earning types.
     */
    public function scopeTaxable($query)
    {
        return $query->where('is_taxable', true);
    }

    /**
     * Scope a query to only include non-taxable earning types.
     */
    public function scopeNonTaxable($query)
    {
        return $query->where('is_taxable', false);
    }

    /**
     * Scope a query to only include earning types for a specific company.
     */
    public function scopeForCompany($query, $companyId)
    {
        return $query->where('company_id', $companyId);
    }

    /**
     * Get the type label for display.
     */
    public function getTypeLabelAttribute()
    {
        $labels = [
            'basic' => 'Basic Salary',
            'gratuity' => 'Gratuity',
            'allowance' => 'Allowance',
            'bonus' => 'Bonus',
            'commission' => 'Commission',
            'overtime' => 'Overtime',
            'other' => 'Other'
        ];

        return $labels[$this->type] ?? ucfirst($this->type);
    }

    /**
     * Get the type color for UI display.
     */
    public function getTypeColorAttribute()
    {
        $colors = [
            'basic' => 'blue',
            'gratuity' => 'purple',
            'allowance' => 'green',
            'bonus' => 'yellow',
            'commission' => 'indigo',
            'overtime' => 'orange',
            'other' => 'gray'
        ];

        return $colors[$this->type] ?? 'gray';
    }

    /**
     * Get the taxable badge for display.
     */
    public function getTaxableBadgeAttribute()
    {
        if ($this->is_taxable) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                        <i class="fas fa-percentage mr-1 text-xs"></i> Taxable
                    </span>';
        }
        return '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                    <i class="fas fa-ban mr-1 text-xs"></i> Non-Taxable
                </span>';
    }

    /**
     * Get the status badge for display.
     */
    public function getStatusBadgeAttribute()
    {
        if ($this->is_active) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                        <i class="fas fa-check-circle mr-1 text-xs"></i> Active
                    </span>';
        }
        return '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                    <i class="fas fa-times-circle mr-1 text-xs"></i> Inactive
                </span>';
    }

    /**
     * Get the display name with code.
     */
    public function getDisplayNameAttribute()
    {
        return "{$this->name} ({$this->code})";
    }

    /**
     * Check if this is a basic salary type.
     */
    public function isBasicSalary()
    {
        return $this->type === 'basic';
    }

    /**
     * Check if this is an allowance type.
     */
    public function isAllowance()
    {
        return $this->type === 'allowance';
    }

    /**
     * Check if this is a bonus type.
     */
    public function isBonus()
    {
        return $this->type === 'bonus';
    }

    /**
     * Get active employee earnings count.
     */
    public function getActiveEmployeeEarningsCountAttribute()
    {
        return $this->employeeEarnings()
            ->where('is_active', true)
            ->whereHas('employee', function ($query) {
                $query->where('is_active', true);
            })
            ->count();
    }

    /**
     * Get the total monthly amount for this earning type across all employees.
     */
    public function getTotalMonthlyAmountAttribute()
    {
        $total = 0;
        
        foreach ($this->employeeEarnings()->with('employee')->get() as $earning) {
            if ($earning->is_active && $earning->employee->is_active && $earning->isCurrent()) {
                $total += $earning->getMonthlyAmount();
            }
        }
        
        return $total;
    }

    /**
     * Get the average monthly amount for this earning type.
     */
    public function getAverageMonthlyAmountAttribute()
    {
        $count = $this->active_employee_earnings_count;
        
        if ($count === 0) {
            return 0;
        }
        
        return $this->total_monthly_amount / $count;
    }

    /**
     * Get the form data for API/JSON responses.
     */
    public function getFormDataAttribute()
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'type' => $this->type,
            'type_label' => $this->type_label,
            'type_color' => $this->type_color,
            'code' => $this->code,
            'description' => $this->description,
            'is_taxable' => $this->is_taxable,
            'taxable_badge' => $this->taxable_badge,
            'is_active' => $this->is_active,
            'status_badge' => $this->status_badge,
            'sort_order' => $this->sort_order,
            'display_name' => $this->display_name,
            'created_at' => $this->created_at?->format('Y-m-d H:i:s'),
            'updated_at' => $this->updated_at?->format('Y-m-d H:i:s'),
            'employee_count' => $this->active_employee_earnings_count,
            'total_monthly_amount' => $this->total_monthly_amount,
            'average_monthly_amount' => $this->average_monthly_amount
        ];
    }

    /**
     * Prepare the model for an array/JSON output.
     */
    public function toArray()
    {
        $array = parent::toArray();
        
        // Append computed attributes
        $array['type_label'] = $this->type_label;
        $array['type_color'] = $this->type_color;
        $array['taxable_badge'] = $this->taxable_badge;
        $array['status_badge'] = $this->status_badge;
        $array['display_name'] = $this->display_name;
        $array['is_basic_salary'] = $this->isBasicSalary();
        $array['is_allowance'] = $this->isAllowance();
        $array['is_bonus'] = $this->isBonus();
        
        return $array;
    }

    /**
     * Boot method for model events.
     */
    protected static function boot()
    {
        parent::boot();

        // Auto-convert code to uppercase before saving
        static::saving(function ($model) {
            if ($model->code) {
                $model->code = strtoupper(trim($model->code));
            }
        });

        // Prevent deletion if there are employee earnings
        static::deleting(function ($model) {
            if ($model->employeeEarnings()->count() > 0) {
                throw new \Exception('Cannot delete earning type that has employee earnings assigned. Please deactivate it instead.');
            }
        });
    }
}