<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class ExpenseTransaction extends Model
{
    use HasFactory, SoftDeletes;

    protected $primaryKey = 'expense_transaction_id';
    
    protected $fillable = [
        'transaction_number',
        'transaction_date',
        'vendor_id',
        'description',
        'amount',
        'currency',
        'exchange_rate',
        'base_amount',
        'payment_method',
        'check_number',
        'reference_number',
        'account_id',
        'status',
        'notes',
        'company_id',
        'branch_id',
        'created_by',
        'posted_at',
        'reversed_at',
        'reversed_by',
        'reversal_reason'
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'amount' => 'decimal:2',
        'exchange_rate' => 'decimal:4',
        'base_amount' => 'decimal:2',
        'posted_at' => 'datetime',
        'reversed_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->transaction_number)) {
                $model->transaction_number = static::generateTransactionNumber();
            }
        });

        // Calculate total from items when saving
        static::saving(function ($model) {
            if ($model->items()->exists()) {
                $totalAmount = $model->items()->sum('amount');
                $model->amount = $totalAmount;
                $model->base_amount = $totalAmount * $model->exchange_rate;
            }
        });

        // Update vendor balance when transaction is posted
        static::updated(function ($model) {
            if ($model->isDirty('status') && $model->status === 'posted' && $model->vendor_id) {
                $vendor = $model->vendor;
                if ($vendor) {
                    $vendor->updateBalance($model->amount);
                    $vendor->last_transaction_date = $model->transaction_date;
                    $vendor->save();
                }
            }
        });
    }

    /**
     * Generate unique transaction number
     */
    public static function generateTransactionNumber(): string
    {
        $prefix = 'EXP';
        $year = date('Y');
        $month = date('m');
        
        $lastNumber = static::where('transaction_number', 'like', "{$prefix}{$year}{$month}%")
            ->orderBy('transaction_number', 'desc')
            ->value('transaction_number');
            
        $sequence = 1;
        if ($lastNumber) {
            $sequence = (int) substr($lastNumber, -4) + 1;
        }
        
        return "{$prefix}{$year}{$month}" . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Relationships
     */
    public function vendor()
    {
        return $this->belongsTo(Vendor::class, 'vendor_id');
    }

    public function account()
    {
        return $this->belongsTo(CashBankAccount::class, 'account_id', 'cashBankId');
    }

    public function items()
    {
        return $this->hasMany(ExpenseItem::class, 'expense_transaction_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function reverser()
    {
        return $this->belongsTo(User::class, 'reversed_by');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branch_id');
    }

    /**
     * Scopes
     */
    public function scopePosted($query)
    {
        return $query->where('status', 'posted');
    }

    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_date', [$startDate, $endDate]);
    }

    public function scopeForVendor($query, $vendorId)
    {
        return $query->where('vendor_id', $vendorId);
    }

    /**
     * Accessors
     */
    public function getFormattedAmountAttribute()
    {
        return number_format($this->amount, 2);
    }

    public function getFormattedBaseAmountAttribute()
    {
        return number_format($this->base_amount, 2);
    }

    public function getFormattedDateAttribute()
    {
        return $this->transaction_date->format('M d, Y');
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'draft' => [
                'label' => 'Draft',
                'color' => 'secondary',
                'class' => 'bg-gray-100 text-gray-800'
            ],
            'posted' => [
                'label' => 'Posted',
                'color' => 'success',
                'class' => 'bg-green-100 text-green-800'
            ],
            'cancelled' => [
                'label' => 'Cancelled',
                'color' => 'danger',
                'class' => 'bg-red-100 text-red-800'
            ],
            'reversed' => [
                'label' => 'Reversed',
                'color' => 'warning',
                'class' => 'bg-orange-100 text-orange-800'
            ],
        ];

        return $badges[$this->status] ?? $badges['draft'];
    }

    public function getPaymentMethodBadgeAttribute()
    {
        $badges = [
            'cash' => [
                'label' => 'Cash',
                'color' => 'success',
                'class' => 'bg-green-100 text-green-800'
            ],
            'check' => [
                'label' => 'Check',
                'color' => 'info',
                'class' => 'bg-blue-100 text-blue-800'
            ],
            'credit_card' => [
                'label' => 'Credit Card',
                'color' => 'warning',
                'class' => 'bg-yellow-100 text-yellow-800'
            ],
            'bank_transfer' => [
                'label' => 'Bank Transfer',
                'color' => 'primary',
                'class' => 'bg-indigo-100 text-indigo-800'
            ],
            'mobile_money' => [
                'label' => 'Mobile Money',
                'color' => 'purple',
                'class' => 'bg-purple-100 text-purple-800'
            ],
        ];

        return $badges[$this->payment_method] ?? ['label' => 'N/A', 'class' => 'bg-gray-100 text-gray-800'];
    }

    /**
     * Check if transaction can be edited
     */
    public function canEdit(): bool
    {
        return $this->status === 'draft';
    }

    /**
     * Check if transaction can be posted
     */
    public function canPost(): bool
    {
        return $this->status === 'draft';
    }

    /**
     * Check if transaction can be reversed
     */
    public function canReverse(): bool
    {
        return $this->status === 'posted';
    }

    /**
     * Post the transaction
     */
    public function post()
    {
        if (!$this->canPost()) {
            return false;
        }

        DB::beginTransaction();
        try {
            $this->status = 'posted';
            $this->posted_at = now();
            $this->save();

            // Update vendor balance
            if ($this->vendor_id) {
                $vendor = $this->vendor;
                $vendor->updateBalance($this->amount);
                $vendor->last_transaction_date = $this->transaction_date;
                $vendor->save();
            }

            // Record in cash book if payment method is cash, bank transfer, or mobile money
            if (in_array($this->payment_method, ['cash', 'bank_transfer', 'mobile_money']) && $this->account_id) {
                $this->recordInCashBook();
            }

            DB::commit();
            return true;

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Failed to post expense transaction: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Reverse the transaction
     */
    public function reverse($reason = null)
    {
        if (!$this->canReverse()) {
            return false;
        }

        DB::beginTransaction();
        try {
            $this->status = 'reversed';
            $this->reversed_at = now();
            $this->reversed_by = auth()->id();
            $this->reversal_reason = $reason;
            $this->save();

            // Reverse vendor balance
            if ($this->vendor_id) {
                $vendor = $this->vendor;
                $vendor->updateBalance(-$this->amount);
                $vendor->save();
            }

            // Reverse cash book entry
            if (in_array($this->payment_method, ['cash', 'bank_transfer', 'mobile_money']) && $this->account_id) {
                $this->reverseCashBookEntry();
            }

            DB::commit();
            return true;

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Failed to reverse expense transaction: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Record transaction in cash book
     */
    private function recordInCashBook()
    {
        try {
            $transactionType = 'Expense Payment';
            if ($this->payment_method === 'cash') {
                $transactionType = 'Cash Expense';
            } elseif ($this->payment_method === 'bank_transfer') {
                $transactionType = 'Bank Transfer - Expense';
            } elseif ($this->payment_method === 'mobile_money') {
                $transactionType = 'Mobile Money Expense';
            }

            CashBook::recordTransaction(
                $this->account_id,
                -$this->base_amount,
                'credit',
                $transactionType,
                "Expense payment: {$this->transaction_number} - {$this->description}",
                $this->transaction_number,
                $this->company_id,
                $this->branch_id,
                $this->currency
            );
        } catch (\Exception $e) {
            \Log::error('Failed to record expense in cash book: ' . $e->getMessage());
        }
    }

    /**
     * Reverse cash book entry
     */
    private function reverseCashBookEntry()
    {
        try {
            $transactionType = 'Expense Reversal';
            
            CashBook::recordTransaction(
                $this->account_id,
                $this->base_amount,
                'debit',
                $transactionType,
                "Expense reversal: {$this->transaction_number} - {$this->description}",
                $this->transaction_number . '_REV',
                $this->company_id,
                $this->branch_id,
                $this->currency
            );
        } catch (\Exception $e) {
            \Log::error('Failed to record expense reversal in cash book: ' . $e->getMessage());
        }
    }
}