<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class FinancialYear extends Model
{
    use HasFactory;

    protected $fillable = [
        'company_id',
        'branch_id',
        'year',
        'start_date',
        'end_date',
        'is_active',
        'is_closed',
        'closed_by',
        'closed_at',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'is_active' => 'boolean',
        'is_closed' => 'boolean',
        'closed_at' => 'datetime'
    ];

    // Relationships
    public function company()
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    public function closedBy()
    {
        return $this->belongsTo(User::class, 'closed_by');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function journalEntries()
    {
        return $this->hasMany(JournalEntry::class);
    }

    public function currentCompanies()
    {
        return $this->hasMany(Company::class, 'current_financial_year_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeClosed($query)
    {
        return $query->where('is_closed', true);
    }

    public function scopeOpen($query)
    {
        return $query->where('is_closed', false);
    }

    public function scopeForCompany($query, $companyId)
    {
        return $query->where('company_id', $companyId);
    }

    public function scopeAvailableForSelection($query, $companyId = null)
    {
        $query = $query->where('is_closed', false);
        
        if ($companyId) {
            $query = $query->where('company_id', $companyId);
        }
        
        return $query->orderBy('start_date', 'desc');
    }

    // Accessor for year_name (if you still want to use it in views)
    public function getYearNameAttribute()
    {
        return "Financial Year {$this->year}";
    }

    public function getFormattedDateRangeAttribute()
    {
        return $this->start_date->format('M Y') . ' - ' . $this->end_date->format('M Y');
    }

    // Method to check if financial year is current
    public function isCurrent()
    {
        return $this->start_date <= now() && $this->end_date >= now();
    }

    // Method to close financial year
    public function close($userId = null)
    {
        $this->update([
            'is_closed' => true,
            'closed_by' => $userId ?? auth()->id(),
            'closed_at' => now(),
            'updated_by' => auth()->id()
        ]);
    }

    // Method to reopen financial year
    public function reopen()
    {
        $this->update([
            'is_closed' => false,
            'closed_by' => null,
            'closed_at' => null,
            'updated_by' => auth()->id()
        ]);
    }
}