<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class GeneralLedgerEntry extends Model
{
    use HasFactory;

    protected $table = 'generalledgerentries';
    protected $primaryKey = 'entryid';
    public $incrementing = true;
    protected $keyType = 'int';

    protected $fillable = [
        'entrydate',
        'description',
        'accountid',
        'referencedocument',
        'reference_type',
        'reference_id',
        'documentno',
        'entrytype',
        'amount',
        'createdby',
        'companyid',
        'branchid',
        'transtype',
        'original_currency',
        'loanid'
    ];

    protected $casts = [
        'entrydate' => 'date',
        'amount' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relationships
    public function account()
    {
        return $this->belongsTo(ChartOfAccount::class, 'accountid');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'createdby');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'companyid');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branchid');
    }

    // Add relationship for internal transfers
    public function internalTransfer()
    {
        return $this->belongsTo(InternalCashBankTransfer::class, 'reference_id', 'transferid')
            ->where('reference_type', 'internal_transfer');
    }

    // Scopes
    public function scopeDebits($query)
    {
        return $query->where('entrytype', 'debit');
    }

    public function scopeCredits($query)
    {
        return $query->where('entrytype', 'credit');
    }

    public function scopeOpeningBalances($query)
    {
        return $query->where('referencedocument', 'OPENING_BALANCE');
    }

    public function scopeByAccount($query, $accountId)
    {
        return $query->where('accountid', $accountId);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('entrydate', [$startDate, $endDate]);
    }

    public function scopeByCompany($query, $companyId)
    {
        return $query->where('companyid', $companyId);
    }

    public function scopeByBranch($query, $branchId)
    {
        return $query->where('branchid', $branchId);
    }

    public function scopeByReference($query, $referenceType, $referenceId)
    {
        return $query->where('reference_type', $referenceType)
            ->where('reference_id', $referenceId);
    }

    // Helper method to determine entry type based on normal balance
    public static function determineEntryType($normalBalance, $amount)
    {
        // If normal_balance is 'debit', positive amounts are debits
        // If normal_balance is 'credit', positive amounts are credits
        if ($normalBalance === 'debit') {
            return $amount >= 0 ? 'debit' : 'credit';
        } elseif ($normalBalance === 'credit') {
            return $amount >= 0 ? 'credit' : 'debit';
        }
        
        // Default fallback - assume debit normal balance
        return $amount >= 0 ? 'debit' : 'credit';
    }

    // Helper method to create opening balance entry
    public static function createOpeningBalanceEntry(ChartOfAccount $account, $openingBalance, $description = null)
    {
        if ($openingBalance != 0) {
            // Get the normal balance from the account type
            $normalBalance = $account->accountType->normal_balance ?? 'debit';
            
            $entryType = self::determineEntryType($normalBalance, $openingBalance);

            return self::create([
                'entrydate' => now(),
                'description' => $description ?? "Opening balance for {$account->name}",
                'accountid' => $account->id,
                'referencedocument' => 'OPENING_BALANCE',
                'documentno' => 'Not Applicable',
                'entrytype' => $entryType,
                'amount' => abs($openingBalance), // Store absolute value
                'createdby' => auth()->id() ?? 1,
                'companyid' => $account->company_id ?? 1,
                'branchid' => $account->branch_id ?? 1,
            ]);
        }
        
        return null;
    }

    // Get the effective amount (positive for debits, negative for credits for balance calculation)
    public function getEffectiveAmountAttribute()
    {
        return $this->entrytype === 'debit' ? $this->amount : -$this->amount;
    }

    // Accessors for compatibility with existing code
    public function getTransactionDateAttribute()
    {
        return $this->entrydate;
    }

    public function getReferenceNumberAttribute()
    {
        return $this->documentno;
    }

    public function getDebitAttribute()
    {
        return $this->entrytype === 'debit' ? $this->amount : 0;
    }

    public function getCreditAttribute()
    {
        return $this->entrytype === 'credit' ? abs($this->amount) : 0;
    }

    // Check if this is a debit entry
    public function isDebit()
    {
        return $this->entrytype === 'debit';
    }

    // Check if this is a credit entry
    public function isCredit()
    {
        return $this->entrytype === 'credit';
    }

    // Get account balance based on normal balance
    public static function calculateAccountBalance($accountId, $asOfDate = null)
    {
        $query = self::where('accountid', $accountId);
        
        if ($asOfDate) {
            $query->where('entrydate', '<=', $asOfDate);
        }
        
        $debits = (clone $query)->debits()->sum('amount');
        $credits = (clone $query)->credits()->sum('amount');
        
        // Get the account and its normal balance
        $account = ChartOfAccount::with('accountType')->find($accountId);
        $normalBalance = $account->accountType->normal_balance ?? 'debit';
        
        if ($normalBalance === 'debit') {
            return $debits - $credits; // Debit normal balance accounts
        } else {
            return $credits - $debits; // Credit normal balance accounts
        }
    }
}