<?php
// app/Models/InterestIncome.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class InterestIncome extends Model
{
    use HasFactory;

    protected $table = 'interestincome';
    protected $primaryKey = 'id';

    protected $fillable = [
        'iloanid',
        'customerid',
        'ipid',
        'charge',
        'chargeddate',
        'poster',
        'branchid',
        'companyid'
    ];

    protected $casts = [
        'chargeddate' => 'date',
        'charge' => 'decimal:2',
        'branchid' => 'integer',
        'companyid' => 'integer',
    ];

    /**
     * Relationship with DisbursedLoan
     */
    public function loan()
    {
        return $this->belongsTo(DisbursedLoan::class, 'iloanid', 'loanid');
    }

    /**
     * Relationship with Customer
     */
    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customerid');
    }

    /**
     * Relationship with LoanType
     */
    public function loanType()
    {
        return $this->belongsTo(LoanType::class, 'ipid', 'id');
    }

    /**
     * Relationship with User who posted
     */
    public function postedBy()
    {
        return $this->belongsTo(User::class, 'poster');
    }

    /**
     * Relationship with Branch
     */
    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branchid', 'id');
    }

    /**
     * Relationship with Company
     */
    public function company()
    {
        return $this->belongsTo(Company::class, 'companyid', 'id');
    }

    /**
     * Scope for date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('chargeddate', [$startDate, $endDate]);
    }

    /**
     * Scope for specific loan
     */
    public function scopeForLoan($query, $loanId)
    {
        return $query->where('iloanid', $loanId);
    }

    /**
     * Scope for specific branch
     */
    public function scopeForBranch($query, $branchId)
    {
        return $query->where('branchid', $branchId);
    }

    /**
     * Scope for specific company
     */
    public function scopeForCompany($query, $companyId)
    {
        return $query->where('companyid', $companyId);
    }

    /**
     * Get formatted charge amount
     */
    public function getFormattedChargeAttribute()
    {
        return 'K' . number_format($this->charge, 2);
    }

    /**
     * Get formatted charged date
     */
    public function getFormattedChargedDateAttribute()
    {
        return $this->chargeddate->format('M j, Y');
    }
}