<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PayeTaxBand extends Model
{
    use HasFactory;

    protected $fillable = [
        'payroll_setting_id',
        'company_id',
        'band_number',
        'description',
        'monthly_from',
        'monthly_to',
        'annual_from',
        'annual_to',
        'tax_rate',
        'is_active',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'monthly_from' => 'decimal:2',
        'monthly_to' => 'decimal:2',
        'annual_from' => 'decimal:2',
        'annual_to' => 'decimal:2',
        'tax_rate' => 'decimal:2',
        'is_active' => 'boolean'
    ];

    // Relationships
    public function payrollSetting()
    {
        return $this->belongsTo(PayrollSetting::class, 'payroll_setting_id');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeForCompany($query, $companyId)
    {
        return $query->where('company_id', $companyId);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('band_number');
    }

    // Helper methods
    public function getMonthlyRangeAttribute()
    {
        if ($this->monthly_to == 999999999.99) {
            return 'K' . number_format($this->monthly_from, 2) . ' and above';
        }
        return 'K' . number_format($this->monthly_from, 2) . ' - K' . number_format($this->monthly_to, 2);
    }

    public function getAnnualRangeAttribute()
    {
        if ($this->annual_to == 999999999.99) {
            return 'K' . number_format($this->annual_from, 2) . ' and above';
        }
        return 'K' . number_format($this->annual_from, 2) . ' - K' . number_format($this->annual_to, 2);
    }

    public function getTaxRateFormattedAttribute()
    {
        return number_format($this->tax_rate, 2) . '%';
    }

    // Calculate tax for a given monthly salary
    public function calculateMonthlyTax($monthlySalary)
    {
        if ($monthlySalary >= $this->monthly_from && $monthlySalary <= $this->monthly_to) {
            $taxableAmount = $monthlySalary - $this->monthly_from;
            return ($taxableAmount * $this->tax_rate) / 100;
        }
        return 0;
    }
}