<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PaymentCollection extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'company_id',
        'created_by',
        'loan_id',
        'loan_customer_id',
        'loan_company_id',
        'loan_user_id',
        'application_id',
        'reference_id',
        'identifier',
        'amount',
        'currency',
        'account_number',
        'status',
        'payment_type',
        'message',
        'narration',
        'ip_address',
        'api_created_at',
        'webhook_received_at',
        'webhook_payload',
        'webhook_url',
        'metadata',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'amount' => 'decimal:4',
        'api_created_at' => 'datetime',
        'webhook_received_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'metadata' => 'array',
        'webhook_payload' => 'array',
    ];

    /**
     * Relationship to the Company
     */
    public function company()
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    /**
     * Relationship to the loan company
     */
    public function loanCompany()
    {
        return $this->belongsTo(Company::class, 'loan_company_id');
    }

    /**
     * Relationship to the User who initiated the collection
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Relationship to the loan user
     */
    public function loanUser()
    {
        return $this->belongsTo(User::class, 'loan_user_id');
    }

    /**
     * Relationship to the DisbursedLoan
     */
    public function loan()
    {
        return $this->belongsTo(DisbursedLoan::class, 'loan_id', 'loanid');
    }

    /**
     * Relationship to the Customer (loan owner)
     */
    public function customer()
    {
        return $this->belongsTo(Customer::class, 'loan_customer_id', 'customerid');
    }

    /**
     * Relationship to the Application
     */
    public function application()
    {
        return $this->belongsTo(Application::class, 'application_id');
    }

    /**
     * Scope to filter by status (Successful, Pending, Failed)
     */
    public function scopeStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope to filter by loan ID
     */
    public function scopeByLoan($query, $loanId)
    {
        return $query->where('loan_id', $loanId);
    }

    /**
     * Scope to filter by customer ID
     */
    public function scopeByCustomer($query, $customerId)
    {
        return $query->where('loan_customer_id', $customerId);
    }

    /**
     * Scope to filter by company ID
     */
    public function scopeByCompany($query, $companyId)
    {
        return $query->where('company_id', $companyId);
    }

    /**
     * Get payment details with loan information
     */
    public function getPaymentDetailsAttribute()
    {
        return [
            'payment_id' => $this->id,
            'reference_id' => $this->reference_id,
            'identifier' => $this->identifier,
            'amount' => $this->amount,
            'currency' => $this->currency,
            'status' => $this->status,
            'payment_type' => $this->payment_type,
            'account_number' => $this->account_number,
            'created_at' => $this->created_at,
            'loan_info' => $this->loan ? [
                'loan_id' => $this->loan_id,
                'loan_number' => $this->loan->loannumber,
                'customer_name' => $this->customer ? $this->customer->name : null,
                'customer_id' => $this->loan_customer_id,
                'company_id' => $this->loan_company_id,
            ] : null,
            'application_info' => $this->application_id ? [
                'application_id' => $this->application_id,
                'status' => $this->application->status ?? null,
            ] : null,
        ];
    }

    /**
     * Check if payment is successful
     */
    public function isSuccessful()
    {
        return in_array($this->status, ['Successful', 'SUCCESS', 'success']);
    }

    /**
     * Check if payment is pending
     */
    public function isPending()
    {
        return in_array($this->status, ['Pending', 'PENDING', 'pending']);
    }

    /**
     * Check if payment is failed
     */
    public function isFailed()
    {
        return in_array($this->status, ['Failed', 'FAILED', 'failed']);
    }

    /**
     * Update payment status from webhook
     */
    public function updateFromWebhook(array $webhookData)
    {
        $this->update([
            'status' => $webhookData['status'] ?? $this->status,
            'message' => $webhookData['message'] ?? $this->message,
            'webhook_received_at' => now(),
            'webhook_payload' => $webhookData,
        ]);
    }

    /**
     * Get metadata value
     */
    public function getMetadataValue(string $key, $default = null)
    {
        return $this->metadata[$key] ?? $default;
    }

    /**
     * Set metadata value
     */
    public function setMetadataValue(string $key, $value)
    {
        $metadata = $this->metadata ?? [];
        $metadata[$key] = $value;
        $this->metadata = $metadata;
        $this->save();
    }
}