<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PaymentSchedule extends Model
{
    use HasFactory;

    protected $primaryKey = 'scheduleid';
    
    protected $table = 'paymentschedule';

    protected $fillable = [
        'loannumber',
        'loanid',
        'interestamount',
        'principalamount',
        'processingfeesamount',
        'totalamount',
        'paymentdate',
        'transactiontype',
        'status',
        'installment_number',
        'outstanding_principal',
        'outstanding_interest',
        'outstanding_processingfees_amoun',
        'cumulative_principal_paid',
        'cumulative_interest_paid',
        'cumulative_processingfees_amoun',
        'paid_date',
        'paid_principal',
        'paid_interest',
        'paid_penalty',
        'paid_processingefees_amount',
        'total_paid',
        'branchid',
        'companyid',
        'createdby',
        'updatedby',
        
        // Accrual fields (ONLY these are needed)
        'actualaccruedinterest',
        'actualaccruedprocessingfee',
        'accrual_start_date',
        'accrual_end_date',
        // REMOVED: 'daily_interest_rate', 'daily_processing_rate'
    ];

    protected $casts = [
        'interestamount' => 'decimal:2',
        'principalamount' => 'decimal:2',
        'processingfeesamount' => 'decimal:2',
        'totalamount' => 'decimal:2',
        
        'outstanding_principal' => 'decimal:2',
        'outstanding_interest' => 'decimal:2',
        'outstanding_processingfees_amoun' => 'decimal:2',
        
        'cumulative_principal_paid' => 'decimal:2',
        'cumulative_interest_paid' => 'decimal:2',
        'cumulative_processingfees_amoun' => 'decimal:2',
        
        'paid_principal' => 'decimal:2',
        'paid_interest' => 'decimal:2',
        'paid_penalty' => 'decimal:2',
        'paid_processingefees_amount' => 'decimal:2',
        'total_paid' => 'decimal:2',
        
        // Accrual field casts
        'actualaccruedinterest' => 'decimal:2',
        'actualaccruedprocessingfee' => 'decimal:2',
        // REMOVED: 'daily_interest_rate', 'daily_processing_rate'
        
        'paymentdate' => 'date',
        'paid_date' => 'date',
        'accrual_start_date' => 'date',
        'accrual_end_date' => 'date',
        'updatedon' => 'datetime',
    ];

    // Relationships
    public function loan()
    {
        return $this->belongsTo(DisbursedLoan::class, 'loanid', 'loanid');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branchid');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'companyid');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'createdby');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updatedby');
    }

    // Accrual-related methods
    public function getTotalAccruedAttribute()
    {
        return bcadd($this->actualaccruedinterest, $this->actualaccruedprocessingfee, 2);
    }

    public function getAccruedBalanceAttribute()
    {
        $totalAccrued = $this->totalaccrued;
        $accruedPaid = bcadd($this->paid_interest ?? 0, $this->paid_processingefees_amount ?? 0, 2);
        
        return bcsub($totalAccrued, $accruedPaid, 2);
    }

    public function getTotalDueAttribute()
    {
        // Use ACCRUED amounts if available, otherwise use scheduled amounts
        $interest = $this->actualaccruedinterest > 0 
            ? $this->actualaccruedinterest 
            : ($this->interestamount ?? 0);
        
        $processing = $this->actualaccruedprocessingfee > 0 
            ? $this->actualaccruedprocessingfee 
            : ($this->processingfeesamount ?? 0);
        
        return bcadd(bcadd(
            $this->principalamount ?? 0, 
            $interest, 
            2
        ), $processing, 2);
    }

    public function getBalanceAttribute()
    {
        return bcsub($this->totaldue, $this->total_paid ?? 0, 2);
    }

    public function getIsAccruedAttribute()
    {
        return $this->actualaccruedinterest > 0 || $this->actualaccruedprocessingfee > 0;
    }

    public function getAccrualDaysAttribute()
    {
        if (!$this->accrual_start_date || !$this->accrual_end_date) {
            return 0;
        }
        
        $start = \Carbon\Carbon::parse($this->accrual_start_date);
        $end = \Carbon\Carbon::parse($this->accrual_end_date);
        
        return $start->diffInDays($end) + 1; // Inclusive of both dates
    }

    /**
     * Calculate daily interest accrual - gets rate from loan
     */
    public function calculateDailyInterest()
    {
        // Get the daily rate from the loan, NOT stored here
        if (!$this->loan) {
            return 0;
        }
        
        $dailyRate = $this->loan->getDailyInterestRate();
        $principalBalance = $this->loan->principalbalance ?? $this->outstanding_principal ?? 0;
        
        return $principalBalance * $dailyRate;
    }
    
    /**
     * Calculate daily processing fee accrual - gets rate from loan
     */
    public function calculateDailyProcessingFee()
    {
        if (!$this->loan) {
            return 0;
        }
        
        $dailyRate = $this->loan->getDailyProcessingFeeRate();
        $principalBalance = $this->loan->principalbalance ?? $this->outstanding_principal ?? 0;
        
        return $principalBalance * $dailyRate;
    }

    // Scopes
    public function scopeScheduled($query)
    {
        return $query->where('status', 'scheduled');
    }

    public function scopePaid($query)
    {
        return $query->where('status', 'paid');
    }

    public function scopeOverdue($query)
    {
        return $query->where('status', 'overdue');
    }

    public function scopeByLoan($query, $loanId)
    {
        return $query->where('loanid', $loanId);
    }

    public function scopeUpcoming($query)
    {
        return $query->where('paymentdate', '>=', now())->where('status', 'scheduled');
    }

    public function scopePastDue($query)
    {
        return $query->where('paymentdate', '<', now())->where('status', 'scheduled');
    }

    public function scopeWithAccruals($query)
    {
        return $query->where(function($q) {
            $q->where('actualaccruedinterest', '>', 0)
              ->orWhere('actualaccruedprocessingfee', '>', 0);
        });
    }

    public function scopeCurrentAccrual($query, $date = null)
    {
        $date = $date ?: now()->format('Y-m-d');
        
        return $query->where('accrual_start_date', '<=', $date)
                     ->where('accrual_end_date', '>=', $date)
                     ->where('status', '!=', 'paid');
    }

    // Helper Methods
    public function isOverdue(){
       return PaymentSchedule::where('loanid', $this->loanid)
            ->whereNotNull('loanid')
            ->where(function($query) {
            $query->where('status', 'overdue')
                  ->orWhere(function($q) {
                      $q->where('status', 'scheduled')
                        ->where('paymentdate', '<', now());
                  });
        })->where(function($query) {
            $query->where(function($q) {
                $q->whereColumn('total_paid', '<', \DB::raw('principalamount + interestamount + processingfeesamount'))
                  ->orWhereNull('total_paid');
            })->orWhere('total_paid', 0);
        }) ->exists();
    }

    public function markAsPaid($paidDate = null, $paidAmounts = [])
    {
        // Use accrued amounts if available
        $principal = $paidAmounts['principal'] ?? $this->principalamount;
        $interest = $paidAmounts['interest'] ?? ($this->actualaccruedinterest > 0 ? $this->actualaccruedinterest : $this->interestamount);
        $penalty = $paidAmounts['penalty'] ?? 0;
        $fee = $paidAmounts['fee'] ?? ($this->actualaccruedprocessingfee > 0 ? $this->actualaccruedprocessingfee : $this->processingfeesamount);

        $this->update([
            'status' => 'paid',
            'paid_date' => $paidDate ?? now(),
            'paid_principal' => $principal,
            'paid_interest' => $interest,
            'paid_penalty' => $penalty,
            'paid_processingefees_amount' => $fee,
            'total_paid' => $principal + $interest + $penalty + $fee,
            'updatedby' => auth()->id(),
            
            // Close accrual period
            'accrual_end_date' => $paidDate ?? now()->format('Y-m-d'),
        ]);
    }

    public function getFormattedPaymentDateAttribute()
    {
        if ($this->paymentdate instanceof \Carbon\Carbon) {
            return $this->paymentdate->format('M d, Y');
        }
        return null;
    }

    public function getFormattedTotalAmountAttribute()
    {
        return 'ZMW ' . number_format($this->totalamount, 2);
    }

    // New: Get accrued amounts formatted
    public function getFormattedAccruedInterestAttribute()
    {
        return 'ZMW ' . number_format($this->actualaccruedinterest, 2);
    }

    public function getFormattedAccruedProcessingFeeAttribute()
    {
        return 'ZMW ' . number_format($this->actualaccruedprocessingfee, 2);
    }

    public function getFormattedAccrualPeriodAttribute()
    {
        if (!$this->accrual_start_date || !$this->accrual_end_date) {
            return 'Not set';
        }
        
        $start = \Carbon\Carbon::parse($this->accrual_start_date)->format('M d, Y');
        $end = \Carbon\Carbon::parse($this->accrual_end_date)->format('M d, Y');
        
        return $start . ' to ' . $end;
    }

    /**
     * Check if schedule is overdue considering accrued amounts
     */
    public function getIsOverdueAttribute()
    {
        $dueDate = \Carbon\Carbon::parse($this->paymentdate);
        return $dueDate->isPast() && $this->balance > 0;
    }

    /**
     * Check if schedule is paid
     */
    public function getIsPaidAttribute()
    {
        return $this->status === 'paid' || $this->balance <= 0;
    }
    
    
    // Add this method to your PaymentSchedule model:
    // Add this to your PaymentSchedule model (App\Models\PaymentSchedule):

/**
 * Mark schedule as closed during loan closure
 */
public function markAsClosed($closureDate, $writeOffAmounts = [])
{
    $this->update([
        'status' => 'closed',
        'paid_date' => $closureDate,
        'paid_principal' => $writeOffAmounts['principal'] ?? 0,
        'paid_interest' => $writeOffAmounts['interest'] ?? ($this->actualaccruedinterest > 0 ? $this->actualaccruedinterest : $this->interestamount),
        'paid_penalty' => $writeOffAmounts['penalty'] ?? 0,
        'paid_processingefees_amount' => $writeOffAmounts['processing_fee'] ?? ($this->actualaccruedprocessingfee > 0 ? $this->actualaccruedprocessingfee : $this->processingfeesamount),
        'total_paid' => $writeOffAmounts['total'] ?? 0,
        'updatedby' => auth()->id() ?? $this->updatedby,
        'accrual_end_date' => $closureDate,
    ]);
    
    return $this;
}
}