<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PayrollSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'company_id',
        'financial_year_id',
        'setting_type',
        'year',
        'monthly_rate',
        'annual_rate',
        'monthly_ceiling',
        'annual_ceiling',
        'start_date',
        'end_date',
        'period_name',
        'period_type',
        'is_active',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'monthly_rate' => 'decimal:2',
        'annual_rate' => 'decimal:2',
        'monthly_ceiling' => 'decimal:2',
        'annual_ceiling' => 'decimal:2',
        'start_date' => 'date',
        'end_date' => 'date',
        'is_active' => 'boolean'
    ];

    // Relationships
    public function company()
    {
        return $this->belongsTo(Company::class, 'company_id');
    }

    public function financialYear()
    {
        return $this->belongsTo(FinancialYear::class, 'financial_year_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function payeTaxBands()
    {
        return $this->hasMany(PayeTaxBand::class, 'payroll_setting_id');
    }

    // Scopes
    public function scopeForCompany($query, $companyId)
    {
        return $query->where('company_id', $companyId);
    }

    public function scopeForFinancialYear($query, $financialYearId)
    {
        return $query->where('financial_year_id', $financialYearId);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('setting_type', $type);
    }

    // Helper methods
    public function getSettingTypeNameAttribute()
    {
        return ucfirst(str_replace('_', ' ', $this->setting_type));
    }

    public function getFormattedRateAttribute()
    {
        if ($this->monthly_rate !== null) {
            return number_format($this->monthly_rate, 2) . '%';
        }
        if ($this->annual_rate !== null) {
            return number_format($this->annual_rate, 2) . '%';
        }
        return 'N/A';
    }

    public function getFormattedCeilingAttribute()
    {
        if ($this->monthly_ceiling !== null) {
            return 'K' . number_format($this->monthly_ceiling, 2) . ' per month';
        }
        if ($this->annual_ceiling !== null) {
            return 'K' . number_format($this->annual_ceiling, 2) . ' per annum';
        }
        return 'No ceiling';
    }
}