<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class PettyCash extends Model
{
    use HasFactory;

    protected $primaryKey = 'petty_cash_id';
    protected $table = 'petty_cash';
    
    protected $fillable = [
        'petty_cash_number',
        'transaction_date',
        'payee',
        'description',
        'amount',
        'currency',
        'exchange_rate',
        'base_amount',
        'source_account_id',
        'receipting_account_id', // NEW FIELD
        'expense_category_id',
        'chart_of_account_id',
        'status',
        'requested_by',
        'approved_by',
        'approved_at',
        'approval_notes',
        'rejection_reason',
        'receipt_number',
        'project_code',
        'department',
        'company_id',
        'branch_id',
        'created_by',
        'updated_by',
        'reimbursed_at',
        'reimbursed_by',
        'reversed_at',
        'reversed_by',
        'reversal_reason'
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'amount' => 'decimal:2',
        'exchange_rate' => 'decimal:4',
        'base_amount' => 'decimal:2',
        'approved_at' => 'datetime',
        'reimbursed_at' => 'datetime',
        'reversed_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->petty_cash_number)) {
                $model->petty_cash_number = static::generatePettyCashNumber();
            }
        });

        // Calculate total from items when saving
        static::saving(function ($model) {
            if ($model->items()->exists()) {
                $totalAmount = $model->items()->sum('amount');
                $model->amount = $totalAmount;
                $model->base_amount = $totalAmount * $model->exchange_rate;
            }
        });
    }

    /**
     * Generate unique petty cash number
     */
    public static function generatePettyCashNumber(): string
    {
        $prefix = 'PC';
        $year = date('Y');
        $month = date('m');
        
        $lastNumber = static::where('petty_cash_number', 'like', "{$prefix}{$year}{$month}%")
            ->orderBy('petty_cash_number', 'desc')
            ->value('petty_cash_number');
            
        $sequence = 1;
        if ($lastNumber) {
            $sequence = (int) substr($lastNumber, -4) + 1;
        }
        
        return "{$prefix}{$year}{$month}" . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    // Items relationship
    public function items(): HasMany
    {
        return $this->hasMany(PettyCashItem::class, 'petty_cash_id');
    }

    /**
     * Get the source cash/bank account
     */
    public function sourceAccount(): BelongsTo
    {
        return $this->belongsTo(CashBankAccount::class, 'source_account_id', 'cashBankId');
    }

    /**
     * Get the receipting cash account (NEW)
     */
    public function receiptingAccount(): BelongsTo
    {
        return $this->belongsTo(CashBankAccount::class, 'receipting_account_id', 'cashBankId');
    }

    /**
     * Get the chart of account
     */
    public function chartOfAccount(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'chart_of_account_id', 'id');
    }

    /**
     * Get the expense category
     */
  

    /**
     * Get the requester
     */
    public function requester(): BelongsTo
    {
        return $this->belongsTo(User::class, 'requested_by', 'id');
    }

    /**
     * Get the approver
     */
    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by', 'id');
    }

    /**
     * Get the creator
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by', 'id');
    }

    /**
     * Get the updater
     */
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by', 'id');
    }

    /**
     * Get the reimbursed by user
     */
    public function reimbursedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reimbursed_by', 'id');
    }

    /**
     * Get the reversed by user
     */
    public function reversedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reversed_by', 'id');
    }

    /**
     * Get the company
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class, 'company_id', 'id');
    }

    /**
     * Get the branch
     */
    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class, 'branch_id', 'id');
    }

    /**
     * Check if petty cash can be edited
     */
    public function canEdit(): bool
    {
        return in_array($this->status, ['Draft', 'Rejected']);
    }

    /**
     * Check if petty cash can be reversed
     */
    public function canReverse(): bool
    {
        return in_array($this->status, ['Paid', 'Reimbursed']);
    }

    /**
     * Get total amount with currency
     */
    public function getFormattedAmountAttribute(): string
    {
        return $this->currency . ' ' . number_format($this->amount, 2);
    }

    /**
     * Get base amount with base currency
     */
    public function getFormattedBaseAmountAttribute(): string
    {
        // Assuming base currency is USD
        return 'USD ' . number_format($this->base_amount, 2);
    }
}