<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Services\NotificationSettingService;
use App\Services\SmsService;
use App\Mail\RepaymentReceivedMail;
use Illuminate\Support\Facades\Mail;

class Repayment extends Model
{
    use HasFactory;

    protected $table = 'repayments';

    protected $fillable = [
        'customerid', // this is the customer id for the loan being paid for
        'rpid', // this is the product id from the loantype table
        'rloanid', // loan id from the disbursedloans tble
        'rtotal', // total amount paid
        'rinterest', // amount paid allocated towards interest
        'processing_fees_amount_paid', // amount paid allocated towards processign fees if if fees are applicable
        'penalties', // This is the amount paid allocated towards penalties
        'rprincipal', // amount paid allocated towards principal
        'rdate', // this is the date when the amount was paid
        'userid', // This is the user id of the person recoding the payment made
        'branchid', // This is the branchid for the loan being paid for
        'companyid', // This is the companyid for the loan being paid for
        'pmethod', // This is the payment method (Cash,Bank,Mobile Money)
        'paccount', // This is the account into which the amount was paid into - refer to the cash and banking table.
        'transtype', // This is the transaction type in this case this is Repayment
        'status', // Verified or Unverified
        'paymentreference', // This is the reference number for the payment document e.g reference number for all payment methods
        
        // NEW FIELDS for payment method details
        'bank_name', // Name of bank (for Bank Transfer)
        'account_holder', // Bank account holder name (for Bank Transfer)
        'account_number', // Bank account number (for Bank Transfer)
        'bank_branch', // Bank branch (for Bank Transfer)
        'swift_code', // SWIFT code (for Bank Transfer, optional)
        'mobile_channel', // Mobile money channel: MTN/Airtel/Zamtel (for Mobile Money)
        'mobile_number', // Mobile phone number (for Mobile Money)
        'payment_notes', // Additional notes about the payment
        'allocation_mode', // Allocation mode: auto or manual
    ];

    protected $casts = [
        'rtotal' => 'decimal:2',
        'rinterest' => 'decimal:2',
        'rprincipal' => 'decimal:2',
        'processing_fees_amount_paid' => 'decimal:2',
        'penalties' => 'decimal:2',
        'rdate' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'allocation_mode' => 'string',
    ];

    /**
     * The "booted" method of the model.
     * Triggers automated notifications upon repayment creation.
     */
    protected static function booted()
    {
        static::created(function ($repayment) {
            $customer = $repayment->customer;
            if (!$customer) return;

            // 1. Check and send Email Notification
            if (NotificationSettingService::isEnabled('repayment_received', 'email')) {
                Mail::to($customer->email)->send(new RepaymentReceivedMail($repayment));
            }

            // 2. Check and send SMS Notification using the dynamic template
            if (NotificationSettingService::isEnabled('repayment_received', 'sms')) {
                NotificationSettingService::sendSmsTemplate('repayment_received', $customer->phone, [
                    'Name'   => $customer->first_name,
                    'Amount' => number_format($repayment->rtotal, 2),
                    'LoanNo' => $repayment->loan->loannumber ?? 'N/A',
                    'Date'   => $repayment->rdate->format('d M Y')
                ]);
            }
        });
    }

    // ====================== RELATIONSHIPS ======================

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customerid');
    }

    /**
     * Relationship to the loan (DisbursedLoan model)
     */
    public function loan()
    {
        return $this->belongsTo(DisbursedLoan::class, 'rloanid', 'loanid');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'userid');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branchid');
    }

    public function company()
    {
        return $this->belongsTo(Company::class, 'companyid');
    }

    /**
     * Get payment account details
     */
    public function paymentAccount()
    {
        return $this->belongsTo(CashBankAccount::class, 'paccount', 'cashBankId');
    }

    // ====================== ACCESSORS ======================

    /**
     * Get formatted payment method with details
     */
    public function getPaymentMethodWithDetailsAttribute()
    {
        switch ($this->pmethod) {
            case 'Bank':
                $details = [];
                if ($this->bank_name) $details[] = "Bank: {$this->bank_name}";
                if ($this->account_number) $details[] = "Acc: {$this->account_number}";
                if ($this->bank_branch) $details[] = "Branch: {$this->bank_branch}";
                return "Bank Transfer" . ($details ? ' (' . implode(', ', $details) . ')' : '');
                
            case 'Mobile Money':
                $details = [];
                if ($this->mobile_channel) $details[] = $this->mobile_channel;
                if ($this->mobile_number) $details[] = $this->mobile_number;
                return "Mobile Money" . ($details ? ' (' . implode(', ', $details) . ')' : '');
                
            case 'Cash':
                return "Cash Payment";
                
            default:
                return $this->pmethod ?? 'N/A';
        }
    }

    /**
     * Get allocation breakdown as array
     */
    public function getAllocationBreakdownAttribute()
    {
        return [
            'principal' => $this->rprincipal,
            'interest' => $this->rinterest,
            'processing_fee' => $this->processing_fees_amount_paid,
            'penalty' => $this->penalties,
            'total' => $this->rtotal,
        ];
    }

    /**
     * Get payment details based on method
     */
    public function getPaymentDetailsAttribute()
    {
        if ($this->pmethod === 'Bank') {
            return [
                'bank_name' => $this->bank_name,
                'account_holder' => $this->account_holder,
                'account_number' => $this->account_number,
                'bank_branch' => $this->bank_branch,
                'swift_code' => $this->swift_code,
            ];
        } elseif ($this->pmethod === 'Mobile Money') {
            return [
                'channel' => $this->mobile_channel,
                'mobile_number' => $this->mobile_number,
            ];
        }
        
        return [];
    }
}