<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    use HasFactory, Notifiable;

    protected $fillable = [
        'firstname',
        'middlename', 
        'lastname',
        'email',
        'password',
        'branchid',
        'company_id',
        'status',
        'last_login',
        'created_by'
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $attributes = [
        'branchid' => 1,
        'status' => 1,
    ];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'last_login' => 'datetime',
            'status' => 'boolean',
        ];
    }

    /**
     * Get the user's full name
     */
    public function getNameAttribute()
    {
        $names = [$this->firstname];
        
        if (!empty($this->middlename)) {
            $names[] = $this->middlename;
        }
        
        $names[] = $this->lastname;
        
        return implode(' ', $names);
    }

    /**
     * Get the user's branch
     */
    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branchid');
    }

    /**
     * Get the user who created this user
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get status as human-readable text
     */
    public function getStatusTextAttribute()
    {
        return $this->status ? 'Active' : 'Inactive';
    }

    /**
     * Scope active users
     */
    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    /**
     * Scope inactive users  
     */
    public function scopeInactive($query)
    {
        return $query->where('status', 0);
    }

  public function company()
  {
    return $this->belongsTo(Company::class, 'company_id');
  }

    /**
     * Scope users by branch
     */
    public function scopeByBranch($query, $branchId)
    {
        return $query->where('branchid', $branchId);
    }

    // Permissions relationship (keep your existing methods)
    public function permissions()
    {
        return $this->belongsToMany(Permission::class);
    }

    // Check if user has a specific permission
    public function hasPermission($permissionSlug)
    {
        // If user has poweruser permission, they have all permissions
        if ($this->permissions->contains('slug', 'poweruser')) {
            return true;
        }
        
        return $this->permissions->contains('slug', $permissionSlug);
    }

    // Check if user has any of the given permissions
    public function hasAnyPermission(array $permissionSlugs)
    {
        if ($this->permissions->contains('slug', 'poweruser')) {
            return true;
        }
        
        return $this->permissions->whereIn('slug', $permissionSlugs)->count() > 0;
    }

    // Check if user has all of the given permissions
    public function hasAllPermissions(array $permissionSlugs)
    {
        if ($this->permissions->contains('slug', 'poweruser')) {
            return true;
        }
        
        foreach ($permissionSlugs as $slug) {
            if (!$this->permissions->contains('slug', $slug)) {
                return false;
            }
        }
        return true;
    }

    // Assign permission to user
    public function assignPermission($permissionSlug)
    {
        $permission = Permission::where('slug', $permissionSlug)->first();
        if ($permission && !$this->permissions->contains($permission->id)) {
            $this->permissions()->attach($permission->id);
        }
    }

    // Revoke permission from user
    public function revokePermission($permissionSlug)
    {
        $permission = Permission::where('slug', $permissionSlug)->first();
        if ($permission) {
            $this->permissions()->detach($permission->id);
        }
    }

    /**
     * Record user login
     */
    public function recordLogin()
    {
        $this->update(['last_login' => now()]);
    }
}