<?php
// app/Models/Vendor.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Vendor extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'display_name',
        'company_name',
        'first_name',
        'middle_name',
        'last_name',
        'suffix',
        'email',
        'mobile_phone',
        'other_phone',
        'fax',
        'website',
        'check_printing_name',
        
        // Billing Address
        'billing_street1',
        'billing_street2',
        'billing_city',
        'billing_state',
        'billing_zip',
        'billing_country',
        
        // Shipping Address
        'shipping_street1',
        'shipping_street2',
        'shipping_city',
        'shipping_state',
        'shipping_zip',
        'shipping_country',
        
        // Additional Information
        'account_number',
        'chart_of_account_id', // Link to chart of accounts
        'currency_id', // NEW: Link to currencies table
        'payment_terms',
        'vendor_1099',
        'tax_id',
        'currency', // Keep for backward compatibility
        'balance',
        'active',
        'overdue',
        'last_transaction_date',
        'company_id',
    ];

    protected $casts = [
        'vendor_1099' => 'boolean',
        'active' => 'boolean',
        'overdue' => 'boolean',
        'balance' => 'decimal:2',
        'last_transaction_date' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    protected $appends = [
        'full_name',
        'billing_address',
        'shipping_address',
        'status_badge',
        'balance_formatted',
        'account_name',
        'currency_with_symbol', // NEW: Currency with symbol
        'currency_symbol', // NEW: Just the symbol
    ];

    /**
     * Scope for active vendors
     */
    public function scopeActive($query)
    {
        return $query->where('active', true);
    }

    /**
     * Scope for inactive vendors
     */
    public function scopeInactive($query)
    {
        return $query->where('active', false);
    }

    /**
     * Scope for vendors with balance
     */
    public function scopeWithBalance($query)
    {
        return $query->where('balance', '>', 0);
    }

    /**
     * Scope for vendors without balance
     */
    public function scopeWithoutBalance($query)
    {
        return $query->where('balance', '<=', 0);
    }

    /**
     * Scope for 1099 vendors
     */
    public function scopeVendor1099($query)
    {
        return $query->where('vendor_1099', true);
    }

    /**
     * Scope for non-1099 vendors
     */
    public function scopeNonVendor1099($query)
    {
        return $query->where('vendor_1099', false);
    }

    /**
     * Scope for overdue vendors
     */
    public function scopeOverdue($query)
    {
        return $query->where('overdue', true);
    }

    /**
     * Search scope
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('display_name', 'like', "%{$search}%")
              ->orWhere('company_name', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%")
              ->orWhere('mobile_phone', 'like', "%{$search}%")
              ->orWhere('account_number', 'like', "%{$search}%");
        });
    }

    /**
     * Get the full name attribute
     */
    public function getFullNameAttribute()
    {
        $parts = [];
        
        if ($this->first_name) $parts[] = $this->first_name;
        if ($this->middle_name) $parts[] = $this->middle_name;
        if ($this->last_name) $parts[] = $this->last_name;
        if ($this->suffix) $parts[] = $this->suffix;
        
        return implode(' ', $parts) ?: null;
    }

    /**
     * Get the billing address as a formatted string
     */
    public function getBillingAddressAttribute()
    {
        $parts = [];
        
        if ($this->billing_street1) $parts[] = $this->billing_street1;
        if ($this->billing_street2) $parts[] = $this->billing_street2;
        
        $cityStateZip = [];
        if ($this->billing_city) $cityStateZip[] = $this->billing_city;
        if ($this->billing_state) $cityStateZip[] = $this->billing_state;
        if ($this->billing_zip) $cityStateZip[] = $this->billing_zip;
        
        if (!empty($cityStateZip)) {
            $parts[] = implode(', ', $cityStateZip);
        }
        
        if ($this->billing_country) $parts[] = $this->billing_country;
        
        return implode("\n", $parts) ?: null;
    }

    /**
     * Get the shipping address as a formatted string
     */
    public function getShippingAddressAttribute()
    {
        $parts = [];
        
        if ($this->shipping_street1) $parts[] = $this->shipping_street1;
        if ($this->shipping_street2) $parts[] = $this->shipping_street2;
        
        $cityStateZip = [];
        if ($this->shipping_city) $cityStateZip[] = $this->shipping_city;
        if ($this->shipping_state) $cityStateZip[] = $this->shipping_state;
        if ($this->shipping_zip) $cityStateZip[] = $this->shipping_zip;
        
        if (!empty($cityStateZip)) {
            $parts[] = implode(', ', $cityStateZip);
        }
        
        if ($this->shipping_country) $parts[] = $this->shipping_country;
        
        return implode("\n", $parts) ?: null;
    }

    /**
     * Get the status badge for display
     */
    public function getStatusBadgeAttribute()
    {
        if (!$this->active) {
            return [
                'label' => 'Inactive',
                'color' => 'danger',
                'class' => 'badge bg-danger'
            ];
        }
        
        if ($this->overdue) {
            return [
                'label' => 'Overdue',
                'color' => 'warning',
                'class' => 'badge bg-warning'
            ];
        }
        
        return [
            'label' => 'Active',
            'color' => 'success',
            'class' => 'badge bg-success'
        ];
    }

    /**
     * Get the formatted balance
     */
    public function getBalanceFormattedAttribute()
    {
        return number_format($this->balance, 2);
    }

    /**
     * Get the account name from chart of accounts
     */
    public function getAccountNameAttribute()
    {
        if ($this->chartOfAccount) {
            return $this->chartOfAccount->code . ' - ' . $this->chartOfAccount->name;
        }
        
        return $this->account_number ? '#' . $this->account_number : '—';
    }

    /**
     * Get the currency with symbol
     */
    public function getCurrencyWithSymbolAttribute()
    {
        if ($this->currencyDetail) {
            return $this->currencyDetail->code . ' (' . $this->currencyDetail->symbol . ')';
        }
        
        return $this->currency ?? 'USD';
    }

    /**
     * Get the currency symbol
     */
    public function getCurrencySymbolAttribute()
    {
        if ($this->currencyDetail) {
            return $this->currencyDetail->symbol;
        }
        
        return '$'; // Default to USD symbol
    }

    /**
     * Get the primary phone number
     */
    public function getPrimaryPhoneAttribute()
    {
        return $this->mobile_phone ?: $this->other_phone;
    }

    /**
     * Relationships
     */
    public function contacts()
    {
        return $this->hasMany(VendorContact::class);
    }

    public function statements()
    {
        return $this->hasMany(VendorStatement::class);
    }

    public function notes()
    {
        return $this->hasMany(VendorNote::class);
    }

    /**
     * Relationship to Chart of Account
     */
    public function chartOfAccount()
    {
        return $this->belongsTo(ChartOfAccount::class, 'chart_of_account_id');
    }

    /**
     * Relationship to Currency
     */
    public function currencyDetail()
    {
        return $this->belongsTo(Currency::class, 'currency_id');
    }

    /**
     * Get the primary contact
     */
    public function primaryContact()
    {
        return $this->contacts()->where('is_primary', true)->first();
    }

    /**
     * Check if vendor has 1099 status
     */
    public function is1099Vendor()
    {
        return $this->vendor_1099;
    }

    /**
     * Update vendor balance
     */
    public function updateBalance($amount)
    {
        $this->balance += $amount;
        $this->save();
        
        // Check if overdue (balance > 0 and last transaction > 30 days ago)
        if ($this->balance > 0 && $this->last_transaction_date) {
            $daysOverdue = now()->diffInDays($this->last_transaction_date);
            $this->overdue = $daysOverdue > 30;
            $this->save();
        }
        
        return $this;
    }
}