<?php

namespace App\Services;

use App\Models\PaymentSchedule;
use App\Models\DisbursedLoan;
use App\Models\Customer;
use App\Services\SmsService;
use App\Mail\PaymentDueReminder;
use App\Mail\PaymentOverdueReminder;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PaymentReminderService
{
    private $smsService;
    
    public function __construct(SmsService $smsService)
    {
        $this->smsService = $smsService;
    }

    /**
     * Main method to process all payment reminders and overdue alerts
     */
    public function processPaymentReminders()
    {
        Log::info('=== STARTING PAYMENT REMINDER PROCESS ===');
        
        $results = [
            'due_reminders_sent' => 0,
            'overdue_alerts_sent' => 0,
            'due_sms_sent' => 0,
            'due_emails_sent' => 0,
            'overdue_sms_sent' => 0,
            'overdue_emails_sent' => 0,
            'errors' => []
        ];

        try {
            // 1. Process payment due reminders (daily reminders for payments due within reminder window)
            $dueResults = $this->processDueReminders();
            $results['due_reminders_sent'] = $dueResults['due_reminders_sent'];
            $results['due_sms_sent'] = $dueResults['due_sms_sent'];
            $results['due_emails_sent'] = $dueResults['due_emails_sent'];
            
            // 2. Process overdue payment alerts
            $overdueResults = $this->processOverdueAlerts();
            $results['overdue_alerts_sent'] = $overdueResults['overdue_alerts_sent'];
            $results['overdue_sms_sent'] = $overdueResults['overdue_sms_sent'];
            $results['overdue_emails_sent'] = $overdueResults['overdue_emails_sent'];
            
            Log::info('Payment reminder process completed', $results);
            
        } catch (\Exception $e) {
            Log::error('Payment reminder process failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            $results['errors'][] = $e->getMessage();
        }
        
        Log::info('=== PAYMENT REMINDER PROCESS COMPLETED ===');
        return $results;
    }

    /**
     * Process payment due reminders for upcoming installments
     * Sends DAILY reminders for all payments due within the reminder window
     * Example: reminder_days = 5 → sends daily reminders for payments due within next 5 days
     */
    private function processDueReminders()
    {
        $results = [
            'due_reminders_sent' => 0,
            'due_sms_sent' => 0,
            'due_emails_sent' => 0
        ];

        try {
            // Get reminder days from notification_events
            $dueEvent = DB::table('notification_events')
                ->where('event_key', 'payment_due')
                ->first();
            
            // Use the actual value from database (5 days in your case) - CAST TO INTEGER
            $reminderWindowDays = $dueEvent ? (int) $dueEvent->reminder_days : 0;
            
            Log::debug('Due reminder - raw value from DB', [
                'raw_value' => $dueEvent->reminder_days ?? 'NULL',
                'casted_value' => $reminderWindowDays,
                'type_raw' => gettype($dueEvent->reminder_days ?? 'null'),
                'type_casted' => gettype($reminderWindowDays)
            ]);
            
            if ($reminderWindowDays <= 0) {
                Log::info('Due reminders disabled (reminder_days = 0 or negative)', [
                    'reminder_days' => $reminderWindowDays
                ]);
                return $results;
            }
            
            $today = Carbon::today();
            $windowEndDate = $today->copy()->addDays($reminderWindowDays);
            
            Log::info('DUE REMINDERS: Calculating window', [
                'today' => $today->format('Y-m-d'),
                'reminder_window_days' => $reminderWindowDays,
                'window_start' => $today->format('Y-m-d'),
                'window_end' => $windowEndDate->format('Y-m-d'),
                'logic' => 'Send daily reminders for payments due within next ' . $reminderWindowDays . ' days'
            ]);
            
            // Find ALL payments due within the next X days (0 to X days from now)
            $dueSchedules = PaymentSchedule::where('status', 'scheduled')
                ->whereBetween('paymentdate', [$today, $windowEndDate]) // Due within 0-X days
                ->where('paymentdate', '>', $today) // Exclude today (today is handled by overdue)
                ->where(function($query) {
                    // Only for schedules that have outstanding balance
                    $query->whereColumn('total_paid', '<', 'totalamount')
                          ->orWhere('total_paid', 0);
                })
                ->with(['loan.customer', 'loan' => function($q) {
                    $q->with('customer');
                }])
                ->get();
            
            Log::info('DUE REMINDERS: Found schedules', [
                'count' => $dueSchedules->count(),
                'reminder_window_days' => $reminderWindowDays,
                'window_end_date' => $windowEndDate->format('Y-m-d')
            ]);

            foreach ($dueSchedules as $schedule) {
                try {
                    // Debug: Show schedule details
                    $daysUntilDue = $today->diffInDays($schedule->paymentdate, false);
                    Log::debug('DUE REMINDERS: Processing schedule', [
                        'schedule_id' => $schedule->scheduleid,
                        'loan_number' => $schedule->loannumber,
                        'due_date' => $schedule->paymentdate->format('Y-m-d'),
                        'days_until_due' => $daysUntilDue,
                        'amount' => $schedule->totalamount,
                        'paid' => $schedule->total_paid
                    ]);
                    
                    // Check if reminder was already sent today
                    $reminderSentToday = $this->wasReminderSentToday($schedule, 'due');
                    if ($reminderSentToday) {
                        Log::info('Due reminder already sent today for schedule', [
                            'schedule_id' => $schedule->scheduleid,
                            'loan_number' => $schedule->loannumber,
                            'due_date' => $schedule->paymentdate->format('Y-m-d')
                        ]);
                        continue;
                    }

                    // Get customer and loan details
                    $customer = $schedule->loan->customer ?? null;
                    $loan = $schedule->loan;
                    
                    if (!$customer || !$loan) {
                        Log::warning('Customer or loan not found for schedule', [
                            'schedule_id' => $schedule->scheduleid,
                            'loan_number' => $schedule->loannumber
                        ]);
                        continue;
                    }

                    // Calculate due amount (use accrued amounts if available)
                    $dueAmount = $schedule->balance > 0 ? $schedule->balance : $schedule->totalamount;
                    
                    // Prepare variables for templates
                    $variables = [
                        'Name' => $customer->first_name,
                        'LoanNo' => $schedule->loannumber,
                        'Amount' => number_format($dueAmount, 2),
                        'Date' => $schedule->paymentdate->format('d/m/Y'),
                        'InstallmentNo' => $schedule->installment_number,
                        'Days' => $daysUntilDue > 0 ? $daysUntilDue : 0
                    ];

                    // Send SMS reminder if enabled
                    $smsSent = $this->sendDueSms($customer, $variables, $dueEvent);
                    if ($smsSent) {
                        $results['due_sms_sent']++;
                    }

                    // Send email reminder if enabled
                    $emailSent = $this->sendDueEmail($customer, $schedule, $loan, $dueAmount, $daysUntilDue);
                    if ($emailSent) {
                        $results['due_emails_sent']++;
                    }

                    if ($smsSent || $emailSent) {
                        $results['due_reminders_sent']++;
                        
                        // Log the reminder in audit table
                        $this->logReminderSent($schedule, 'due', $dueAmount, $daysUntilDue);
                        
                        Log::info('✅ Due reminder sent successfully', [
                            'schedule_id' => $schedule->scheduleid,
                            'loan_number' => $schedule->loannumber,
                            'customer_id' => $customer->id,
                            'due_date' => $schedule->paymentdate->format('Y-m-d'),
                            'due_amount' => $dueAmount,
                            'days_until_due' => $daysUntilDue
                        ]);
                    } else {
                        Log::warning('Due reminder NOT sent (SMS and email both failed or disabled)', [
                            'schedule_id' => $schedule->scheduleid,
                            'loan_number' => $schedule->loannumber
                        ]);
                    }

                } catch (\Exception $e) {
                    Log::error('Error sending due reminder for schedule', [
                        'schedule_id' => $schedule->scheduleid,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }

        } catch (\Exception $e) {
            Log::error('Error processing due reminders', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }

        return $results;
    }

    /**
     * Process overdue payment alerts
     */
    private function processOverdueAlerts()
    {
        $results = [
            'overdue_alerts_sent' => 0,
            'overdue_sms_sent' => 0,
            'overdue_emails_sent' => 0
        ];

        try {
            // Get overdue frequency from notification_events
            $overdueEvent = DB::table('notification_events')
                ->where('event_key', 'payment_overdue')
                ->first();
            
            // CAST TO INTEGER
            $frequencyDays = $overdueEvent ? (int) $overdueEvent->overdue_frequency_days : 1; // Default to daily
            
            // Get overdue schedules (payment date is past and status is not paid)
            $overdueSchedules = PaymentSchedule::where('status', 'scheduled')
                ->where('paymentdate', '<', Carbon::today())
                ->where(function($query) {
                    // Only for schedules that have outstanding balance
                    $query->whereColumn('total_paid', '<', 'totalamount')
                          ->orWhere('total_paid', 0);
                })
                ->with(['loan.customer', 'loan' => function($q) {
                    $q->with('customer');
                }])
                ->get();
            
            Log::info('OVERDUE ALERTS: Found schedules', ['count' => $overdueSchedules->count()]);

            foreach ($overdueSchedules as $schedule) {
                try {
                    // Check if overdue alert should be sent based on frequency
                    $shouldSendAlert = $this->shouldSendOverdueAlert($schedule, $frequencyDays);
                    if (!$shouldSendAlert) {
                        continue;
                    }

                    // Get customer and loan details
                    $customer = $schedule->loan->customer ?? null;
                    $loan = $schedule->loan;
                    
                    if (!$customer || !$loan) {
                        Log::warning('Customer or loan not found for overdue schedule', [
                            'schedule_id' => $schedule->scheduleid
                        ]);
                        continue;
                    }

                    // Calculate overdue amount and days
                    $overdueAmount = $schedule->balance > 0 ? $schedule->balance : $schedule->totalamount;
                    $overdueDays = Carbon::today()->diffInDays($schedule->paymentdate);
                    
                    // Prepare variables for templates
                    $variables = [
                        'Name' => $customer->first_name,
                        'LoanNo' => $schedule->loannumber,
                        'Amount' => number_format($overdueAmount, 2),
                        'Date' => $schedule->paymentdate->format('d/m/Y'),
                        'Days' => $overdueDays,
                        'InstallmentNo' => $schedule->installment_number
                    ];

                    // Send SMS overdue alert if enabled
                    $smsSent = $this->sendOverdueSms($customer, $variables, $overdueEvent);
                    if ($smsSent) {
                        $results['overdue_sms_sent']++;
                    }

                    // Send email overdue alert if enabled
                    $emailSent = $this->sendOverdueEmail($customer, $schedule, $loan, $overdueAmount, $overdueDays);
                    if ($emailSent) {
                        $results['overdue_emails_sent']++;
                    }

                    if ($smsSent || $emailSent) {
                        $results['overdue_alerts_sent']++;
                        
                        // Log the overdue alert in audit table
                        $this->logReminderSent($schedule, 'overdue', $overdueAmount, $overdueDays);
                        
                        Log::info('✅ Overdue alert sent successfully', [
                            'schedule_id' => $schedule->scheduleid,
                            'loan_number' => $schedule->loannumber,
                            'customer_id' => $customer->id,
                            'overdue_days' => $overdueDays,
                            'overdue_amount' => $overdueAmount
                        ]);
                    }

                } catch (\Exception $e) {
                    Log::error('Error sending overdue alert for schedule', [
                        'schedule_id' => $schedule->scheduleid,
                        'error' => $e->getMessage()
                    ]);
                }
            }

        } catch (\Exception $e) {
            Log::error('Error processing overdue alerts', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }

        return $results;
    }

    /**
     * Send due SMS reminder
     */
    private function sendDueSms($customer, $variables, $event)
    {
        try {
            // Check if SMS is enabled for payment_due
            $smsEnabled = $this->isNotificationEnabled('payment_due', 'sms');
            
            if (!$smsEnabled) {
                Log::info('SMS disabled for payment_due notifications');
                return false;
            }

            if (empty($customer->phone)) {
                Log::warning('Customer phone not available for due reminder', [
                    'customer_id' => $customer->id,
                    'customer_name' => $customer->first_name
                ]);
                return false;
            }

            // Get SMS template
            $smsTemplate = $event ? $event->sms_template : null;
            if (!$smsTemplate) {
                $smsTemplate = 'Reminder: Your payment of ZMW [Amount] for loan #[LoanNo] is due on [Date].';
            }

            // Replace template variables
            $message = $this->replaceTemplateVariables($smsTemplate, $variables);
            
            Log::info('Sending due SMS', [
                'customer_id' => $customer->id,
                'phone' => $customer->phone,
                'message' => $message,
                'days_until_due' => $variables['Days'] ?? 'N/A'
            ]);

            // Send SMS
            $smsSent = $this->smsService->sendSms($customer->phone, $message);
            
            if ($smsSent) {
                Log::info('✅ Due SMS sent successfully', [
                    'customer_id' => $customer->id,
                    'phone' => $customer->phone,
                    'loan_no' => $variables['LoanNo'] ?? 'N/A'
                ]);
            } else {
                Log::warning('❌ Failed to send due SMS', [
                    'customer_id' => $customer->id,
                    'phone' => $customer->phone
                ]);
            }
            
            return $smsSent;

        } catch (\Exception $e) {
            Log::error('❌ Exception in sendDueSms', [
                'customer_id' => $customer->id ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send due email reminder
     */
    private function sendDueEmail($customer, $schedule, $loan, $dueAmount, $daysUntilDue)
    {
        try {
            // Check if email is enabled for payment_due
            $emailEnabled = $this->isNotificationEnabled('payment_due', 'email');
            
            if (!$emailEnabled) {
                Log::info('Email disabled for payment_due notifications');
                return false;
            }

            if (empty($customer->email)) {
                Log::warning('Customer email not available for due reminder', [
                    'customer_id' => $customer->id,
                    'customer_name' => $customer->first_name
                ]);
                return false;
            }

            // Send email with days until due parameter
            Mail::to($customer->email)->send(new PaymentDueReminder($customer, $schedule, $loan, $dueAmount, $daysUntilDue));
            
            Log::info('✅ Due email sent successfully', [
                'customer_id' => $customer->id,
                'customer_email' => $customer->email,
                'loan_number' => $schedule->loannumber,
                'days_until_due' => $daysUntilDue
            ]);
            
            return true;

        } catch (\Exception $e) {
            Log::error('❌ Error sending due email', [
                'customer_email' => $customer->email ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send overdue SMS alert
     */
    private function sendOverdueSms($customer, $variables, $event)
    {
        try {
            // Check if SMS is enabled for payment_overdue
            $smsEnabled = $this->isNotificationEnabled('payment_overdue', 'sms');
            
            if (!$smsEnabled) {
                Log::info('SMS disabled for payment_overdue notifications');
                return false;
            }

            if (empty($customer->phone)) {
                Log::warning('Customer phone not available for overdue alert', [
                    'customer_id' => $customer->id
                ]);
                return false;
            }

            // Get SMS template
            $smsTemplate = $event ? $event->sms_template : null;
            if (!$smsTemplate) {
                $smsTemplate = 'URGENT: Your loan payment for [LoanNo] is OVERDUE. Please pay ZMW [Amount] immediately.';
            }

            // Replace template variables
            $message = $this->replaceTemplateVariables($smsTemplate, $variables);
            
            Log::info('Sending overdue SMS', [
                'customer_id' => $customer->id,
                'phone' => $customer->phone,
                'message_length' => strlen($message),
                'overdue_days' => $variables['Days'] ?? 'N/A'
            ]);

            // Send SMS
            $smsSent = $this->smsService->sendSms($customer->phone, $message);
            
            if ($smsSent) {
                Log::info('✅ Overdue SMS sent successfully', [
                    'customer_id' => $customer->id,
                    'phone' => $customer->phone,
                    'loan_no' => $variables['LoanNo'] ?? 'N/A'
                ]);
            } else {
                Log::warning('❌ Failed to send overdue SMS', [
                    'customer_id' => $customer->id,
                    'phone' => $customer->phone
                ]);
            }
            
            return $smsSent;

        } catch (\Exception $e) {
            Log::error('❌ Exception in sendOverdueSms', [
                'customer_id' => $customer->id ?? 'unknown',
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send overdue email alert
     */
    private function sendOverdueEmail($customer, $schedule, $loan, $overdueAmount, $overdueDays)
    {
        try {
            // Check if email is enabled for payment_overdue
            $emailEnabled = $this->isNotificationEnabled('payment_overdue', 'email');
            
            if (!$emailEnabled) {
                Log::info('Email disabled for payment_overdue notifications');
                return false;
            }

            if (empty($customer->email)) {
                Log::warning('Customer email not available for overdue alert', [
                    'customer_id' => $customer->id
                ]);
                return false;
            }

            // Send email
            Mail::to($customer->email)->send(new PaymentOverdueReminder($customer, $schedule, $loan, $overdueAmount, $overdueDays));
            
            Log::info('✅ Overdue email sent successfully', [
                'customer_id' => $customer->id,
                'customer_email' => $customer->email,
                'loan_number' => $schedule->loannumber,
                'overdue_days' => $overdueDays
            ]);
            
            return true;

        } catch (\Exception $e) {
            Log::error('❌ Error sending overdue email', [
                'customer_email' => $customer->email ?? 'unknown',
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Check if notification is enabled for a specific channel
     */
    private function isNotificationEnabled($eventKey, $channel)
    {
        try {
            $setting = DB::table('notification_settings')
                ->where('event_key', $eventKey)
                ->where('channel', $channel)
                ->first();
                
            $isEnabled = $setting && $setting->is_enabled == 1;
            
            Log::debug('Notification setting check', [
                'event_key' => $eventKey,
                'channel' => $channel,
                'is_enabled' => $isEnabled
            ]);
                
            return $isEnabled;
        } catch (\Exception $e) {
            Log::error('❌ Failed to check notification setting', [
                'event_key' => $eventKey,
                'channel' => $channel,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Replace variables in template
     */
    private function replaceTemplateVariables($template, $variables)
    {
        if (empty($template)) {
            return '';
        }
        
        foreach ($variables as $key => $value) {
            $template = str_replace("[$key]", $value, $template);
        }
        return $template;
    }

    /**
     * Check if reminder was already sent today
     */
    private function wasReminderSentToday($schedule, $type)
    {
        try {
            $today = Carbon::today()->format('Y-m-d');
            
            $exists = DB::table('reminder_audit_log')
                ->where('schedule_id', $schedule->scheduleid)
                ->where('reminder_type', $type)
                ->whereDate('sent_at', $today)
                ->exists();
                
            return $exists;
        } catch (\Exception $e) {
            // If table doesn't exist, return false
            Log::warning('reminder_audit_log table may not exist or error', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Check if overdue alert should be sent based on frequency
     */
    private function shouldSendOverdueAlert($schedule, $frequencyDays)
    {
        try {
            $lastAlert = DB::table('reminder_audit_log')
                ->where('schedule_id', $schedule->scheduleid)
                ->where('reminder_type', 'overdue')
                ->orderBy('sent_at', 'desc')
                ->first();
                
            if (!$lastAlert) {
                // No alert sent yet, send first one
                return true;
            }
            
            $lastAlertDate = Carbon::parse($lastAlert->sent_at);
            $daysSinceLastAlert = Carbon::today()->diffInDays($lastAlertDate);
            
            return $daysSinceLastAlert >= $frequencyDays;
        } catch (\Exception $e) {
            // If table doesn't exist or error, send alert
            Log::warning('Error checking overdue alert frequency', [
                'error' => $e->getMessage()
            ]);
            return true;
        }
    }

    /**
     * Log reminder sent to audit table
     */
    private function logReminderSent($schedule, $type, $amount, $days = null)
    {
        try {
            DB::table('reminder_audit_log')->insert([
                'schedule_id' => $schedule->scheduleid,
                'loan_id' => $schedule->loanid,
                'loan_number' => $schedule->loannumber,
                'customer_id' => $schedule->loan->customerid ?? null,
                'reminder_type' => $type,
                'amount' => $amount,
                'due_date' => $schedule->paymentdate,
                'overdue_days' => $days, // Using your column name
                'sent_at' => now(),
                'created_at' => now(),
                'updated_at' => now()
            ]);
            
            Log::debug('Reminder logged to audit table', [
                'schedule_id' => $schedule->scheduleid,
                'type' => $type,
                'days' => $days
            ]);
        } catch (\Exception $e) {
            Log::warning('Failed to log reminder audit', [
                'error' => $e->getMessage(),
                'schedule_id' => $schedule->scheduleid ?? 'unknown'
            ]);
        }
    }

    /**
     * Debug method to check what payments are due within reminder window
     */
    public function debugDuePayments()
    {
        try {
            // Get reminder days from notification_events
            $dueEvent = DB::table('notification_events')
                ->where('event_key', 'payment_due')
                ->first();
            
            $reminderWindowDays = $dueEvent ? (int) $dueEvent->reminder_days : 0;
            
            $today = Carbon::today();
            $windowEndDate = $today->copy()->addDays($reminderWindowDays);
            
            Log::info('DEBUG: Checking due payments', [
                'today' => $today->format('Y-m-d'),
                'reminder_days_from_db' => $reminderWindowDays,
                'window_end' => $windowEndDate->format('Y-m-d')
            ]);
            
            // First, check the notification_events table
            $notificationEvent = DB::table('notification_events')
                ->where('event_key', 'payment_due')
                ->first();
                
            Log::info('DEBUG: Notification event from DB', [
                'event_key' => $notificationEvent->event_key ?? 'not found',
                'reminder_days' => $notificationEvent->reminder_days ?? 'not found',
                'reminder_days_type' => gettype($notificationEvent->reminder_days ?? 'null'),
                'sms_template' => $notificationEvent->sms_template ?? 'not found'
            ]);
            
            // Check notification settings
            $smsSetting = DB::table('notification_settings')
                ->where('event_key', 'payment_due')
                ->where('channel', 'sms')
                ->first();
                
            $emailSetting = DB::table('notification_settings')
                ->where('event_key', 'payment_due')
                ->where('channel', 'email')
                ->first();
                
            Log::info('DEBUG: Notification settings', [
                'sms_enabled' => $smsSetting ? ($smsSetting->is_enabled ? 'YES' : 'NO') : 'not found',
                'email_enabled' => $emailSetting ? ($emailSetting->is_enabled ? 'YES' : 'NO') : 'not found'
            ]);
            
            // Get schedules
            $schedules = PaymentSchedule::where('status', 'scheduled')
                ->whereBetween('paymentdate', [$today, $windowEndDate])
                ->where('paymentdate', '>', $today)
                ->where(function($query) {
                    $query->whereColumn('total_paid', '<', 'totalamount')
                          ->orWhere('total_paid', 0);
                })
                ->with(['loan.customer'])
                ->get();
            
            $result = [];
            foreach ($schedules as $schedule) {
                $daysUntilDue = $today->diffInDays($schedule->paymentdate, false);
                $result[] = [
                    'schedule_id' => $schedule->scheduleid,
                    'loan_number' => $schedule->loannumber,
                    'due_date' => $schedule->paymentdate->format('Y-m-d'),
                    'days_until_due' => $daysUntilDue,
                    'amount_due' => $schedule->totalamount - ($schedule->total_paid ?? 0),
                    'total_amount' => $schedule->totalamount,
                    'total_paid' => $schedule->total_paid ?? 0,
                    'customer_name' => $schedule->loan->customer->first_name . ' ' . $schedule->loan->customer->surname ?? 'N/A',
                    'customer_phone' => $schedule->loan->customer->phone ?? 'N/A',
                    'customer_email' => $schedule->loan->customer->email ?? 'N/A',
                    'in_reminder_window' => $daysUntilDue <= $reminderWindowDays ? 'YES' : 'NO'
                ];
            }
            
            return [
                'today' => $today->format('Y-m-d'),
                'reminder_days_from_db' => $reminderWindowDays,
                'window_end_date' => $windowEndDate->format('Y-m-d'),
                'total_schedules_in_window' => count($result),
                'schedules' => $result,
                'notification_event' => [
                    'exists' => $notificationEvent ? 'YES' : 'NO',
                    'reminder_days' => $notificationEvent->reminder_days ?? 'N/A',
                    'reminder_days_type' => gettype($notificationEvent->reminder_days ?? 'null')
                ],
                'notification_settings' => [
                    'sms_enabled' => $smsSetting ? ($smsSetting->is_enabled ? 'YES' : 'NO') : 'NO',
                    'email_enabled' => $emailSetting ? ($emailSetting->is_enabled ? 'YES' : 'NO') : 'NO'
                ]
            ];
            
        } catch (\Exception $e) {
            return [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ];
        }
    }

    /**
     * Test the complete reminder process
     */
    public function testReminderProcess()
    {
        Log::info('=== TESTING REMINDER PROCESS ===');
        
        $results = [
            'due_reminders_test' => [],
            'overdue_test' => []
        ];

        try {
            // Test due reminders
            $dueEvent = DB::table('notification_events')
                ->where('event_key', 'payment_due')
                ->first();
            
            $results['due_reminders_test']['reminder_days'] = $dueEvent ? $dueEvent->reminder_days : 'NOT FOUND';
            $results['due_reminders_test']['reminder_days_type'] = $dueEvent ? gettype($dueEvent->reminder_days) : 'NOT FOUND';
            $results['due_reminders_test']['sms_template'] = $dueEvent ? $dueEvent->sms_template : 'NOT FOUND';
            
            // Test overdue
            $overdueEvent = DB::table('notification_events')
                ->where('event_key', 'payment_overdue')
                ->first();
            
            $results['overdue_test']['frequency_days'] = $overdueEvent ? $overdueEvent->overdue_frequency_days : 'NOT FOUND';
            $results['overdue_test']['frequency_days_type'] = $overdueEvent ? gettype($overdueEvent->overdue_frequency_days) : 'NOT FOUND';
            $results['overdue_test']['sms_template'] = $overdueEvent ? $overdueEvent->sms_template : 'NOT FOUND';
            
            // Check settings
            $results['settings'] = [
                'payment_due_sms' => $this->isNotificationEnabled('payment_due', 'sms') ? 'ENABLED' : 'DISABLED',
                'payment_due_email' => $this->isNotificationEnabled('payment_due', 'email') ? 'ENABLED' : 'DISABLED',
                'payment_overdue_sms' => $this->isNotificationEnabled('payment_overdue', 'sms') ? 'ENABLED' : 'DISABLED',
                'payment_overdue_email' => $this->isNotificationEnabled('payment_overdue', 'email') ? 'ENABLED' : 'DISABLED'
            ];
            
        } catch (\Exception $e) {
            $results['error'] = $e->getMessage();
        }
        
        Log::info('Test results', $results);
        return $results;
    }

    /**
     * Get summary of upcoming due payments
     */
    public function getDueRemindersSummary($daysAhead = 7)
    {
        try {
            $startDate = Carbon::today();
            $endDate = Carbon::today()->addDays($daysAhead);
            
            $dueSchedules = PaymentSchedule::where('status', 'scheduled')
                ->whereBetween('paymentdate', [$startDate, $endDate])
                ->where(function($query) {
                    $query->whereColumn('total_paid', '<', 'totalamount')
                          ->orWhere('total_paid', 0);
                })
                ->with(['loan.customer'])
                ->select([
                    'scheduleid', 'loannumber', 'loanid', 'paymentdate', 
                    'totalamount', 'total_paid', 'installment_number',
                    DB::raw('(totalamount - COALESCE(total_paid, 0)) as due_amount')
                ])
                ->orderBy('paymentdate')
                ->get();
            
            $summary = [
                'total_due' => $dueSchedules->count(),
                'total_amount_due' => $dueSchedules->sum('due_amount'),
                'schedule_count_by_day' => [],
                'schedules' => $dueSchedules->map(function($schedule) {
                    return [
                        'schedule_id' => $schedule->scheduleid,
                        'loan_number' => $schedule->loannumber,
                        'customer_name' => $schedule->loan->customer->first_name . ' ' . $schedule->loan->customer->surname,
                        'customer_phone' => $schedule->loan->customer->phone,
                        'customer_email' => $schedule->loan->customer->email,
                        'due_date' => $schedule->paymentdate->format('Y-m-d'),
                        'due_amount' => $schedule->due_amount,
                        'installment_number' => $schedule->installment_number,
                        'days_until_due' => Carbon::today()->diffInDays($schedule->paymentdate, false)
                    ];
                })
            ];
            
            // Group by day
            foreach ($dueSchedules as $schedule) {
                $date = $schedule->paymentdate->format('Y-m-d');
                if (!isset($summary['schedule_count_by_day'][$date])) {
                    $summary['schedule_count_by_day'][$date] = 0;
                }
                $summary['schedule_count_by_day'][$date]++;
            }
            
            return $summary;
            
        } catch (\Exception $e) {
            Log::error('Error getting due reminders summary', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Get summary of overdue payments
     */
    public function getOverdueSummary()
    {
        try {
            $overdueSchedules = PaymentSchedule::where('status', 'scheduled')
                ->where('paymentdate', '<', Carbon::today())
                ->where(function($query) {
                    $query->whereColumn('total_paid', '<', 'totalamount')
                          ->orWhere('total_paid', 0);
                })
                ->with(['loan.customer'])
                ->select([
                    'scheduleid', 'loannumber', 'loanid', 'paymentdate', 
                    'totalamount', 'total_paid', 'installment_number',
                    DB::raw('(totalamount - COALESCE(total_paid, 0)) as overdue_amount'),
                    DB::raw('DATEDIFF(CURDATE(), paymentdate) as days_overdue')
                ])
                ->orderBy('paymentdate')
                ->get();
            
            $summary = [
                'total_overdue' => $overdueSchedules->count(),
                'total_amount_overdue' => $overdueSchedules->sum('overdue_amount'),
                'overdue_by_days' => [
                    '1-7' => 0,
                    '8-30' => 0,
                    '31-90' => 0,
                    '90+' => 0
                ],
                'schedules' => $overdueSchedules->map(function($schedule) {
                    return [
                        'schedule_id' => $schedule->scheduleid,
                        'loan_number' => $schedule->loannumber,
                        'customer_name' => $schedule->loan->customer->first_name . ' ' . $schedule->loan->customer->surname,
                        'customer_phone' => $schedule->loan->customer->phone,
                        'customer_email' => $schedule->loan->customer->email,
                        'due_date' => $schedule->paymentdate->format('Y-m-d'),
                        'overdue_amount' => $schedule->overdue_amount,
                        'days_overdue' => $schedule->days_overdue,
                        'installment_number' => $schedule->installment_number
                    ];
                })
            ];
            
            // Categorize by days overdue
            foreach ($overdueSchedules as $schedule) {
                $days = $schedule->days_overdue;
                if ($days <= 7) {
                    $summary['overdue_by_days']['1-7']++;
                } elseif ($days <= 30) {
                    $summary['overdue_by_days']['8-30']++;
                } elseif ($days <= 90) {
                    $summary['overdue_by_days']['31-90']++;
                } else {
                    $summary['overdue_by_days']['90+']++;
                }
            }
            
            return $summary;
            
        } catch (\Exception $e) {
            Log::error('Error getting overdue summary', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
}