<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vendors | {{ config('app.name', 'Laravel') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#146CAA',
                        primaryLight: '#E1F0FA',
                        secondary: '#6b7280',
                        success: '#107C10',
                        warning: '#F68C1E',
                        danger: '#D83B01',
                        info: '#0078D4',
                        gray50: '#FAF9F8',
                        gray100: '#F3F2F1',
                        gray200: '#EDEBE9',
                        gray300: '#E1DFDD',
                        gray400: '#C8C6C4',
                        gray500: '#A19F9D',
                        gray600: '#605E5C',
                        gray700: '#323130',
                        gray800: '#201F1E',
                        gray900: '#000000'
                    }
                }
            }
        }
    </script>
    <style>
        .quickbooks-table {
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
        }
        .quickbooks-table tbody tr:hover {
            background-color: #F3F2F1;
        }
        .quickbooks-table tbody tr.selected {
            background-color: #E1F0FA;
        }
        .quickbooks-input {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
        }
        .quickbooks-input:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-checkbox {
            width: 18px;
            height: 18px;
            border: 2px solid #605E5C;
            border-radius: 3px;
        }
        .quickbooks-checkbox:checked {
            background-color: #146CAA;
            border-color: #146CAA;
        }
        .quickbooks-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .quickbooks-pagination-btn {
            padding: 6px 12px;
            border: 1px solid #EDEBE9;
            background: white;
            color: #323130;
            font-size: 14px;
            cursor: pointer;
        }
        .quickbooks-pagination-btn:hover:not(:disabled) {
            background-color: #F3F2F1;
        }
        .quickbooks-pagination-btn:disabled {
            color: #A19F9D;
            cursor: not-allowed;
        }
        .quickbooks-pagination-btn.active {
            background-color: #146CAA;
            color: white;
            border-color: #146CAA;
        }
        .quickbooks-dropdown {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            background: white;
            padding: 6px 12px;
            font-size: 14px;
        }
        .quickbooks-search {
            background-color: white;
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px 6px 36px;
            font-size: 14px;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='16' height='16' viewBox='0 0 24 24' fill='none' stroke='%23605E5C' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3E%3Ccircle cx='11' cy='11' r='8'%3E%3C/circle%3E%3Cpath d='m21 21-4.35-4.35'%3E%3C/path%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: 12px center;
        }
        .quickbooks-search:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .action-btn {
            width: 32px;
            height: 32px;
            border-radius: 4px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            margin: 0 2px;
            transition: all 0.2s;
        }
        .action-btn:hover {
            transform: translateY(-1px);
        }
        .action-btn-edit {
            background-color: #E1F0FA;
            color: #146CAA;
        }
        .action-btn-edit:hover {
            background-color: #146CAA;
            color: white;
        }
        .action-btn-delete {
            background-color: #FEE;
            color: #D83B01;
        }
        .action-btn-delete:hover {
            background-color: #D83B01;
            color: white;
        }
        .action-btn-view {
            background-color: #F3F2F1;
            color: #323130;
        }
        .action-btn-view:hover {
            background-color: #146CAA;
            color: white;
        }
        .action-btn-statement {
            background-color: #E6F4EA;
            color: #107C10;
        }
        .action-btn-statement:hover {
            background-color: #107C10;
            color: white;
        }
        .modal-backdrop {
            background-color: rgba(0, 0, 0, 0.5);
        }
        .modal-content {
            max-height: 90vh;
            overflow-y: auto;
        }
        .modal-section {
            border-top: 1px solid #EDEBE9;
            padding-top: 24px;
            margin-top: 24px;
        }
        .modal-divider {
            border-top: 1px solid #EDEBE9;
            margin: 20px 0;
        }
        .modal-info {
            background-color: #F3F2F1;
            border: 1px solid #EDEBE9;
            border-radius: 4px;
            padding: 12px;
            margin-top: 8px;
            font-size: 13px;
            color: #605E5C;
        }
        .modal-info i {
            color: #605E5C;
            margin-right: 8px;
        }
        .address-field {
            background-color: #FAF9F8;
            border: 1px solid #EDEBE9;
            border-radius: 4px;
            padding: 12px;
            margin-bottom: 12px;
        }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Vendors', 'subtitle' => 'Manage your vendors'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Stats Cards -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-white rounded-lg border border-gray200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-primaryLight rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-users text-primary text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray600">Total vendors</p>
                            <p class="text-xl font-semibold text-gray900">{{ $stats['total'] ?? 0 }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-check-circle text-success text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray600">Active</p>
                            <p class="text-xl font-semibold text-gray900">{{ $stats['active'] ?? 0 }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-yellow-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-clock text-warning text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray600">Overdue</p>
                            <p class="text-xl font-semibold text-gray900">{{ $stats['overdue'] ?? 0 }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-red-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-exclamation-triangle text-danger text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray600">1099 Vendors</p>
                            <p class="text-xl font-semibold text-gray900">{{ $stats['vendor_1099'] ?? 0 }}</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Search and Action Bar -->
            <div class="bg-white rounded-lg border border-gray200 p-4 mb-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between space-y-4 md:space-y-0">
                    <div class="flex items-center space-x-4">
                        <!-- New Vendor Button (on left) -->
                        <button onclick="openNewVendorModal()" 
                                class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center">
                            <i class="fas fa-plus mr-2"></i>New Vendor
                        </button>
                        
                        <!-- Search -->
                        <div class="relative">
                            <input type="text" 
                                   name="search"
                                   id="searchInput"
                                   class="quickbooks-search w-64"
                                   placeholder="Search vendors"
                                   value="{{ request('search') }}"
                                   onkeypress="handleSearchKeyPress(event)">
                        </div>
                    </div>
                    
                    <!-- Filter and Bulk Action Buttons -->
                    <div class="flex items-center space-x-4">
                        <!-- Filters -->
                        <div class="flex items-center space-x-2">
                            <!-- Status Filter -->
                            <select id="statusFilter" class="quickbooks-dropdown">
                                <option value="">All status</option>
                                <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                                <option value="inactive" {{ request('status') == 'inactive' ? 'selected' : '' }}>Inactive</option>
                            </select>
                            
                            <!-- Balance Filter -->
                            <select id="balanceFilter" class="quickbooks-dropdown">
                                <option value="">All balances</option>
                                <option value="with_balance" {{ request('balance') == 'with_balance' ? 'selected' : '' }}>With balance</option>
                                <option value="no_balance" {{ request('balance') == 'no_balance' ? 'selected' : '' }}>No balance</option>
                            </select>
                            
                            <!-- 1099 Filter -->
                            <select id="vendor1099Filter" class="quickbooks-dropdown">
                                <option value="">All 1099 status</option>
                                <option value="1099" {{ request('vendor_1099') == '1099' ? 'selected' : '' }}>1099 Eligible</option>
                                <option value="non_1099" {{ request('vendor_1099') == 'non_1099' ? 'selected' : '' }}>Non-1099</option>
                            </select>
                        </div>
                        
                        <!-- Bulk Action Buttons -->
                        <div class="flex items-center space-x-2">
                            <button onclick="exportSelected()" 
                                    id="exportBtn"
                                    class="border border-gray300 text-gray700 px-3 py-2 rounded text-sm font-medium hover:bg-gray50 flex items-center disabled:opacity-50 disabled:cursor-not-allowed"
                                    disabled>
                                <i class="fas fa-download mr-2"></i>Export
                            </button>
                            <button onclick="emailStatements()" 
                                    id="emailBtn"
                                    class="border border-gray300 text-gray700 px-3 py-2 rounded text-sm font-medium hover:bg-gray50 flex items-center disabled:opacity-50 disabled:cursor-not-allowed"
                                    disabled>
                                <i class="fas fa-envelope mr-2"></i>Email
                            </button>
                            <button onclick="deleteSelected()" 
                                    id="deleteBtn"
                                    class="border border-red-300 text-red-700 px-3 py-2 rounded text-sm font-medium hover:bg-red-50 flex items-center disabled:opacity-50 disabled:cursor-not-allowed"
                                    disabled>
                                <i class="fas fa-trash mr-2"></i>Delete
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Vendors Table -->
            <div class="bg-white rounded-lg border border-gray200 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="quickbooks-table">
                        <thead>
                            <tr>
                                <th class="w-12">
                                    <input type="checkbox" 
                                           id="selectAll" 
                                           class="quickbooks-checkbox">
                                </th>
                                <th>Vendor</th>
                                <th>Company</th>
                                <th>Contact</th>
                                <th>Account</th>
                                <th>Currency</th>
                                <th>Balance</th>
                                <th>Payment Terms</th>
                                <th class="w-48">Actions</th>
                            </tr>
                        </thead>
                        <tbody id="vendorsTableBody">
                            @forelse($vendors as $vendor)
                                <tr class="hover:bg-gray100 transition-colors" id="vendor-row-{{ $vendor->id }}">
                                    <td>
                                        <input type="checkbox" 
                                               class="vendor-checkbox quickbooks-checkbox" 
                                               data-id="{{ $vendor->id }}">
                                    </td>
                                    <td>
                                        <div class="flex items-center">
                                            <div class="w-8 h-8 {{ $vendor->active ? 'bg-primaryLight' : 'bg-gray200' }} rounded flex items-center justify-center mr-3">
                                                <span class="{{ $vendor->active ? 'text-primary' : 'text-gray600' }} font-semibold text-sm">
                                                    {{ strtoupper(substr($vendor->display_name, 0, 1)) }}
                                                </span>
                                            </div>
                                            <div>
                                                <div class="font-medium text-gray900">{{ $vendor->display_name }}</div>
                                                @if($vendor->vendor_1099)
                                                    <span class="quickbooks-badge bg-blue-100 text-blue-800 text-xs mt-1">1099</span>
                                                @endif
                                                @if($vendor->full_name && $vendor->full_name != $vendor->display_name)
                                                    <div class="text-xs text-gray500">{{ $vendor->full_name }}</div>
                                                @endif
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="text-gray700">{{ $vendor->company_name ?: '—' }}</div>
                                        @if($vendor->account_number)
                                            <div class="text-xs text-gray500">#{{ $vendor->account_number }}</div>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="text-gray700">{{ $vendor->email ?: '—' }}</div>
                                        <div class="text-sm text-gray500">{{ $vendor->mobile_phone ?: '—' }}</div>
                                    </td>
                                    <td>
                                        <div class="text-gray700">{{ $vendor->account_name ?: '—' }}</div>
                                    </td>
                                    <td>
                                        @if($vendor->currencyDetail)
                                            <div class="flex items-center">
                                                <span class="quickbooks-badge bg-blue-100 text-blue-800 mr-2">
                                                    {{ $vendor->currencyDetail->code }}
                                                </span>
                                                <span class="text-sm text-gray600">{{ $vendor->currencyDetail->symbol }}</span>
                                            </div>
                                        @else
                                            <span class="quickbooks-badge bg-gray-100 text-gray-800">
                                                {{ $vendor->currency ?? 'USD' }}
                                            </span>
                                        @endif
                                    </td>
                                    <td>
                                        @if($vendor->balance > 0)
                                            <div class="font-medium {{ $vendor->overdue ? 'text-danger' : 'text-gray900' }}">
                                                {{ $vendor->currency_symbol }}{{ number_format($vendor->balance, 2) }}
                                            </div>
                                            @if($vendor->overdue)
                                                <div class="text-xs text-danger font-medium">Overdue</div>
                                            @else
                                                <div class="text-xs text-gray500">Due</div>
                                            @endif
                                        @else
                                            <div class="text-gray500">{{ $vendor->currency_symbol }}0.00</div>
                                        @endif
                                    </td>
                                    <td>
                                        <span class="text-sm {{ $vendor->payment_terms == 'Due on receipt' ? 'text-warning' : 'text-gray700' }}">
                                            {{ $vendor->payment_terms ?: '—' }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="flex items-center">
                                            <a href="{{ route('admin.accounting.expenses.vendors.show', $vendor->id) }}" 
                                               class="action-btn action-btn-view"
                                               title="View details">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <a href="{{ route('admin.accounting.expenses.vendors.edit', $vendor->id) }}" 
                                               class="action-btn action-btn-edit"
                                               title="Edit vendor">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <button onclick="generateStatement({{ $vendor->id }})" 
                                                    class="action-btn action-btn-statement"
                                                    title="Generate statement">
                                                <i class="fas fa-file-invoice-dollar"></i>
                                            </button>
                                            <button onclick="openDeleteModal({{ $vendor->id }})" 
                                                    class="action-btn action-btn-delete"
                                                    title="Delete vendor">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="9" class="text-center py-8">
                                        <div id="emptyState" class="py-16 text-center">
                                            <div class="w-16 h-16 bg-gray100 rounded-full flex items-center justify-center mx-auto mb-4">
                                                <i class="fas fa-users text-gray500 text-2xl"></i>
                                            </div>
                                            <h3 class="text-lg font-medium text-gray900 mb-2">No vendors found</h3>
                                            <p class="text-gray600 mb-6 max-w-md mx-auto">Get started by adding your first vendor.</p>
                                            <div class="space-x-3">
                                                <button onclick="openNewVendorModal()" 
                                                        class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90">
                                                    <i class="fas fa-plus mr-2"></i>Add vendor
                                                </button>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                @if($vendors->hasPages())
                    <div class="border-t border-gray200 p-4 flex items-center justify-between">
                        <div class="text-sm text-gray600">
                            Showing {{ $vendors->firstItem() }} to {{ $vendors->lastItem() }} of {{ $vendors->total() }} vendors
                        </div>
                        <div class="flex items-center space-x-1">
                            {{ $vendors->appends(request()->query())->links('vendor.pagination.tailwind') }}
                        </div>
                    </div>
                @endif
            </div>
        </main>
    </div>

    <!-- New Vendor Modal -->
    <div id="newVendorModal" class="fixed inset-0 bg-gray900 bg-opacity-50 hidden items-center justify-center z-50 p-4 overflow-y-auto">
        <div class="bg-white rounded-lg shadow-lg max-w-4xl w-full modal-content">
            <!-- Modal Header -->
            <div class="border-b border-gray200 p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <h2 class="text-xl font-semibold text-gray900">New vendor</h2>
                        <p class="text-gray600 text-sm">Add a new vendor to QuickBooks</p>
                    </div>
                    <button onclick="closeNewVendorModal()" class="text-gray500 hover:text-gray700">
                        <i class="fas fa-times text-lg"></i>
                    </button>
                </div>
            </div>

            <!-- Modal Body -->
            <div class="p-6">
                <form id="newVendorForm" action="{{ route('admin.accounting.expenses.vendors.store') }}" method="POST">
                    @csrf
                    <!-- Name and Contact Section -->
                    <div>
                        <h3 class="text-lg font-semibold text-gray900 mb-6">Name and contact</h3>
                        
                        <!-- Company Name -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-2">Company name</label>
                            <div class="flex items-center space-x-4">
                                <div class="flex-1">
                                    <input type="text" 
                                           id="companyName"
                                           name="company_name"
                                           class="quickbooks-input w-full" 
                                           placeholder="Company name">
                                </div>
                            </div>
                        </div>

                        <!-- Vendor Display Name -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-2">
                                Vendor display name
                                <span class="text-danger">*</span>
                            </label>
                            <input type="text" 
                                   id="displayName"
                                   name="display_name"
                                   class="quickbooks-input w-full" 
                                   placeholder="Vendor display name"
                                   required>
                        </div>

                        <!-- Personal Name Fields -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-4">Personal name</label>
                            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                                <div>
                                    <label class="text-sm text-gray600 block mb-1">First name</label>
                                    <input type="text" id="firstName" name="first_name" class="quickbooks-input w-full">
                                </div>
                                <div>
                                    <label class="text-sm text-gray600 block mb-1">Middle name</label>
                                    <input type="text" id="middleName" name="middle_name" class="quickbooks-input w-full">
                                </div>
                                <div>
                                    <label class="text-sm text-gray600 block mb-1">Last name</label>
                                    <input type="text" id="lastName" name="last_name" class="quickbooks-input w-full">
                                </div>
                                <div>
                                    <label class="text-sm text-gray600 block mb-1">Suffix</label>
                                    <input type="text" id="suffix" name="suffix" class="quickbooks-input w-full" placeholder="Jr, Sr, III, etc.">
                                </div>
                            </div>
                        </div>

                        <!-- Contact Information -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-4">Contact information</label>
                            
                            <!-- Email Section -->
                            <div class="mb-6">
                                <div class="flex items-center mb-2">
                                    <label class="text-sm text-gray600">Email</label>
                                </div>
                                <div class="relative">
                                    <input type="email" 
                                           id="email"
                                           name="email"
                                           class="quickbooks-input w-full pl-10" 
                                           placeholder="email@example.com">
                                    <i class="fas fa-envelope absolute left-3 top-1/2 transform -translate-y-1/2 text-gray400"></i>
                                </div>
                            </div>

                            <!-- Phone Section -->
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label class="text-sm text-gray600 block mb-2">Mobile number</label>
                                    <div class="relative">
                                        <input type="tel" 
                                               id="mobilePhone"
                                               name="mobile_phone"
                                               class="quickbooks-input w-full pl-10" 
                                               placeholder="(555) 123-4567">
                                        <i class="fas fa-mobile-alt absolute left-3 top-1/2 transform -translate-y-1/2 text-gray400"></i>
                                    </div>
                                </div>
                                <div>
                                    <label class="text-sm text-gray600 block mb-2">Other</label>
                                    <div class="relative">
                                        <input type="tel" 
                                               id="otherPhone"
                                               name="other_phone"
                                               class="quickbooks-input w-full pl-10" 
                                               placeholder="(555) 123-4567">
                                        <i class="fas fa-phone absolute left-3 top-1/2 transform -translate-y-1/2 text-gray400"></i>
                                    </div>
                                </div>
                            </div>

                            <!-- Additional Contact Info -->
                            <div class="mt-6">
                                <label class="text-sm text-gray600 block mb-2">Fax</label>
                                <input type="text" id="fax" name="fax" class="quickbooks-input w-full" placeholder="(555) 123-4567">
                            </div>
                            <div class="mt-4">
                                <label class="text-sm text-gray600 block mb-2">Website</label>
                                <input type="url" id="website" name="website" class="quickbooks-input w-full" placeholder="https://example.com">
                            </div>
                            <div class="mt-4">
                                <label class="text-sm text-gray600 block mb-2">Name to print on checks</label>
                                <input type="text" id="checkName" name="check_printing_name" class="quickbooks-input w-full" placeholder="Check printing name">
                            </div>
                        </div>
                    </div>

                    <div class="modal-divider"></div>

                    <!-- Address Section -->
                    <div>
                        <h3 class="text-lg font-semibold text-gray900 mb-6">Address</h3>
                        
                        <!-- Billing Address -->
                        <div class="mb-8">
                            <div class="flex items-center justify-between mb-4">
                                <label class="text-sm font-medium text-gray700">Billing address</label>
                                <button type="button" onclick="copyShippingToBilling()" class="text-primary hover:text-primary/90 text-sm font-medium">
                                    Copy from shipping
                                </button>
                            </div>
                            
                            <div class="address-field">
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">Street address 1</label>
                                        <input type="text" id="billingStreet1" name="billing_street1" class="quickbooks-input w-full">
                                    </div>
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">Street address 2</label>
                                        <input type="text" id="billingStreet2" name="billing_street2" class="quickbooks-input w-full">
                                    </div>
                                </div>
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">City</label>
                                        <input type="text" id="billingCity" name="billing_city" class="quickbooks-input w-full">
                                    </div>
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">State/Province</label>
                                        <input type="text" id="billingState" name="billing_state" class="quickbooks-input w-full">
                                    </div>
                                </div>
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">ZIP/Postal code</label>
                                        <input type="text" id="billingZip" name="billing_zip" class="quickbooks-input w-full">
                                    </div>
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">Country</label>
                                        <select id="billingCountry" name="billing_country" class="quickbooks-input w-full">
                                            <option value="">Select country</option>
                                            <option value="US">United States</option>
                                            <option value="CA">Canada</option>
                                            <option value="UK">United Kingdom</option>
                                            <option value="AU">Australia</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Shipping Address -->
                        <div class="mb-8">
                            <div class="flex items-center justify-between mb-4">
                                <label class="text-sm font-medium text-gray700">Shipping address</label>
                                <button type="button" onclick="copyBillingToShipping()" class="text-primary hover:text-primary/90 text-sm font-medium">
                                    Copy from billing
                                </button>
                            </div>
                            
                            <div class="flex items-center mb-4">
                                <input type="checkbox" id="sameAsBilling" class="quickbooks-checkbox mr-2">
                                <label for="sameAsBilling" class="text-sm text-gray600">Same as billing address</label>
                            </div>
                            
                            <div id="shippingAddressFields" class="address-field">
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">Street address 1</label>
                                        <input type="text" id="shippingStreet1" name="shipping_street1" class="quickbooks-input w-full">
                                    </div>
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">Street address 2</label>
                                        <input type="text" id="shippingStreet2" name="shipping_street2" class="quickbooks-input w-full">
                                    </div>
                                </div>
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">City</label>
                                        <input type="text" id="shippingCity" name="shipping_city" class="quickbooks-input w-full">
                                    </div>
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">State/Province</label>
                                        <input type="text" id="shippingState" name="shipping_state" class="quickbooks-input w-full">
                                    </div>
                                </div>
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">ZIP/Postal code</label>
                                        <input type="text" id="shippingZip" name="shipping_zip" class="quickbooks-input w-full">
                                    </div>
                                    <div>
                                        <label class="text-sm text-gray600 block mb-1">Country</label>
                                        <select id="shippingCountry" name="shipping_country" class="quickbooks-input w-full">
                                            <option value="">Select country</option>
                                            <option value="US">United States</option>
                                            <option value="CA">Canada</option>
                                            <option value="UK">United Kingdom</option>
                                            <option value="AU">Australia</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="modal-divider"></div>

                    <!-- Additional Information Section -->
                    <div>
                        <h3 class="text-lg font-semibold text-gray900 mb-6">Additional information</h3>
                        
                        <!-- Account Dropdown -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-2">Account</label>
                            <p class="text-sm text-gray600 mb-3">Link vendor to a chart of account</p>
                            <select id="chartOfAccountId" name="chart_of_account_id" class="quickbooks-input w-full">
                                <option value="">Select an account</option>
                                @php
                                    $accounts = \App\Models\ChartOfAccount::where('is_active', true)
                                        ->orderBy('code')
                                        ->get(['id', 'code', 'name']);
                                @endphp
                                @foreach($accounts as $account)
                                    <option value="{{ $account->id }}">
                                        {{ $account->code }} - {{ $account->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>

                        <!-- Account Number (optional) -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-2">Account Number (Optional)</label>
                            <p class="text-sm text-gray600 mb-3">Vendor's account number with your company</p>
                            <input type="text" id="accountNumber" name="account_number" class="quickbooks-input w-full" placeholder="Account number">
                        </div>

                        <!-- Payment Terms -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-2">Payment terms</label>
                            <select id="paymentTerms" name="payment_terms" class="quickbooks-input w-full">
                                <option value="">Select payment terms</option>
                                <option value="net_15">Net 15</option>
                                <option value="net_30">Net 30</option>
                                <option value="net_45">Net 45</option>
                                <option value="net_60">Net 60</option>
                                <option value="due_on_receipt">Due on receipt</option>
                            </select>
                        </div>

                        <!-- Tax Information -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-2">Tax information</label>
                            
                            <div class="flex items-center mb-4">
                                <input type="checkbox" id="vendor1099" name="vendor_1099" value="1" class="quickbooks-checkbox mr-2">
                                <label for="vendor1099" class="text-sm text-gray600">This vendor is eligible for 1099</label>
                            </div>
                            
                            <div id="taxFields">
                                <label class="text-sm text-gray600 block mb-2">Tax ID (SSN/EIN)</label>
                                <input type="text" id="taxId" name="tax_id" class="quickbooks-input w-full" placeholder="Tax identification number">
                            </div>
                        </div>

                        <!-- Currency Dropdown -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-2">Currency</label>
                            <select id="currency_id" name="currency_id" class="quickbooks-input w-full" required>
                                <option value="">Select currency</option>
                                @php
                                    $currencies = \App\Models\Currency::where('is_active', true)
                                        ->orderBy('code')
                                        ->get(['id', 'code', 'name', 'symbol']);
                                @endphp
                                @foreach($currencies as $currency)
                                    <option value="{{ $currency->id }}">
                                        {{ $currency->code }} - {{ $currency->name }} ({{ $currency->symbol }})
                                    </option>
                                @endforeach
                            </select>
                        </div>

                        <!-- Vendor Status -->
                        <div class="mb-8">
                            <label class="block text-sm font-medium text-gray700 mb-4">Vendor status</label>
                            <div class="flex items-center space-x-6">
                                <div class="flex items-center">
                                    <input type="radio" id="active" name="active" value="1" checked class="mr-2">
                                    <label for="active" class="text-sm text-gray600">Active</label>
                                </div>
                                <div class="flex items-center">
                                    <input type="radio" id="inactive" name="active" value="0" class="mr-2">
                                    <label for="inactive" class="text-sm text-gray600">Inactive</label>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Form Actions -->
                    <div class="border-t border-gray200 pt-6 mt-6">
                        <div class="flex justify-between items-center">
                            <div>
                                <button type="button" 
                                        onclick="closeNewVendorModal()" 
                                        class="text-primary hover:text-primary/90 font-medium text-sm">
                                    Cancel
                                </button>
                            </div>
                            <div class="flex space-x-3">
                                <button type="button" 
                                        onclick="saveAndNew()"
                                        class="px-4 py-2 border border-gray300 rounded text-sm font-medium text-gray700 hover:bg-gray50">
                                    Save and new
                                </button>
                                <button type="submit" 
                                        class="px-4 py-2 bg-primary text-white rounded text-sm font-medium hover:bg-primary/90">
                                    Save and close
                                </button>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="deleteModal" class="fixed inset-0 bg-gray900 bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="w-10 h-10 bg-red-100 rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-exclamation-triangle text-danger"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray900" id="deleteModalTitle">Delete vendor</h3>
                        <p class="text-gray600 text-sm" id="deleteModalMessage">Are you sure you want to delete this vendor?</p>
                    </div>
                </div>
            </div>
            
            <div class="border-t border-gray200 p-6 flex justify-end space-x-3">
                <button onclick="closeDeleteModal()" class="px-4 py-2 text-gray700 hover:bg-gray100 rounded text-sm font-medium">
                    Cancel
                </button>
                <form id="deleteForm" method="POST" style="display: none;">
                    @csrf
                    @method('DELETE')
                </form>
                <button onclick="confirmDelete()" class="bg-danger text-white px-4 py-2 rounded text-sm font-medium hover:bg-danger/90">
                    Delete
                </button>
            </div>
        </div>
    </div>

    <!-- Email Statements Modal -->
    <div id="emailModal" class="fixed inset-0 bg-gray900 bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="w-10 h-10 bg-primaryLight rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-envelope text-primary"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray900">Email statements</h3>
                        <p class="text-gray600 text-sm" id="emailModalMessage">Send statements to selected vendors</p>
                    </div>
                </div>
                
                <div class="space-y-4">
                    <div>
                        <label class="text-sm font-medium text-gray700 block mb-2">Statement period</label>
                        <select id="statementPeriod" class="quickbooks-dropdown w-full">
                            <option value="this_month">This month</option>
                            <option value="last_month">Last month</option>
                            <option value="this_quarter">This quarter</option>
                            <option value="last_quarter">Last quarter</option>
                            <option value="this_year">This year</option>
                            <option value="custom">Custom period</option>
                        </select>
                    </div>
                    
                    <div id="customPeriodFields" class="hidden space-y-3">
                        <div class="grid grid-cols-2 gap-3">
                            <div>
                                <label class="text-sm font-medium text-gray700 block mb-1">From</label>
                                <input type="date" class="quickbooks-input w-full" id="periodStart">
                            </div>
                            <div>
                                <label class="text-sm font-medium text-gray700 block mb-1">To</label>
                                <input type="date" class="quickbooks-input w-full" id="periodEnd">
                            </div>
                        </div>
                    </div>
                    
                    <div>
                        <label class="text-sm font-medium text-gray700 block mb-2">Include</label>
                        <div class="space-y-2">
                            <label class="flex items-center">
                                <input type="checkbox" checked class="quickbooks-checkbox mr-2" id="openTransactions">
                                <span class="text-sm">Open transactions only</span>
                            </label>
                            <label class="flex items-center">
                                <input type="checkbox" class="quickbooks-checkbox mr-2" id="includeAging">
                                <span class="text-sm">Include aging summary</span>
                            </label>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="border-t border-gray200 p-6 flex justify-end space-x-3">
                <button onclick="closeEmailModal()" class="px-4 py-2 text-gray700 hover:bg-gray100 rounded text-sm font-medium">
                    Cancel
                </button>
                <button onclick="sendStatements()" class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90">
                    Send statements
                </button>
            </div>
        </div>
    </div>

    <!-- Statement Generation Modal -->
    <div id="statementModal" class="fixed inset-0 bg-gray900 bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-file-invoice-dollar text-success"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray900">Generate Statement</h3>
                        <p class="text-gray600 text-sm" id="statementModalMessage"></p>
                    </div>
                </div>
                
                <form id="statementForm">
                    @csrf
                    <div class="space-y-4">
                        <div>
                            <label class="text-sm font-medium text-gray700 block mb-2">Period Start</label>
                            <input type="date" 
                                   id="period_start"
                                   name="period_start" 
                                   class="quickbooks-input w-full"
                                   value="{{ date('Y-m-01') }}">
                        </div>
                        <div>
                            <label class="text-sm font-medium text-gray700 block mb-2">Period End</label>
                            <input type="date" 
                                   id="period_end"
                                   name="period_end" 
                                   class="quickbooks-input w-full"
                                   value="{{ date('Y-m-t') }}">
                        </div>
                    </div>
                </form>
            </div>
            
            <div class="border-t border-gray200 p-6 flex justify-end space-x-3">
                <button onclick="closeStatementModal()" class="px-4 py-2 text-gray700 hover:bg-gray100 rounded text-sm font-medium">
                    Cancel
                </button>
                <button onclick="confirmGenerateStatement()" class="bg-success text-white px-4 py-2 rounded text-sm font-medium hover:bg-success/90">
                    Generate
                </button>
            </div>
        </div>
    </div>

    <script>
        // State management
        let selectedVendorIds = new Set();
        let currentDeleteId = null;
        let isBulkDelete = false;
        let currentStatementVendorId = null;
        let saveAndNewMode = false;

        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            setupEventListeners();
            setupNewVendorModal();
            setupFilterEvents();
        });

        // Setup new vendor modal
        function setupNewVendorModal() {
            const sameAsBillingCheckbox = document.getElementById('sameAsBilling');
            const shippingAddressFields = document.getElementById('shippingAddressFields');
            const vendor1099Checkbox = document.getElementById('vendor1099');
            const taxFields = document.getElementById('taxFields');
            const newVendorForm = document.getElementById('newVendorForm');
            const statementPeriod = document.getElementById('statementPeriod');

            // Toggle shipping address fields
            if (sameAsBillingCheckbox) {
                sameAsBillingCheckbox.addEventListener('change', function() {
                    if (shippingAddressFields) {
                        const inputs = shippingAddressFields.querySelectorAll('input, select');
                        inputs.forEach(input => {
                            input.disabled = this.checked;
                        });
                        
                        if (this.checked) {
                            shippingAddressFields.style.opacity = '0.5';
                            copyBillingToShipping();
                        } else {
                            shippingAddressFields.style.opacity = '1';
                        }
                    }
                });
            }

            // Toggle tax fields
            if (vendor1099Checkbox) {
                vendor1099Checkbox.addEventListener('change', function() {
                    if (taxFields) {
                        taxFields.style.display = this.checked ? 'block' : 'none';
                    }
                });
            }

            // Statement period change
            if (statementPeriod) {
                statementPeriod.addEventListener('change', function() {
                    const customFields = document.getElementById('customPeriodFields');
                    if (customFields) {
                        customFields.classList.toggle('hidden', this.value !== 'custom');
                    }
                });
            }

            // Form submission
            if (newVendorForm) {
                newVendorForm.addEventListener('submit', function(e) {
                    if (saveAndNewMode) {
                        e.preventDefault();
                        saveVendor();
                    }
                    // If not save and new mode, let the form submit normally
                });
            }

            // Initialize
            if (sameAsBillingCheckbox) sameAsBillingCheckbox.dispatchEvent(new Event('change'));
            if (vendor1099Checkbox) vendor1099Checkbox.dispatchEvent(new Event('change'));
        }

        // Modal functions
        function openNewVendorModal() {
            document.getElementById('newVendorModal').classList.remove('hidden');
            document.body.style.overflow = 'hidden';
        }

        function closeNewVendorModal() {
            document.getElementById('newVendorModal').classList.add('hidden');
            document.body.style.overflow = 'auto';
            resetNewVendorForm();
        }

        function resetNewVendorForm() {
            const form = document.getElementById('newVendorForm');
            if (form) {
                form.reset();
                
                // Reset specific fields
                const sameAsBilling = document.getElementById('sameAsBilling');
                const vendor1099 = document.getElementById('vendor1099');
                const active = document.getElementById('active');
                
                if (sameAsBilling) sameAsBilling.checked = false;
                if (vendor1099) vendor1099.checked = false;
                if (active) active.checked = true;
                
                // Trigger change events
                if (sameAsBilling) sameAsBilling.dispatchEvent(new Event('change'));
                if (vendor1099) vendor1099.dispatchEvent(new Event('change'));
            }
        }

        function saveVendor() {
            const displayName = document.getElementById('displayName').value;
            
            // Basic validation
            if (!displayName) {
                showNotification('Please enter a vendor display name', 'error');
                return;
            }

            // Submit form via AJAX for better UX
            const form = document.getElementById('newVendorForm');
            const formData = new FormData(form);

            fetch(form.action, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json',
                },
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification(data.message, 'success');
                    
                    if (saveAndNewMode) {
                        // Reset form for next entry
                        resetNewVendorForm();
                        document.getElementById('displayName').focus();
                    } else {
                        // Close modal and reload page
                        closeNewVendorModal();
                        window.location.href = data.redirect || window.location.href;
                    }
                } else {
                    showNotification(data.errors ? Object.values(data.errors)[0][0] : 'Failed to create vendor', 'error');
                }
            })
            .catch(error => {
                showNotification('Failed to create vendor', 'error');
            });
        }

        function saveAndNew() {
            saveAndNewMode = true;
            saveVendor();
        }

        function copyBillingToShipping() {
            const fields = ['Street1', 'Street2', 'City', 'State', 'Zip', 'Country'];
            fields.forEach(field => {
                const billingField = document.getElementById('billing' + field);
                const shippingField = document.getElementById('shipping' + field);
                if (billingField && shippingField) {
                    shippingField.value = billingField.value;
                }
            });
        }

        function copyShippingToBilling() {
            const fields = ['Street1', 'Street2', 'City', 'State', 'Zip', 'Country'];
            fields.forEach(field => {
                const billingField = document.getElementById('billing' + field);
                const shippingField = document.getElementById('shipping' + field);
                if (billingField && shippingField) {
                    billingField.value = shippingField.value;
                }
            });
        }

        // Setup filter events
        function setupFilterEvents() {
            const filters = ['statusFilter', 'balanceFilter', 'vendor1099Filter'];
            
            filters.forEach(filterId => {
                const filter = document.getElementById(filterId);
                if (filter) {
                    filter.addEventListener('change', applyFilters);
                }
            });
        }

        // Handle search key press
        function handleSearchKeyPress(event) {
            if (event.key === 'Enter') {
                applyFilters();
            }
        }

        // Apply filters
        function applyFilters() {
            const search = document.getElementById('searchInput').value;
            const status = document.getElementById('statusFilter').value;
            const balance = document.getElementById('balanceFilter').value;
            const vendor1099 = document.getElementById('vendor1099Filter').value;
            
            let url = '{{ route("admin.accounting.expenses.vendors.index") }}?';
            const params = new URLSearchParams();
            
            if (search) params.append('search', search);
            if (status) params.append('status', status);
            if (balance) params.append('balance', balance);
            if (vendor1099) params.append('vendor_1099', vendor1099);
            
            window.location.href = url + params.toString();
        }

        // Update bulk actions
        function updateBulkActions() {
            const count = selectedVendorIds.size;
            const exportBtn = document.getElementById('exportBtn');
            const emailBtn = document.getElementById('emailBtn');
            const deleteBtn = document.getElementById('deleteBtn');
            const selectAllCheckbox = document.getElementById('selectAll');

            if (count > 0) {
                exportBtn.disabled = false;
                emailBtn.disabled = false;
                deleteBtn.disabled = false;
            } else {
                exportBtn.disabled = true;
                emailBtn.disabled = true;
                deleteBtn.disabled = true;
            }

            // Update select all checkbox
            if (selectAllCheckbox) {
                const totalRows = document.querySelectorAll('.vendor-checkbox').length;
                selectAllCheckbox.checked = count > 0 && count === totalRows;
                selectAllCheckbox.indeterminate = count > 0 && count < totalRows;
            }
        }

        // Setup event listeners
        function setupEventListeners() {
            // Select all checkbox
            const selectAllCheckbox = document.getElementById('selectAll');
            if (selectAllCheckbox) {
                selectAllCheckbox.addEventListener('change', handleSelectAll);
            }

            // Vendor checkboxes
            document.querySelectorAll('.vendor-checkbox').forEach(checkbox => {
                checkbox.addEventListener('change', function() {
                    const vendorId = parseInt(this.getAttribute('data-id'));
                    const row = this.closest('tr');
                    
                    if (this.checked) {
                        selectedVendorIds.add(vendorId);
                        if (row) row.classList.add('selected');
                    } else {
                        selectedVendorIds.delete(vendorId);
                        if (row) row.classList.remove('selected');
                    }
                    
                    updateBulkActions();
                });
            });
        }

        // Handle select all
        function handleSelectAll() {
            const isChecked = this.checked;
            const checkboxes = document.querySelectorAll('.vendor-checkbox');
            
            checkboxes.forEach(checkbox => {
                checkbox.checked = isChecked;
                const vendorId = parseInt(checkbox.getAttribute('data-id'));
                const row = checkbox.closest('tr');
                
                if (isChecked) {
                    selectedVendorIds.add(vendorId);
                    if (row) row.classList.add('selected');
                } else {
                    selectedVendorIds.delete(vendorId);
                    if (row) row.classList.remove('selected');
                }
            });
            
            updateBulkActions();
        }

        // Vendor actions
        function generateStatement(vendorId) {
            currentStatementVendorId = vendorId;
            const vendorName = document.querySelector(`#vendor-row-${vendorId} .font-medium`).textContent;
            
            document.getElementById('statementModalMessage').textContent = 
                `Generate statement for "${vendorName}"`;
            document.getElementById('statementModal').classList.remove('hidden');
        }

        function closeStatementModal() {
            document.getElementById('statementModal').classList.add('hidden');
            currentStatementVendorId = null;
        }

        function confirmGenerateStatement() {
            if (!currentStatementVendorId) return;
            
            const formData = new FormData(document.getElementById('statementForm'));
            
            fetch(`/admin/accounting/expenses/vendors/${currentStatementVendorId}/generate-statement`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json',
                },
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification(data.message, 'success');
                    if (data.download_url) {
                        window.open(data.download_url, '_blank');
                    }
                    closeStatementModal();
                } else {
                    showNotification(data.message, 'error');
                }
            })
            .catch(error => {
                showNotification('Failed to generate statement', 'error');
            });
        }

        // Bulk actions
        function exportSelected() {
            if (selectedVendorIds.size === 0) return;
            
            // Create form and submit
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '{{ route("admin.accounting.expenses.vendors.export") }}';
            form.style.display = 'none';
            
            const csrfToken = document.createElement('input');
            csrfToken.type = 'hidden';
            csrfToken.name = '_token';
            csrfToken.value = '{{ csrf_token() }}';
            form.appendChild(csrfToken);
            
            selectedVendorIds.forEach(id => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'vendor_ids[]';
                input.value = id;
                form.appendChild(input);
            });
            
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }

        function emailStatements() {
            const selectedVendors = Array.from(selectedVendorIds);
            if (selectedVendors.length === 0) return;
            
            document.getElementById('emailModalMessage').textContent = 
                `Send statements to ${selectedVendors.length} vendor${selectedVendors.length > 1 ? 's' : ''}`;
            document.getElementById('emailModal').classList.remove('hidden');
        }

        function closeEmailModal() {
            document.getElementById('emailModal').classList.add('hidden');
        }

        function sendStatements() {
            const selectedVendors = Array.from(selectedVendorIds);
            if (selectedVendors.length === 0) return;
            
            showNotification(`Sending statements to ${selectedVendors.length} vendors...`, 'info');
            
            setTimeout(() => {
                closeEmailModal();
                showNotification(`Statements sent to ${selectedVendors.length} vendors`, 'success');
            }, 2000);
        }

        function deleteSelected() {
            if (selectedVendorIds.size === 0) return;
            
            currentDeleteId = 'bulk';
            isBulkDelete = true;
            
            document.getElementById('deleteModalTitle').textContent = 
                `Delete ${selectedVendorIds.size} vendor${selectedVendorIds.size > 1 ? 's' : ''}`;
            document.getElementById('deleteModalMessage').textContent = 
                `Are you sure you want to delete ${selectedVendorIds.size} vendor${selectedVendorIds.size > 1 ? 's' : ''}? This action cannot be undone.`;
            
            document.getElementById('deleteModal').classList.remove('hidden');
        }

        // Delete modal functions
        function openDeleteModal(vendorId) {
            currentDeleteId = vendorId;
            isBulkDelete = false;
            
            if (vendorId === 'bulk') return; // Already handled
            
            const vendorName = document.querySelector(`#vendor-row-${vendorId} .font-medium`).textContent;
            
            document.getElementById('deleteModalTitle').textContent = 'Delete vendor';
            document.getElementById('deleteModalMessage').textContent = 
                `Are you sure you want to delete "${vendorName}"? This action cannot be undone.`;
            
            document.getElementById('deleteModal').classList.remove('hidden');
        }

        function closeDeleteModal() {
            document.getElementById('deleteModal').classList.add('hidden');
            currentDeleteId = null;
            isBulkDelete = false;
        }

        function confirmDelete() {
            if (isBulkDelete) {
                // Bulk delete - create form and submit
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = '{{ route("admin.accounting.expenses.vendors.bulk-destroy") }}';
                form.style.display = 'none';
                
                const csrfToken = document.createElement('input');
                csrfToken.type = 'hidden';
                csrfToken.name = '_token';
                csrfToken.value = '{{ csrf_token() }}';
                form.appendChild(csrfToken);
                
                const methodInput = document.createElement('input');
                methodInput.type = 'hidden';
                methodInput.name = '_method';
                methodInput.value = 'DELETE';
                form.appendChild(methodInput);
                
                selectedVendorIds.forEach(id => {
                    const input = document.createElement('input');
                    input.type = 'hidden';
                    input.name = 'vendor_ids[]';
                    input.value = id;
                    form.appendChild(input);
                });
                
                document.body.appendChild(form);
                form.submit();
            } else {
                // Single delete
                const form = document.getElementById('deleteForm');
                form.action = '{{ route("admin.accounting.expenses.vendors.destroy", ":id") }}'.replace(':id', currentDeleteId);
                form.submit();
            }
        }

        // Utility functions
        function showNotification(message, type = 'info') {
            // Remove existing notification
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-success',
                error: 'bg-danger',
                info: 'bg-primary',
                warning: 'bg-warning'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            // Remove after 3 seconds
            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Add animation styles
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeIn {
                from { opacity: 0; transform: translateY(-10px); }
                to { opacity: 1; transform: translateY(0); }
            }
            @keyframes fadeOut {
                from { opacity: 1; transform: translateY(0); }
                to { opacity: 0; transform: translateY(-10px); }
            }
            .animate-fade-in {
                animation: fadeIn 0.3s ease-out;
            }
            .animate-fade-out {
                animation: fadeOut 0.3s ease-in;
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>