<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Approve Journal Entries | IMMIA FINANCE</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Approve Journal Entries', 'subtitle' => 'Review and approve pending journal entries'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Action Bar -->
            <div class="flex justify-between items-center mb-6">
                <div>
                    <h2 class="text-xl font-bold text-gray-900">Pending Approval</h2>
                    <p class="text-gray-600">{{ $journalEntries->count() }} entries pending approval</p>
                </div>
                <div class="flex space-x-3">
                    <form action="{{ route('admin.accounting.journals.bulk-approve') }}" method="POST" id="bulkApproveForm">
                        @csrf
                        <button type="submit" 
                                class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg font-semibold flex items-center space-x-2 transition duration-150"
                                id="bulkApproveBtn" disabled>
                            <i class="fas fa-check-double"></i>
                            <span>Approve Selected</span>
                        </button>
                    </form>
                    <a href="{{ route('admin.accounting.journals.index') }}" 
                       class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg font-semibold flex items-center space-x-2 transition duration-150">
                        <i class="fas fa-arrow-left"></i>
                        <span>Back to Journals</span>
                    </a>
                </div>
            </div>

            <!-- Journal Entries Table -->
            <div class="bg-white rounded-xl shadow-sm border">
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50 border-b">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-12">
                                    <input type="checkbox" id="selectAll" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                </th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Entry Number</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Description</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Reference</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Currency</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created By</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            @forelse($journalEntries as $entry)
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <input type="checkbox" name="journal_entries[]" value="{{ $entry->id }}" 
                                           class="journal-checkbox rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm font-medium text-gray-900">{{ $entry->entry_number }}</div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900">{{ $entry->entry_date->format('M d, Y') }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ Str::limit($entry->description, 60) }}</div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-500">{{ $entry->reference ?? 'N/A' }}</div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900">{{ $entry->currency->code ?? 'N/A' }}</div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm font-medium text-gray-900">
                                        {{ $entry->currency->symbol ?? '$' }}{{ number_format($entry->total_debit, 2) }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900">{{ $entry->creator->name ?? 'System' }}</div>
                                    <div class="text-xs text-gray-500">{{ $entry->created_at->format('M d, Y') }}</div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div class="flex items-center space-x-2">
                                        <a href="{{ route('admin.accounting.journals.show', $entry->id) }}" 
                                           class="text-blue-600 hover:text-blue-900 transition duration-150" title="Review">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <form action="{{ route('admin.accounting.journals.post', $entry->id) }}" 
                                              method="POST" class="inline">
                                            @csrf
                                            <button type="submit" 
                                                    class="text-green-600 hover:text-green-900 transition duration-150"
                                                    onclick="return confirm('Are you sure you want to approve and post this journal entry? This will update the general ledger.')"
                                                    title="Approve & Post">
                                                <i class="fas fa-check"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                            @empty
                            <tr>
                                <td colspan="9" class="px-6 py-4 text-center text-sm text-gray-500">
                                    <div class="flex flex-col items-center justify-center py-8">
                                        <i class="fas fa-check-circle text-4xl text-gray-300 mb-4"></i>
                                        <p class="text-gray-500">No journal entries pending approval</p>
                                        <a href="{{ route('admin.accounting.journals.index') }}" 
                                           class="mt-2 text-blue-600 hover:text-blue-700 font-medium transition duration-150">
                                            View all journal entries
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Pagination -->
            @if($journalEntries->hasPages())
            <div class="mt-6">
                {{ $journalEntries->links() }}
            </div>
            @endif
        </main>
    </div>

    <script>
        // Bulk selection functionality
        document.addEventListener('DOMContentLoaded', function() {
            const selectAll = document.getElementById('selectAll');
            const checkboxes = document.querySelectorAll('.journal-checkbox');
            const bulkApproveBtn = document.getElementById('bulkApproveBtn');
            const bulkApproveForm = document.getElementById('bulkApproveForm');

            // Select all functionality
            selectAll.addEventListener('change', function() {
                checkboxes.forEach(checkbox => {
                    checkbox.checked = this.checked;
                });
                updateBulkApproveButton();
            });

            // Individual checkbox functionality
            checkboxes.forEach(checkbox => {
                checkbox.addEventListener('change', function() {
                    updateBulkApproveButton();
                    updateSelectAll();
                });
            });

            function updateBulkApproveButton() {
                const checkedCount = document.querySelectorAll('.journal-checkbox:checked').length;
                bulkApproveBtn.disabled = checkedCount === 0;
                
                if (checkedCount > 0) {
                    bulkApproveBtn.innerHTML = `<i class="fas fa-check-double"></i><span>Approve Selected (${checkedCount})</span>`;
                } else {
                    bulkApproveBtn.innerHTML = `<i class="fas fa-check-double"></i><span>Approve Selected</span>`;
                }
            }

            function updateSelectAll() {
                const checkedCount = document.querySelectorAll('.journal-checkbox:checked').length;
                const totalCount = checkboxes.length;
                selectAll.checked = checkedCount === totalCount && totalCount > 0;
                selectAll.indeterminate = checkedCount > 0 && checkedCount < totalCount;
            }

            // Form submission
            bulkApproveForm.addEventListener('submit', function(e) {
                const checkedCount = document.querySelectorAll('.journal-checkbox:checked').length;
                if (checkedCount === 0) {
                    e.preventDefault();
                    alert('Please select at least one journal entry to approve.');
                    return false;
                }
                
                if (!confirm(`Are you sure you want to approve ${checkedCount} journal entry(ies)? This will update the general ledger.`)) {
                    e.preventDefault();
                    return false;
                }
            });

            // Initialize
            updateBulkApproveButton();
        });
    </script>
</body>
</html>