<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Journal Entry | IMMIA FINANCE</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        [x-cloak] { display: none !important; }
    </style>
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Create Journal Entry', 'subtitle' => 'Create new accounting journal entry'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if($errors->any())
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">Please fix the following errors:</span>
                    </div>
                    <ul class="mt-2 list-disc list-inside text-red-700">
                        @foreach($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <form action="{{ route('admin.accounting.journals.store') }}" method="POST" id="journalForm">
                @csrf
                
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-6">
                    <!-- Left Column - Header Information -->
                    <div class="lg:col-span-2">
                        <div class="bg-white rounded-xl shadow-sm border p-6">
                            <h3 class="text-lg font-semibold text-gray-900 mb-4">Journal Information</h3>
                            
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                <div>
                                    <label for="entry_number" class="block text-sm font-medium text-gray-700 mb-1">Journal Number *</label>
                                    <input type="text" id="entry_number" name="entry_number" 
                                           value="{{ old('entry_number', 'JE-' . date('Ymd-His')) }}"
                                           class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                           required>
                                </div>
                                <div>
                                    <label for="entry_date" class="block text-sm font-medium text-gray-700 mb-1">Entry Date *</label>
                                    <input type="date" id="entry_date" name="entry_date" 
                                           value="{{ old('entry_date', date('Y-m-d')) }}"
                                           class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                           required>
                                </div>
                            </div>

                            <div class="mb-4">
                                <label for="description" class="block text-sm font-medium text-gray-700 mb-1">Description *</label>
                                <textarea id="description" name="description" rows="2"
                                          class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                          required>{{ old('description') }}</textarea>
                            </div>

                            <div class="mb-4">
                                <label for="reference" class="block text-sm font-medium text-gray-700 mb-1">Reference</label>
                                <input type="text" id="reference" name="reference" 
                                       value="{{ old('reference') }}"
                                       class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            </div>
                        </div>

                        <!-- Currency Section -->
                        <div class="bg-white rounded-xl shadow-sm border p-6 mt-6">
                            <h3 class="text-lg font-semibold text-gray-900 mb-4">Currency Information</h3>
                            
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <div>
                                    <label for="currency_id" class="block text-sm font-medium text-gray-700 mb-1">Currency *</label>
                                    <select id="currency_id" name="currency_id" 
                                            class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                            required>
                                        <option value="">Select Currency</option>
                                        @foreach($currencies as $currency)
                                            <option value="{{ $currency->id }}" 
                                                    data-symbol="{{ $currency->symbol }}"
                                                    {{ old('currency_id') == $currency->id ? 'selected' : '' }}>
                                                {{ $currency->code }} - {{ $currency->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                </div>
                                <div>
                                    <label for="exchange_rate" class="block text-sm font-medium text-gray-700 mb-1">Exchange Rate *</label>
                                    <input type="number" id="exchange_rate" name="exchange_rate" 
                                           value="{{ old('exchange_rate', 1) }}" step="0.000001"
                                           class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                           required>
                                    <div class="text-xs text-gray-500 mt-1" id="exchange-rate-info">
                                        1 <span id="selected-currency-code">USD</span> = <span id="base-rate">1.000000</span> USD
                                    </div>
                                </div>
                                <div>
                                    <label for="exchange_date" class="block text-sm font-medium text-gray-700 mb-1">Exchange Date</label>
                                    <input type="date" id="exchange_date" name="exchange_date" 
                                           value="{{ old('exchange_date', date('Y-m-d')) }}"
                                           class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                                </div>
                            </div>
                        </div>

                        <!-- Line Items -->
                        <div class="bg-white rounded-xl shadow-sm border p-6 mt-6">
                            <div class="flex justify-between items-center mb-4">
                                <h3 class="text-lg font-semibold text-gray-900">Journal Lines</h3>
                                <button type="button" id="add-line" 
                                        class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-2 rounded-lg text-sm font-semibold flex items-center space-x-2 transition duration-150">
                                    <i class="fas fa-plus"></i>
                                    <span>Add Line</span>
                                </button>
                            </div>

                            <!-- Balance Alert -->
                            <div id="balance-alert" class="hidden bg-red-50 border border-red-200 rounded-lg p-4 mb-4">
                                <div class="flex items-center">
                                    <i class="fas fa-exclamation-triangle text-red-500 mr-2"></i>
                                    <span class="text-red-800 font-semibold">OUT OF BALANCE BY: <span id="balance-difference">0.00</span></span>
                                </div>
                            </div>

                            <div class="overflow-x-auto">
                                <table class="w-full">
                                    <thead class="bg-gray-50 border-b">
                                        <tr>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Account *</th>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Debit</th>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Credit</th>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Memo</th>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-20">Action</th>
                                        </tr>
                                    </thead>
                                    <tbody id="journal-lines">
                                        <!-- Lines will be added here dynamically -->
                                        <tr class="journal-line border-b">
                                            <td class="px-4 py-3">
                                                <select name="items[0][account_id]" class="account-select w-full border border-gray-300 rounded px-2 py-1 text-sm" required>
                                                    <option value="">Select Account</option>
                                                    @foreach($accounts as $account)
                                                        <option value="{{ $account->id }}">{{ $account->code }} - {{ $account->name }}</option>
                                                    @endforeach
                                                </select>
                                            </td>
                                            <td class="px-4 py-3">
                                                <div class="flex items-center">
                                                    <span class="currency-symbol mr-1">$</span>
                                                    <input type="number" name="items[0][debit]" value="0" step="0.01" min="0" 
                                                           class="debit-amount w-full border border-gray-300 rounded px-2 py-1 text-sm text-right" 
                                                           oninput="updateTotals()">
                                                </div>
                                            </td>
                                            <td class="px-4 py-3">
                                                <div class="flex items-center">
                                                    <span class="currency-symbol mr-1">$</span>
                                                    <input type="number" name="items[0][credit]" value="0" step="0.01" min="0" 
                                                           class="credit-amount w-full border border-gray-300 rounded px-2 py-1 text-sm text-right" 
                                                           oninput="updateTotals()">
                                                </div>
                                            </td>
                                            <td class="px-4 py-3">
                                                <input type="text" name="items[0][description]" 
                                                       class="w-full border border-gray-300 rounded px-2 py-1 text-sm" 
                                                       placeholder="Memo">
                                            </td>
                                            <td class="px-4 py-3">
                                                <button type="button" class="delete-line text-red-600 hover:text-red-800">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>

                            <!-- Totals -->
                            <div class="mt-4 flex justify-end space-x-6 text-sm">
                                <div class="text-right">
                                    <div class="font-semibold text-gray-700">Total Debits:</div>
                                    <div class="text-green-600 font-bold" id="total-debits">$0.00</div>
                                </div>
                                <div class="text-right">
                                    <div class="font-semibold text-gray-700">Total Credits:</div>
                                    <div class="text-red-600 font-bold" id="total-credits">$0.00</div>
                                </div>
                                <div class="text-right">
                                    <div class="font-semibold text-gray-700">Difference:</div>
                                    <div class="text-yellow-600 font-bold" id="total-difference">$0.00</div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Right Column - Actions & Summary -->
                    <div class="lg:col-span-1">
                        <div class="bg-white rounded-xl shadow-sm border p-6 sticky top-6">
                            <h3 class="text-lg font-semibold text-gray-900 mb-4">Actions</h3>
                            
                            <div class="space-y-3">
                                <button type="submit" name="status" value="draft" 
                                        class="w-full bg-gray-600 hover:bg-gray-700 text-white py-2 px-4 rounded-lg font-semibold transition duration-150">
                                    Save as Draft
                                </button>
                                <button type="submit" name="status" value="posted" 
                                        class="w-full bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-semibold transition duration-150">
                                    Save & Post
                                </button>
                                <a href="{{ route('admin.accounting.journals.index') }}" 
                                   class="w-full bg-white border border-gray-300 hover:bg-gray-50 text-gray-700 py-2 px-4 rounded-lg font-semibold transition duration-150 text-center block">
                                    Cancel
                                </a>
                            </div>

                            <div class="mt-6 pt-6 border-t border-gray-200">
                                <h4 class="text-sm font-semibold text-gray-700 mb-2">Summary</h4>
                                <div class="space-y-2 text-sm">
                                    <div class="flex justify-between">
                                        <span class="text-gray-600">Total Lines:</span>
                                        <span id="line-count" class="font-semibold">1</span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span class="text-gray-600">Status:</span>
                                        <span class="font-semibold text-yellow-600">Draft</span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span class="text-gray-600">Created By:</span>
                                        <span class="font-semibold">{{ auth()->user()->name }}</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </main>
    </div>

    <script>
        let lineCount = 1;

        // Add new line
        document.getElementById('add-line').addEventListener('click', function() {
            const tbody = document.getElementById('journal-lines');
            const newLine = document.createElement('tr');
            newLine.className = 'journal-line border-b';
            newLine.innerHTML = `
                <td class="px-4 py-3">
                    <select name="items[${lineCount}][account_id]" class="account-select w-full border border-gray-300 rounded px-2 py-1 text-sm" required>
                        <option value="">Select Account</option>
                        @foreach($accounts as $account)
                            <option value="{{ $account->id }}">{{ $account->code }} - {{ $account->name }}</option>
                        @endforeach
                    </select>
                </td>
                <td class="px-4 py-3">
                    <div class="flex items-center">
                        <span class="currency-symbol mr-1">$</span>
                        <input type="number" name="items[${lineCount}][debit]" value="0" step="0.01" min="0" 
                               class="debit-amount w-full border border-gray-300 rounded px-2 py-1 text-sm text-right" 
                               oninput="updateTotals()">
                    </div>
                </td>
                <td class="px-4 py-3">
                    <div class="flex items-center">
                        <span class="currency-symbol mr-1">$</span>
                        <input type="number" name="items[${lineCount}][credit]" value="0" step="0.01" min="0" 
                               class="credit-amount w-full border border-gray-300 rounded px-2 py-1 text-sm text-right" 
                               oninput="updateTotals()">
                    </div>
                </td>
                <td class="px-4 py-3">
                    <input type="text" name="items[${lineCount}][description]" 
                           class="w-full border border-gray-300 rounded px-2 py-1 text-sm" 
                           placeholder="Memo">
                </td>
                <td class="px-4 py-3">
                    <button type="button" class="delete-line text-red-600 hover:text-red-800">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            `;
            tbody.appendChild(newLine);
            lineCount++;
            updateLineCount();
        });

        // Delete line
        document.addEventListener('click', function(e) {
            if (e.target.closest('.delete-line')) {
                const line = e.target.closest('.journal-line');
                if (document.querySelectorAll('.journal-line').length > 1) {
                    line.remove();
                    updateTotals();
                    updateLineCount();
                }
            }
        });

        // Update currency symbols when currency changes
        document.getElementById('currency_id').addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const symbol = selectedOption.getAttribute('data-symbol') || '$';
            const code = selectedOption.text.split(' - ')[0];
            
            // Update all currency symbols
            document.querySelectorAll('.currency-symbol').forEach(el => {
                el.textContent = symbol;
            });
            
            // Update exchange rate info
            document.getElementById('selected-currency-code').textContent = code;
            document.getElementById('base-rate').textContent = document.getElementById('exchange_rate').value;
        });

        // Update exchange rate info
        document.getElementById('exchange_rate').addEventListener('input', function() {
            document.getElementById('base-rate').textContent = this.value;
        });

        function updateLineCount() {
            document.getElementById('line-count').textContent = document.querySelectorAll('.journal-line').length;
        }

        function updateTotals() {
            let totalDebits = 0;
            let totalCredits = 0;

            document.querySelectorAll('.journal-line').forEach(line => {
                const debitInput = line.querySelector('.debit-amount');
                const creditInput = line.querySelector('.credit-amount');
                
                totalDebits += parseFloat(debitInput.value) || 0;
                totalCredits += parseFloat(creditInput.value) || 0;
            });

            const difference = totalDebits - totalCredits;
            const symbol = document.querySelector('.currency-symbol').textContent;

            document.getElementById('total-debits').textContent = symbol + totalDebits.toFixed(2);
            document.getElementById('total-credits').textContent = symbol + totalCredits.toFixed(2);
            document.getElementById('total-difference').textContent = symbol + Math.abs(difference).toFixed(2);

            const balanceAlert = document.getElementById('balance-alert');
            const balanceDifference = document.getElementById('balance-difference');

            if (Math.abs(difference) > 0.01) {
                balanceAlert.classList.remove('hidden');
                balanceDifference.textContent = symbol + Math.abs(difference).toFixed(2);
            } else {
                balanceAlert.classList.add('hidden');
            }
        }

        // Form validation
        document.getElementById('journalForm').addEventListener('submit', function(e) {
            const totalDebits = Array.from(document.querySelectorAll('.debit-amount'))
                .reduce((sum, input) => sum + (parseFloat(input.value) || 0), 0);
            const totalCredits = Array.from(document.querySelectorAll('.credit-amount'))
                .reduce((sum, input) => sum + (parseFloat(input.value) || 0), 0);

            if (Math.abs(totalDebits - totalCredits) > 0.01) {
                e.preventDefault();
                alert('Journal entry is not balanced. Total debits must equal total credits.');
                return false;
            }

            // Validate at least one line has amount
            const hasAmount = Array.from(document.querySelectorAll('.journal-line')).some(line => {
                const debit = parseFloat(line.querySelector('.debit-amount').value) || 0;
                const credit = parseFloat(line.querySelector('.credit-amount').value) || 0;
                return debit > 0 || credit > 0;
            });

            if (!hasAmount) {
                e.preventDefault();
                alert('Please enter at least one debit or credit amount.');
                return false;
            }
        });

        // Initialize
        updateTotals();
        updateLineCount();
    </script>
</body>
</html>