<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Employee | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#146CAA',
                        primaryLight: '#E1F0FA',
                        secondary: '#6b7280',
                        success: '#107C10',
                        warning: '#F68C1E',
                        danger: '#D83B01',
                        info: '#0078D4',
                        gray50: '#FAF9F8',
                        gray100: '#F3F2F1',
                        gray200: '#EDEBE9',
                        gray300: '#E1DFDD',
                        gray400: '#C8C6C4',
                        gray500: '#A19F9D',
                        gray600: '#605E5C',
                        gray700: '#323130',
                        gray800: '#201F1E',
                        gray900: '#000000'
                    }
                }
            }
        }
    </script>
    <style>
        /* QuickBooks Online Styling */
        .quickbooks-input {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            transition: all 0.2s;
        }
        .quickbooks-input:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-select {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            transition: all 0.2s;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='16' height='16' viewBox='0 0 24 24' fill='none' stroke='%23605E5C' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3E%3Cpath d='m6 9 6 6 6-6'%3E%3C/path%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: right 12px center;
            background-size: 16px;
            padding-right: 36px;
        }
        .quickbooks-select:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-textarea {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            resize: vertical;
            min-height: 60px;
        }
        .quickbooks-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
        }
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
            vertical-align: middle;
        }
        .quickbooks-table tbody tr:hover {
            background-color: #f8fafc;
        }
        .quickbooks-checkbox {
            width: 18px;
            height: 18px;
            border: 2px solid #605E5C;
            border-radius: 3px;
            cursor: pointer;
            accent-color: #146CAA;
        }
        .quickbooks-checkbox:checked {
            background-color: #146CAA;
            border-color: #146CAA;
        }
        .quickbooks-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .form-section {
            background: white;
            border-radius: 8px;
            padding: 24px;
            margin-bottom: 16px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .form-section h3 {
            font-size: 18px;
            font-weight: 600;
            color: #323130;
            margin-bottom: 20px;
            padding-bottom: 12px;
            border-bottom: 1px solid #EDEBE9;
        }
        .required-field::after {
            content: ' *';
            color: #D83B01;
        }
        .quickbooks-button-primary {
            background-color: #146CAA;
            color: white;
            border: none;
            border-radius: 4px;
            padding: 8px 16px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background-color 0.2s;
        }
        .quickbooks-button-primary:hover {
            background-color: #11598C;
        }
        .quickbooks-button-secondary {
            background-color: white;
            color: #323130;
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 8px 16px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
        }
        .quickbooks-button-secondary:hover {
            background-color: #F3F2F1;
        }
        .is-invalid {
            border-color: #D83B01 !important;
        }
        .is-valid {
            border-color: #107C10 !important;
        }
        .removed-earning {
            opacity: 0.6;
            background-color: #fef2f2;
            border-color: #fee2e2;
        }
        .removed-earning td {
            text-decoration: line-through;
        }
        .removed-earning select,
        .removed-earning input,
        .removed-earning button {
            opacity: 0.5;
            cursor: not-allowed;
        }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Edit Employee', 'subtitle' => 'Update employee information'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Back Button -->
            <div class="mb-6">
                <a href="{{ route('admin.accounting.payroll.employees.show', $employee->id) }}" 
                   class="text-primary hover:text-primary/90 font-medium text-sm flex items-center">
                    <i class="fas fa-arrow-left mr-2"></i>Back to Employee Details
                </a>
            </div>

            <!-- Employee Form -->
            <div class="bg-white rounded-lg border border-gray200 shadow-sm">
                <form id="employeeForm" action="{{ route('admin.accounting.payroll.employees.update', $employee->id) }}" method="POST">
                    @csrf
                    @method('PUT')
                    
                    <!-- Personal Information Section -->
                    <div class="form-section">
                        <h3>Personal Information</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">First Name</label>
                                <input type="text" 
                                       name="first_name" 
                                       class="quickbooks-input" 
                                       value="{{ old('first_name', $employee->first_name) }}"
                                       placeholder="John"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Middle Name</label>
                                <input type="text" 
                                       name="middle_name" 
                                       class="quickbooks-input" 
                                       value="{{ old('middle_name', $employee->middle_name) }}"
                                       placeholder="Middle name">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Last Name</label>
                                <input type="text" 
                                       name="last_name" 
                                       class="quickbooks-input" 
                                       value="{{ old('last_name', $employee->last_name) }}"
                                       placeholder="Doe"
                                       required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Employee Number</label>
                                <input type="text" 
                                       name="employee_number" 
                                       class="quickbooks-input" 
                                       value="{{ old('employee_number', $employee->employee_number) }}"
                                       placeholder="EMP001"
                                       required>
                                <p class="text-xs text-gray500 mt-1">Unique identifier for the employee</p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Email Address</label>
                                <input type="email" 
                                       name="email" 
                                       class="quickbooks-input" 
                                       value="{{ old('email', $employee->email) }}"
                                       placeholder="john.doe@example.com"
                                       required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Phone Number</label>
                                <input type="tel" 
                                       name="phone" 
                                       class="quickbooks-input" 
                                       value="{{ old('phone', $employee->phone) }}"
                                       placeholder="+260 97 123 4567">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Date of Birth</label>
                                <input type="date" 
                                       name="date_of_birth" 
                                       class="quickbooks-input"
                                       value="{{ old('date_of_birth', $employee->date_of_birth ? $employee->date_of_birth->format('Y-m-d') : '') }}"
                                       required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Gender</label>
                                <select name="gender" class="quickbooks-select" required>
                                    <option value="">Select Gender</option>
                                    <option value="male" {{ old('gender', $employee->gender) == 'male' ? 'selected' : '' }}>Male</option>
                                    <option value="female" {{ old('gender', $employee->gender) == 'female' ? 'selected' : '' }}>Female</option>
                                    <option value="other" {{ old('gender', $employee->gender) == 'other' ? 'selected' : '' }}>Other</option>
                                </select>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">National ID</label>
                                <input type="text" 
                                       name="national_id" 
                                       class="quickbooks-input" 
                                       value="{{ old('national_id', $employee->national_id) }}"
                                       placeholder="123456/78/9"
                                       required>
                            </div>
                        </div>
                    </div>

                    <!-- Employment Details Section -->
                    <div class="form-section">
                        <h3>Employment Details</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Hire Date</label>
                                <input type="date" 
                                       name="hire_date" 
                                       class="quickbooks-input"
                                       value="{{ old('hire_date', $employee->hire_date->format('Y-m-d')) }}"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Employment Type</label>
                                <select name="employment_type" class="quickbooks-select" required>
                                    <option value="">Select Type</option>
                                    <option value="permanent" {{ old('employment_type', $employee->employment_type) == 'permanent' ? 'selected' : '' }}>Permanent</option>
                                    <option value="contract" {{ old('employment_type', $employee->employment_type) == 'contract' ? 'selected' : '' }}>Contract</option>
                                    <option value="temporary" {{ old('employment_type', $employee->employment_type) == 'temporary' ? 'selected' : '' }}>Temporary</option>
                                    <option value="casual" {{ old('employment_type', $employee->employment_type) == 'casual' ? 'selected' : '' }}>Casual</option>
                                </select>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Department</label>
                                <input type="text" 
                                       name="department" 
                                       class="quickbooks-input" 
                                       value="{{ old('department', $employee->department) }}"
                                       placeholder="Finance">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Position/Title</label>
                                <input type="text" 
                                       name="position" 
                                       class="quickbooks-input" 
                                       value="{{ old('position', $employee->position) }}"
                                       placeholder="Accountant">
                            </div>
                        </div>
                    </div>

                    <!-- Earnings Section (Table-based like Petty Cash) -->
                    <div class="form-section">
                        <div class="flex justify-between items-center mb-6">
                            <h3>Earnings</h3>
                            <button type="button" 
                                    id="addEarningRowBtn" 
                                    class="quickbooks-button-secondary flex items-center">
                                <i class="fas fa-plus mr-2"></i> Add Row
                            </button>
                        </div>

                        <!-- Earnings Table -->
                        <div class="border border-gray200 rounded-lg overflow-hidden">
                            <table class="quickbooks-table">
                                <thead>
                                    <tr>
                                        <th>Earning Type</th>
                                        <th>Amount</th>
                                        <th>Frequency</th>
                                        <th>Effective Date</th>
                                        <th>Taxable</th>
                                        <th>Recurring</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="earningsContainer">
                                    <!-- Earnings will be loaded here dynamically -->
                                </tbody>
                                <tfoot class="bg-gray100">
                                    <tr>
                                        <td colspan="2" class="text-right text-sm font-medium text-gray700 py-3">
                                            <div>Total Gross Salary:</div>
                                            <div class="text-xs text-gray500">Taxable Earnings:</div>
                                            <div class="text-xs text-gray500">Non-Taxable Earnings:</div>
                                        </td>
                                        <td colspan="5" class="py-3">
                                            <div class="text-lg font-semibold text-gray900" id="totalGrossSalary">
                                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($totalEarnings, 2) }}
                                            </div>
                                            <div class="text-sm text-red-600" id="totalTaxableEarnings">
                                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($taxableEarnings, 2) }}
                                            </div>
                                            <div class="text-sm text-green-600" id="totalNonTaxableEarnings">
                                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($nonTaxableEarnings, 2) }}
                                            </div>
                                        </td>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>

                        <!-- Empty State Message -->
                        <div id="noEarningsMessage" class="text-center py-8 text-gray500 {{ $employee->earnings->count() > 0 ? 'hidden' : '' }}">
                            <i class="fas fa-money-bill-wave text-3xl mb-3 text-gray300"></i>
                            <p>No earnings added yet</p>
                            <p class="text-sm mt-1">Click "Add Row" to add earnings items</p>
                        </div>
                    </div>

                    <!-- Statutory Information Section -->
                    <div class="form-section">
                        <h3>Statutory Information</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">NAPSA Number</label>
                                <input type="text" 
                                       name="napsa_number" 
                                       class="quickbooks-input" 
                                       value="{{ old('napsa_number', $employee->napsa_number) }}"
                                       placeholder="NPSA123456">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">NHIMA Number</label>
                                <input type="text" 
                                       name="nhima_number" 
                                       class="quickbooks-input" 
                                       value="{{ old('nhima_number', $employee->nhima_number) }}"
                                       placeholder="NHIMA123456">
                            </div>
                        </div>
                    </div>

                    <!-- Payment Details Section -->
                    <div class="form-section">
                        <h3>Payment Details</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Payment Method</label>
                                <select name="payment_method" class="quickbooks-select" required>
                                    <option value="">Select Method</option>
                                    <option value="bank" {{ old('payment_method', $employee->payment_method) == 'bank' ? 'selected' : '' }}>Bank Transfer</option>
                                    <option value="cash" {{ old('payment_method', $employee->payment_method) == 'cash' ? 'selected' : '' }}>Cash</option>
                                    <option value="mobile_money" {{ old('payment_method', $employee->payment_method) == 'mobile_money' ? 'selected' : '' }}>Mobile Money</option>
                                </select>
                            </div>
                            <div id="bankFields" style="{{ old('payment_method', $employee->payment_method) == 'bank' ? 'display: block;' : 'display: none;' }}">
                                <label class="block text-sm font-medium text-gray700 mb-2">Bank Name</label>
                                <input type="text" 
                                       name="bank_name" 
                                       class="quickbooks-input" 
                                       value="{{ old('bank_name', $employee->bank_name) }}"
                                       placeholder="Bank of Zambia">
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6" id="bankAccountFields" style="{{ old('payment_method', $employee->payment_method) == 'bank' ? 'display: grid;' : 'display: none;' }}">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Bank Account Number</label>
                                <input type="text" 
                                       name="bank_account_number" 
                                       class="quickbooks-input" 
                                       value="{{ old('bank_account_number', $employee->bank_account_number) }}"
                                       placeholder="1234567890">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Bank Account Name</label>
                                <input type="text" 
                                       name="bank_account_name" 
                                       class="quickbooks-input" 
                                       value="{{ old('bank_account_name', $employee->bank_account_name) }}"
                                       placeholder="John Doe">
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6" id="bankBranchField" style="{{ old('payment_method', $employee->payment_method) == 'bank' ? 'display: grid;' : 'display: none;' }}">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Bank Branch</label>
                                <input type="text" 
                                       name="bank_branch" 
                                       class="quickbooks-input" 
                                       value="{{ old('bank_branch', $employee->bank_branch) }}"
                                       placeholder="Main Branch">
                            </div>
                        </div>
                    </div>

                    <!-- Additional Information Section -->
                    <div class="form-section">
                        <h3>Additional Information</h3>
                        
                        <div class="mb-6">
                            <label class="block text-sm font-medium text-gray700 mb-2">Notes</label>
                            <textarea name="notes" 
                                      class="quickbooks-textarea" 
                                      rows="3"
                                      placeholder="Any additional notes about the employee...">{{ old('notes', $employee->notes) }}</textarea>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray700 mb-4">Employee Status</label>
                            <div class="flex items-center space-x-6">
                                <div class="flex items-center">
                                    <input type="radio" id="active" name="is_active" value="1" {{ old('is_active', $employee->is_active) ? 'checked' : '' }} class="quickbooks-checkbox mr-2">
                                    <label for="active" class="text-sm text-gray600">Active</label>
                                </div>
                                <div class="flex items-center">
                                    <input type="radio" id="inactive" name="is_active" value="0" {{ !old('is_active', $employee->is_active) ? 'checked' : '' }} class="quickbooks-checkbox mr-2">
                                    <label for="inactive" class="text-sm text-gray600">Inactive</label>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Hidden earnings inputs will be added here by JavaScript -->
                    <div id="earningsData"></div>

                    <!-- Form Actions -->
                    <div class="border-t border-gray200 p-6">
                        <div class="flex justify-between items-center">
                            <div>
                                <a href="{{ route('admin.accounting.payroll.employees.show', $employee->id) }}" 
                                   class="quickbooks-button-secondary">
                                    Cancel
                                </a>
                            </div>
                            <div class="flex space-x-3">
                                <button type="submit" 
                                        class="quickbooks-button-primary">
                                    Update Employee
                                </button>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </main>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Get earning types from server
            let earningTypes = [];
            
            // Fetch earning types
            fetchEarningTypes();
            
            // Earnings management
            const earningsContainer = document.getElementById('earningsContainer');
            const noEarningsMessage = document.getElementById('noEarningsMessage');
            const addEarningRowBtn = document.getElementById('addEarningRowBtn');
            const earningsData = document.getElementById('earningsData');
            
            let earningCounter = 0;
            
            // Fetch earning types from server
            async function fetchEarningTypes() {
                try {
                    const response = await fetch('{{ route("admin.accounting.payroll.api.earning-types") }}');
                    if (response.ok) {
                        earningTypes = await response.json();
                        // Load existing earnings after fetching earning types
                        loadExistingEarnings();
                    }
                } catch (error) {
                    console.error('Failed to fetch earning types:', error);
                    // Still try to load existing earnings even if fetch fails
                    loadExistingEarnings();
                }
            }
            
            // Load existing earnings from server
            function loadExistingEarnings() {
                @foreach($employee->earnings as $index => $earning)
                    addEarningRow(
                        '{{ $earning->id }}',
                        '{{ $earning->earning_type_id }}',
                        '{{ $earning->earningType->name ?? "" }}',
                        '{{ $earning->earningType->code ?? "" }}',
                        '{{ $earning->earningType->type_label ?? "" }}',
                        {{ $earning->earningType->is_taxable ? 'true' : 'false' }},
                        {{ $earning->amount }},
                        '{{ $earning->frequency }}',
                        '{{ $earning->effective_date->format("Y-m-d") }}',
                        {{ $earning->is_taxable ? 'true' : 'false' }},
                        {{ $earning->is_recurring ? 'true' : 'false' }}
                    );
                @endforeach
                
                // If no earnings, show the empty message
                if (earningsContainer.children.length === 0) {
                    noEarningsMessage.classList.remove('hidden');
                }
            }
            
            // Add earning row button event
            addEarningRowBtn.addEventListener('click', function() {
                addEarningRow();
            });
            
            function addEarningRow(earningId = '', typeId = '', typeName = '', typeCode = '', typeLabel = '', isTaxableDefault = false, amount = '', frequency = '', effectiveDate = '', isTaxable = false, isRecurring = true) {
                // Hide "no earnings" message
                noEarningsMessage.classList.add('hidden');
                
                earningCounter++;
                const rowId = `earningRow${earningCounter}`;
                
                // Generate unique names for checkboxes
                const taxableCheckboxId = `taxable${earningCounter}`;
                const recurringCheckboxId = `recurring${earningCounter}`;
                const deleteCheckboxId = `delete${earningCounter}`;
                
                // Default values
                if (!frequency) frequency = 'monthly';
                if (!effectiveDate) effectiveDate = new Date().toISOString().split('T')[0];
                if (isTaxable === '') isTaxable = isTaxableDefault;
                if (isRecurring === '') isRecurring = true;
                
                // Build select options with proper selection
                const selectOptions = earningTypes.map(type => {
                    const isSelected = typeId && type.id == typeId;
                    return `
                        <option value="${type.id}" 
                                data-type-name="${type.name}"
                                data-type-code="${type.code}"
                                data-type-label="${type.type_label}"
                                data-is-taxable-default="${type.is_taxable}"
                                ${isSelected ? 'selected' : ''}>
                            ${type.name} (${type.code})
                        </option>
                    `;
                }).join('');
                
                const row = document.createElement('tr');
                row.id = rowId;
                if (earningId) {
                    row.dataset.earningId = earningId;
                }
                
                row.innerHTML = `
                    <td>
                        <select name="earnings[${earningCounter}][earning_type_id]" 
                                class="quickbooks-select w-full earning-type-select"
                                required
                                onchange="updateEarningTypeSelection(this, ${earningCounter})">
                            <option value="">Select Earning Type</option>
                            ${selectOptions}
                        </select>
                        <div class="mt-2" id="earningTypeDisplay${earningCounter}"></div>
                        <input type="hidden" name="earnings[${earningCounter}][earning_type_name]" value="${typeName}">
                        <input type="hidden" name="earnings[${earningCounter}][earning_type_code]" value="${typeCode}">
                        ${earningId ? `<input type="hidden" name="earnings[${earningCounter}][id]" value="${earningId}">` : ''}
                    </td>
                    <td>
                        <div class="relative">
                            <input type="number" 
                                   name="earnings[${earningCounter}][amount]" 
                                   class="quickbooks-input w-full pl-8 earning-amount"
                                   placeholder="0.00"
                                   step="0.01"
                                   min="0"
                                   value="${amount}"
                                   required
                                   oninput="validateEarningField(this); calculateEarningsTotal()">
                            <span class="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 text-sm">{{ $company->currency_symbol ?? 'K' }}</span>
                        </div>
                    </td>
                    <td>
                        <select name="earnings[${earningCounter}][frequency]" 
                                class="quickbooks-select w-full earning-frequency"
                                required
                                onchange="validateEarningField(this); calculateEarningsTotal()">
                            <option value="monthly" ${frequency === 'monthly' ? 'selected' : ''}>Monthly</option>
                            <option value="quarterly" ${frequency === 'quarterly' ? 'selected' : ''}>Quarterly</option>
                            <option value="annually" ${frequency === 'annually' ? 'selected' : ''}>Annually</option>
                            <option value="one-time" ${frequency === 'one-time' ? 'selected' : ''}>One Time</option>
                        </select>
                    </td>
                    <td>
                        <input type="date" 
                               name="earnings[${earningCounter}][effective_date]" 
                               class="quickbooks-input w-full earning-effective-date"
                               value="${effectiveDate}"
                               required
                               onchange="validateEarningField(this)">
                    </td>
                    <td class="text-center">
                        <div class="flex items-center justify-center">
                            <input type="checkbox" 
                                   id="${taxableCheckboxId}" 
                                   name="earnings[${earningCounter}][is_taxable]" 
                                   value="1"
                                   class="quickbooks-checkbox taxable-checkbox"
                                   ${isTaxable ? 'checked' : ''}
                                   onchange="calculateEarningsTotal()">
                            <label for="${taxableCheckboxId}" class="sr-only">Taxable</label>
                        </div>
                    </td>
                    <td class="text-center">
                        <div class="flex items-center justify-center">
                            <input type="checkbox" 
                                   id="${recurringCheckboxId}" 
                                   name="earnings[${earningCounter}][is_recurring]" 
                                   value="1"
                                   class="quickbooks-checkbox recurring-checkbox"
                                   ${isRecurring ? 'checked' : ''}
                                   onchange="calculateEarningsTotal()">
                            <label for="${recurringCheckboxId}" class="sr-only">Recurring</label>
                        </div>
                    </td>
                    <td class="text-center">
                        <div class="flex items-center justify-center space-x-2">
                            <input type="checkbox" 
                                   id="${deleteCheckboxId}" 
                                   name="earnings[${earningCounter}][_delete]" 
                                   value="1"
                                   class="quickbooks-checkbox delete-checkbox hidden"
                                   onchange="toggleRowDeletion(this, '${rowId}')">
                            <button type="button" 
                                    onclick="removeEarningRow('${rowId}', ${earningId ? 'true' : 'false'})"
                                    class="quickbooks-button-secondary p-2 text-danger hover:text-danger/90"
                                    title="Remove this earning">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                `;
                
                earningsContainer.appendChild(row);
                
                // If we have a typeId, we need to update the hidden inputs
                if (typeId) {
                    const selectElement = row.querySelector('.earning-type-select');
                    // Ensure the hidden inputs are updated
                    setTimeout(() => {
                        updateEarningTypeSelection(selectElement, earningCounter);
                    }, 100);
                }
                
                calculateEarningsTotal();
            }
            
            window.updateEarningTypeSelection = function(selectElement, rowNumber) {
                const selectedOption = selectElement.options[selectElement.selectedIndex];
                const typeNameInput = selectElement.closest('tr').querySelector('input[name$="[earning_type_name]"]');
                const typeCodeInput = selectElement.closest('tr').querySelector('input[name$="[earning_type_code]"]');
                const taxableCheckbox = selectElement.closest('tr').querySelector('.taxable-checkbox');
                
                if (selectedOption.value) {
                    const typeName = selectedOption.dataset.typeName;
                    const typeCode = selectedOption.dataset.typeCode;
                    const isTaxableDefault = selectedOption.dataset.isTaxableDefault === 'true';
                    
                    // Update hidden inputs
                    typeNameInput.value = typeName;
                    typeCodeInput.value = typeCode;
                    
                    // Update taxable checkbox based on earning type if not already manually set
                    const row = selectElement.closest('tr');
                    if (!row.classList.contains('removed-earning')) {
                        if (!taxableCheckbox.hasAttribute('data-manually-set')) {
                            taxableCheckbox.checked = isTaxableDefault;
                        }
                    }
                    
                    // Auto-fill amount for basic salary if empty
                    if ((selectedOption.dataset.typeLabel === 'Basic Salary' || typeName.toLowerCase().includes('basic')) && !row.classList.contains('removed-earning')) {
                        const amountInput = selectElement.closest('tr').querySelector('.earning-amount');
                        if (!amountInput.value || amountInput.value === '0' || amountInput.value === '') {
                            amountInput.value = '5000.00';
                            validateEarningField(amountInput);
                            calculateEarningsTotal();
                        }
                    }
                } else {
                    // Clear hidden inputs if no type selected
                    typeNameInput.value = '';
                    typeCodeInput.value = '';
                }
                
                if (!selectElement.closest('tr').classList.contains('removed-earning')) {
                    validateEarningField(selectElement);
                }
            };
            
            window.toggleRowDeletion = function(checkboxElement, rowId) {
                const row = document.getElementById(rowId);
                if (checkboxElement.checked) {
                    row.classList.add('removed-earning');
                    // Disable all inputs in the row
                    row.querySelectorAll('input, select').forEach(input => {
                        input.disabled = true;
                    });
                } else {
                    row.classList.remove('removed-earning');
                    // Re-enable all inputs in the row
                    row.querySelectorAll('input, select').forEach(input => {
                        input.disabled = false;
                    });
                }
                calculateEarningsTotal();
            };
            
            window.removeEarningRow = function(rowId, isExisting = false) {
                const row = document.getElementById(rowId);
                if (row) {
                    if (isExisting) {
                        // For existing earnings, show delete checkbox
                        const deleteCheckbox = row.querySelector('.delete-checkbox');
                        if (deleteCheckbox) {
                            deleteCheckbox.checked = !deleteCheckbox.checked;
                            toggleRowDeletion(deleteCheckbox, rowId);
                        }
                    } else {
                        // For new earnings, remove completely
                        row.remove();
                        calculateEarningsTotal();
                        if (earningsContainer.children.length === 0) {
                            noEarningsMessage.classList.remove('hidden');
                        }
                    }
                }
            };
            
            window.validateEarningField = function(field) {
                const row = field.closest('tr');
                if (row && row.classList.contains('removed-earning')) {
                    return;
                }
                
                field.classList.remove('is-invalid', 'is-valid');
                
                if (field.tagName === 'SELECT') {
                    if (field.value === '') {
                        field.classList.add('is-invalid');
                    } else {
                        field.classList.add('is-valid');
                    }
                } else if (field.type === 'number') {
                    const value = parseFloat(field.value);
                    if (isNaN(value) || value < 0) {
                        field.classList.add('is-invalid');
                    } else {
                        field.classList.add('is-valid');
                    }
                } else if (field.type === 'date') {
                    if (!field.value) {
                        field.classList.add('is-invalid');
                    } else {
                        field.classList.add('is-valid');
                    }
                } else if (field.type === 'checkbox') {
                    // Checkboxes are always valid
                    field.classList.add('is-valid');
                }
            };
            
            window.calculateEarningsTotal = function() {
                let totalGross = 0;
                let totalTaxable = 0;
                let totalNonTaxable = 0;
                
                document.querySelectorAll('tr[id^="earningRow"]:not(.removed-earning)').forEach(row => {
                    const amountInput = row.querySelector('.earning-amount');
                    const frequencySelect = row.querySelector('.earning-frequency');
                    const taxableCheckbox = row.querySelector('.taxable-checkbox');
                    const recurringCheckbox = row.querySelector('.recurring-checkbox');
                    
                    const amount = parseFloat(amountInput.value) || 0;
                    const frequency = frequencySelect.value;
                    const isTaxable = taxableCheckbox.checked;
                    const isRecurring = recurringCheckbox.checked;
                    
                    // Calculate monthly amount
                    let monthlyAmount = 0;
                    if (isRecurring) {
                        monthlyAmount = amount;
                        if (frequency === 'quarterly') monthlyAmount = amount / 3;
                        if (frequency === 'annually') monthlyAmount = amount / 12;
                        // One-time recurring? That doesn't make sense, so treat as monthly
                        if (frequency === 'one-time') monthlyAmount = amount;
                    }
                    // Non-recurring (one-time) earnings are not included in monthly total
                    
                    totalGross += monthlyAmount;
                    if (isTaxable) {
                        totalTaxable += monthlyAmount;
                    } else {
                        totalNonTaxable += monthlyAmount;
                    }
                });
                
                document.getElementById('totalGrossSalary').textContent = formatCurrency(totalGross);
                document.getElementById('totalTaxableEarnings').textContent = formatCurrency(totalTaxable);
                document.getElementById('totalNonTaxableEarnings').textContent = formatCurrency(totalNonTaxable);
            };
            
            function formatCurrency(amount) {
                return '{{ $company->currency_symbol ?? "K" }}' + amount.toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,');
            }
            
            // Payment method toggle
            const paymentMethodSelect = document.querySelector('select[name="payment_method"]');
            const bankFields = document.getElementById('bankFields');
            const bankAccountFields = document.getElementById('bankAccountFields');
            const bankBranchField = document.getElementById('bankBranchField');
            
            function toggleBankFields() {
                const isBank = paymentMethodSelect.value === 'bank';
                
                bankFields.style.display = isBank ? 'block' : 'none';
                bankAccountFields.style.display = isBank ? 'grid' : 'none';
                bankBranchField.style.display = isBank ? 'grid' : 'none';
                
                // Add/remove required attribute
                const bankInputs = document.querySelectorAll('#bankFields input, #bankAccountFields input, #bankBranchField input');
                bankInputs.forEach(input => {
                    if (isBank) {
                        input.setAttribute('required', 'required');
                    } else {
                        input.removeAttribute('required');
                    }
                });
            }
            
            if (paymentMethodSelect) {
                paymentMethodSelect.addEventListener('change', toggleBankFields);
                // Initial check
                toggleBankFields();
            }
            
            // Form validation
            const form = document.getElementById('employeeForm');
            form.addEventListener('submit', function(e) {
                let isValid = true;
                let errorMessage = '';
                
                // Validate required fields
                const requiredFields = form.querySelectorAll('[required]');
                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        isValid = false;
                        field.classList.add('is-invalid');
                        if (!errorMessage) {
                            errorMessage = 'Please fill in all required fields.';
                        }
                    } else {
                        field.classList.remove('is-invalid');
                    }
                });
                
                // Validate at least one earning (not marked for deletion)
                const hasEarnings = document.querySelectorAll('tr[id^="earningRow"]:not(.removed-earning)').length > 0;
                if (!hasEarnings) {
                    isValid = false;
                    errorMessage = 'Please add at least one earning for the employee.';
                }
                
                // Validate all earning rows have required fields
                document.querySelectorAll('.earning-type-select').forEach((select, index) => {
                    const row = select.closest('tr');
                    if (!row.classList.contains('removed-earning')) {
                        if (!select.value) {
                            isValid = false;
                            select.classList.add('is-invalid');
                            errorMessage = 'Please select an earning type for all rows.';
                        }
                        
                        const amountInput = row.querySelector('.earning-amount');
                        if (!amountInput.value || parseFloat(amountInput.value) <= 0) {
                            isValid = false;
                            amountInput.classList.add('is-invalid');
                            errorMessage = 'Please enter a valid amount for all earnings.';
                        }
                        
                        const dateInput = row.querySelector('.earning-effective-date');
                        if (!dateInput.value) {
                            isValid = false;
                            dateInput.classList.add('is-invalid');
                            errorMessage = 'Please enter an effective date for all earnings.';
                        }
                    }
                });
                
                // Validate email format
                const emailField = form.querySelector('input[type="email"]');
                if (emailField && emailField.value) {
                    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (!emailRegex.test(emailField.value)) {
                        isValid = false;
                        emailField.classList.add('is-invalid');
                        errorMessage = 'Please enter a valid email address.';
                    }
                }
                
                if (!isValid) {
                    e.preventDefault();
                    showNotification(errorMessage, 'error');
                    
                    // Scroll to first error
                    const firstError = document.querySelector('.is-invalid');
                    if (firstError) {
                        firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
                        firstError.focus();
                    }
                }
            });
            
            // Initial calculation
            calculateEarningsTotal();
        });
        
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'border-success bg-green-50',
                error: 'border-danger bg-red-50',
                info: 'border-primary bg-blue-50',
                warning: 'border-warning bg-yellow-50'
            };

            const icons = {
                success: 'fa-check-circle text-success',
                error: 'fa-exclamation-circle text-danger',
                info: 'fa-info-circle text-primary',
                warning: 'fa-exclamation-triangle text-warning'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 bg-white border-l-4 ${colors[type]} p-4 rounded shadow-lg z-50 flex items-center max-w-md animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-3 text-lg"></i>
                <span class="text-gray-900">${message}</span>
                <button onclick="this.parentElement.remove()" class="ml-auto text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times"></i>
                </button>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 5000);
        }

        // Add animation styles
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeIn {
                from { opacity: 0; transform: translateY(-10px); }
                to { opacity: 1; transform: translateY(0); }
            }
            @keyframes fadeOut {
                from { opacity: 1; transform: translateY(0); }
                to { opacity: 0; transform: translateY(-10px); }
            }
            .animate-fade-in {
                animation: fadeIn 0.3s ease-out;
            }
            .animate-fade-out {
                animation: fadeOut 0.3s ease-in;
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>