<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>NAPSA Settings | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .quickbooks-table { border-collapse: separate; border-spacing: 0; width: 100%; }
        .quickbooks-table th { background-color: #F3F2F1; color: #323130; font-weight: 600; font-size: 13px; text-transform: uppercase; padding: 12px 16px; border-bottom: 1px solid #EDEBE9; text-align: left; }
        .quickbooks-table td { padding: 12px 16px; border-bottom: 1px solid #EDEBE9; font-size: 14px; color: #323130; }
        .quickbooks-input { border: 1px solid #8C8C8C; border-radius: 4px; padding: 6px 12px; font-size: 14px; }
        .quickbooks-input:focus { outline: 2px solid #146CAA; outline-offset: -1px; border-color: #146CAA; }
        .quickbooks-badge { display: inline-flex; align-items: center; padding: 4px 8px; border-radius: 12px; font-size: 12px; font-weight: 600; }
        .modal-backdrop { background-color: rgba(0, 0, 0, 0.5); }
        .modal-content { max-height: 90vh; overflow-y: auto; }
        .modal-divider { border-top: 1px solid #EDEBE9; margin: 20px 0; }
        .modal-info { background-color: #F3F2F1; border: 1px solid #EDEBE9; border-radius: 4px; padding: 12px; margin-top: 8px; font-size: 13px; color: #605E5C; }
        .modal-info i { color: #605E5C; margin-right: 8px; }
        .calculation-card { background: linear-gradient(135deg, #E1F0FA 0%, #F3F2F1 100%); border-radius: 8px; padding: 16px; }
        .calculation-value { font-size: 24px; font-weight: bold; color: #146CAA; }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'NAPSA Settings', 'subtitle' => 'Configure NAPSA contribution rates and ceilings'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Current Settings Info -->
            <div class="mb-8 bg-white rounded-lg border border-gray200 p-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between">
                    <div>
                        <h2 class="text-xl font-semibold text-gray900">{{ $company->name }}</h2>
                        <p class="text-gray600 text-sm mt-1">
                            Current Financial Year: 
                            <span class="font-medium">{{ $currentFinancialYear->year ?? 'Not Set' }}</span>
                            @if($currentFinancialYear)
                                <span class="text-gray500 ml-2">
                                    ({{ $currentFinancialYear->start_date->format('M d, Y') }} - {{ $currentFinancialYear->end_date->format('M d, Y') }})
                                </span>
                            @endif
                        </p>
                    </div>
                    <div class="mt-4 md:mt-0">
                        @if($napsaSettings)
                            <span class="quickbooks-badge bg-green-100 text-green-800">Configured</span>
                        @else
                            <span class="quickbooks-badge bg-yellow-100 text-yellow-800">Not Configured</span>
                        @endif
                    </div>
                </div>
            </div>

            <!-- NAPSA Configuration Form -->
            <div class="bg-white rounded-lg border border-gray200 p-6">
                <form id="napsaForm" action="{{ route('admin.accounting.payroll.napsa.store') }}" method="POST">
                    @csrf
                    
                    <div class="mb-8">
                        <h3 class="text-lg font-semibold text-gray900 mb-6">NAPSA Contribution Settings</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Financial Year
                                    <span class="text-danger">*</span>
                                </label>
                                <select id="financial_year_id" name="financial_year_id" class="quickbooks-input w-full" required onchange="loadNapsaSettings()">
                                    <option value="">Select Financial Year</option>
                                    @foreach($financialYears as $year)
                                        <option value="{{ $year->id }}" 
                                                {{ ($napsaSettings && $napsaSettings->financial_year_id == $year->id) ? 'selected' : '' }}
                                                {{ ($currentFinancialYear && $year->id == $currentFinancialYear->id && !$napsaSettings) ? 'selected' : '' }}
                                                data-start-date="{{ $year->start_date->format('Y-m-d') }}"
                                                data-end-date="{{ $year->end_date->format('Y-m-d') }}">
                                            {{ $year->year }} ({{ $year->start_date->format('M Y') }} - {{ $year->end_date->format('M Y') }})
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Year Label
                                    <span class="text-danger">*</span>
                                </label>
                                <input type="text" 
                                       id="year" 
                                       name="year" 
                                       class="quickbooks-input w-full" 
                                       placeholder="e.g., 2025"
                                       value="{{ $napsaSettings->year ?? '' }}"
                                       required>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Effective From
                                    <span class="text-danger">*</span>
                                </label>
                                <input type="date" 
                                       id="start_date" 
                                       name="start_date" 
                                       class="quickbooks-input w-full" 
                                       value="{{ $napsaSettings->start_date ?? '' }}"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Effective To
                                    <span class="text-danger">*</span>
                                </label>
                                <input type="date" 
                                       id="end_date" 
                                       name="end_date" 
                                       class="quickbooks-input w-full" 
                                       value="{{ $napsaSettings->end_date ?? '' }}"
                                       required>
                            </div>
                        </div>
                    </div>

                    <div class="modal-divider"></div>

                    <!-- NAPSA Rates Configuration -->
                    <div class="mb-8">
                        <h3 class="text-lg font-semibold text-gray900 mb-6">NAPSA Contribution Rates</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Monthly Contribution Rate (%)
                                    <span class="text-danger">*</span>
                                </label>
                                <div class="relative">
                                    <input type="number" 
                                           id="monthly_rate" 
                                           name="monthly_rate" 
                                           class="quickbooks-input w-full pr-8" 
                                           step="0.01" 
                                           min="0" 
                                           max="100" 
                                           value="{{ $napsaSettings->monthly_rate ?? '5.00' }}"
                                           required
                                           oninput="updateCalculations()">
                                    <span class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray500">%</span>
                                </div>
                                <p class="text-sm text-gray500 mt-2">
                                    <i class="fas fa-info-circle mr-1"></i>
                                    Current Zambian NAPSA rate: 5% of pensionable emoluments
                                </p>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Monthly Ceiling (ZMW)
                                    <span class="text-danger">*</span>
                                </label>
                                <div class="relative">
                                    <span class="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray500">K</span>
                                    <input type="number" 
                                           id="monthly_ceiling" 
                                           name="monthly_ceiling" 
                                           class="quickbooks-input w-full pl-8" 
                                           step="0.01" 
                                           min="0" 
                                           value="{{ $napsaSettings->monthly_ceiling ?? '6300.00' }}"
                                           required
                                           oninput="updateCalculations()">
                                </div>
                                <p class="text-sm text-gray500 mt-2">
                                    <i class="fas fa-info-circle mr-1"></i>
                                    Current ceiling: K6,300 per month
                                </p>
                            </div>
                        </div>
                        
                        <div class="modal-info mt-4">
                            <i class="fas fa-calculator"></i>
                            <div>
                                <p class="font-medium mb-1">NAPSA Contribution Calculation:</p>
                                <p class="text-sm">Monthly NAPSA = Min(Pensionable Emoluments, Monthly Ceiling) × Monthly Rate</p>
                                <p class="text-sm mt-1">Example: For salary of K8,000: Min(K8,000, K6,300) × 5% = K6,300 × 5% = K315</p>
                            </div>
                        </div>
                    </div>

                    <!-- Calculations Display -->
                    <div class="mb-8">
                        <h3 class="text-lg font-semibold text-gray900 mb-4">Calculations Preview</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div class="calculation-card">
                                <div class="text-sm text-gray600 mb-1">Monthly Contribution</div>
                                <div class="calculation-value" id="monthlyContribution">K0.00</div>
                                <div class="text-xs text-gray500 mt-1">Per employee/month</div>
                            </div>
                            
                            <div class="calculation-card">
                                <div class="text-sm text-gray600 mb-1">Annual Contribution</div>
                                <div class="calculation-value" id="annualContribution">K0.00</div>
                                <div class="text-xs text-gray500 mt-1">Per employee/year</div>
                            </div>
                            
                            <div class="calculation-card">
                                <div class="text-sm text-gray600 mb-1">Annual Ceiling</div>
                                <div class="calculation-value" id="annualCeiling">K0.00</div>
                                <div class="text-xs text-gray500 mt-1">Maximum per year</div>
                            </div>
                        </div>
                    </div>

                    <!-- Test Calculator -->
                    <div class="mb-8 bg-gray50 border border-gray200 rounded-lg p-6">
                        <h3 class="text-lg font-semibold text-gray900 mb-4">Test Calculator</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Test Salary (ZMW)</label>
                                <div class="relative">
                                    <span class="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray500">K</span>
                                    <input type="number" 
                                           id="testSalary" 
                                           class="quickbooks-input w-full pl-8" 
                                           placeholder="Enter salary amount"
                                           step="0.01"
                                           min="0"
                                           oninput="calculateTestNAPSA()">
                                </div>
                            </div>
                            
                            <div>
                                <div id="testResult" class="hidden">
                                    <div class="text-sm text-gray600 mb-1">NAPSA Contribution</div>
                                    <div class="text-xl font-semibold text-blue-600" id="testContribution">K0.00</div>
                                    <div class="text-xs text-gray500 mt-1" id="testCalculation"></div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="modal-divider"></div>

                    <!-- Form Actions -->
                    <div class="flex justify-between items-center">
                        <div class="flex space-x-3">
                            <a href="{{ route('admin.accounting.payroll.index') }}" 
                               class="text-primary hover:text-primary/90 font-medium text-sm">
                                <i class="fas fa-arrow-left mr-1"></i>Back to Payroll
                            </a>
                            <button type="button" 
                                    onclick="resetToDefaults()" 
                                    class="text-gray-600 hover:text-gray-800 font-medium text-sm">
                                <i class="fas fa-undo mr-1"></i>Reset to Defaults
                            </button>
                        </div>
                        <div class="flex space-x-3">
                            <button type="button" 
                                    onclick="validateAndSubmit()" 
                                    class="px-4 py-2 bg-primary text-white rounded text-sm font-medium hover:bg-primary/90">
                                <i class="fas fa-save mr-2"></i>Save NAPSA Settings
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Current NAPSA Settings Display -->
            @if($napsaSettings)
                <div class="mt-8 bg-white rounded-lg border border-gray200">
                    <div class="border-b border-gray200 p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <h3 class="text-lg font-semibold text-gray900">Current NAPSA Settings for {{ $napsaSettings->year }}</h3>
                                <p class="text-gray600 text-sm">Effective: {{ $napsaSettings->start_date->format('M d, Y') }} to {{ $napsaSettings->end_date->format('M d, Y') }}</p>
                            </div>
                            <div class="flex space-x-2">
                                <button onclick="printSettings()" 
                                        class="text-primary hover:text-primary/90 text-sm font-medium flex items-center">
                                    <i class="fas fa-print mr-1"></i> Print
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="p-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                            <div class="bg-gray50 rounded-lg p-6">
                                <h4 class="text-lg font-semibold text-gray900 mb-4">Monthly Contributions</h4>
                                <div class="space-y-4">
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray600">Contribution Rate:</span>
                                        <span class="font-semibold text-gray900">{{ $napsaSettings->monthly_rate }}%</span>
                                    </div>
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray600">Monthly Ceiling:</span>
                                        <span class="font-semibold text-gray900">K{{ number_format($napsaSettings->monthly_ceiling, 2) }}</span>
                                    </div>
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray600">Maximum Monthly Contribution:</span>
                                        <span class="font-semibold text-blue-600">
                                            K{{ number_format(($napsaSettings->monthly_ceiling * $napsaSettings->monthly_rate) / 100, 2) }}
                                        </span>
                                    </div>
                                </div>
                            </div>

                            <div class="bg-gray50 rounded-lg p-6">
                                <h4 class="text-lg font-semibold text-gray900 mb-4">Annual Contributions</h4>
                                <div class="space-y-4">
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray600">Annual Rate:</span>
                                        <span class="font-semibold text-gray900">{{ $napsaSettings->annual_rate }}%</span>
                                    </div>
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray600">Annual Ceiling:</span>
                                        <span class="font-semibold text-gray900">K{{ number_format($napsaSettings->annual_ceiling, 2) }}</span>
                                    </div>
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray600">Maximum Annual Contribution:</span>
                                        <span class="font-semibold text-blue-600">
                                            K{{ number_format(($napsaSettings->annual_ceiling * $napsaSettings->annual_rate) / 100, 2) }}
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mt-6 modal-info">
                            <i class="fas fa-info-circle"></i>
                            <div>
                                <p class="font-medium mb-1">Last Updated:</p>
                                <p class="text-sm">
                                    @if($napsaSettings->updated_by && $napsaSettings->updater)
                                        By {{ $napsaSettings->updater->name }} on {{ $napsaSettings->updated_at->format('M d, Y \a\t h:i A') }}
                                    @elseif($napsaSettings->created_by && $napsaSettings->creator)
                                        By {{ $napsaSettings->creator->name }} on {{ $napsaSettings->created_at->format('M d, Y \a\t h:i A') }}
                                    @endif
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            @endif
        </main>
    </div>

    <script>
        // Load NAPSA settings when page loads
        document.addEventListener('DOMContentLoaded', function() {
            loadNapsaSettings();
            
            // Set default dates if not set
            const startDateInput = document.getElementById('start_date');
            const endDateInput = document.getElementById('end_date');
            
            if (startDateInput && !startDateInput.value) {
                startDateInput.value = '{{ date("Y") }}-01-01';
            }
            
            if (endDateInput && !endDateInput.value) {
                endDateInput.value = '{{ date("Y") }}-12-31';
            }
            
            // Initial calculations
            updateCalculations();
        });

        // Load NAPSA settings for selected financial year
        function loadNapsaSettings() {
            const yearSelect = document.getElementById('financial_year_id');
            const yearId = yearSelect.value;
            
            if (!yearId) return;
            
            // Get selected option data
            const selectedOption = yearSelect.options[yearSelect.selectedIndex];
            const yearText = selectedOption.text.split(' ')[0];
            const startDate = selectedOption.getAttribute('data-start-date');
            const endDate = selectedOption.getAttribute('data-end-date');
            
            // Update year and dates
            document.getElementById('year').value = yearText;
            document.getElementById('start_date').value = startDate;
            document.getElementById('end_date').value = endDate;
            
            // Fetch settings from server
            fetch(`{{ route('admin.accounting.payroll.get-settings') }}?financial_year_id=${yearId}&setting_type=napsa`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.setting) {
                        // Load existing settings
                        document.getElementById('monthly_rate').value = data.setting.monthly_rate;
                        document.getElementById('monthly_ceiling').value = data.setting.monthly_ceiling;
                        
                        updateCalculations();
                        showNotification('Loaded existing NAPSA settings', 'success');
                    } else {
                        // Set default values
                        resetToDefaults();
                        showNotification('Loaded default NAPSA settings', 'info');
                    }
                })
                .catch(error => {
                    console.error('Error loading settings:', error);
                    resetToDefaults();
                    showNotification('Loaded default NAPSA settings', 'info');
                });
        }

        // Update calculations based on input values
        function updateCalculations() {
            const monthlyRate = parseFloat(document.getElementById('monthly_rate').value) || 0;
            const monthlyCeiling = parseFloat(document.getElementById('monthly_ceiling').value) || 0;
            
            // Calculate monthly contribution
            const monthlyContribution = (monthlyCeiling * monthlyRate) / 100;
            document.getElementById('monthlyContribution').textContent = 'K' + monthlyContribution.toFixed(2);
            
            // Calculate annual values
            const annualCeiling = monthlyCeiling * 12;
            const annualContribution = monthlyContribution * 12;
            
            document.getElementById('annualContribution').textContent = 'K' + annualContribution.toFixed(2);
            document.getElementById('annualCeiling').textContent = 'K' + annualCeiling.toFixed(2);
            
            // Update test calculation if salary is entered
            calculateTestNAPSA();
        }

        // Calculate test NAPSA contribution
        function calculateTestNAPSA() {
            const testSalary = parseFloat(document.getElementById('testSalary').value) || 0;
            const monthlyRate = parseFloat(document.getElementById('monthly_rate').value) || 0;
            const monthlyCeiling = parseFloat(document.getElementById('monthly_ceiling').value) || 0;
            
            if (testSalary > 0) {
                const contributionBase = Math.min(testSalary, monthlyCeiling);
                const contribution = (contributionBase * monthlyRate) / 100;
                
                document.getElementById('testContribution').textContent = 'K' + contribution.toFixed(2);
                document.getElementById('testCalculation').textContent = 
                    `Min(K${testSalary.toFixed(2)}, K${monthlyCeiling.toFixed(2)}) × ${monthlyRate}% = K${contribution.toFixed(2)}`;
                
                document.getElementById('testResult').classList.remove('hidden');
            } else {
                document.getElementById('testResult').classList.add('hidden');
            }
        }

        // Reset to default Zambian NAPSA rates
        function resetToDefaults() {
            if (confirm('Reset to default Zambian NAPSA rates?')) {
                document.getElementById('monthly_rate').value = '5.00';
                document.getElementById('monthly_ceiling').value = '6300.00';
                updateCalculations();
                showNotification('Reset to default Zambian NAPSA rates', 'info');
            }
        }

        // Validate and submit form
        function validateAndSubmit() {
            const monthlyRate = parseFloat(document.getElementById('monthly_rate').value) || 0;
            const monthlyCeiling = parseFloat(document.getElementById('monthly_ceiling').value) || 0;
            
            if (monthlyRate < 0 || monthlyRate > 100) {
                showNotification('Monthly rate must be between 0% and 100%', 'error');
                return;
            }
            
            if (monthlyCeiling <= 0) {
                showNotification('Monthly ceiling must be greater than 0', 'error');
                return;
            }
            
            // Show confirmation
            if (confirm('Save NAPSA settings for this financial year?')) {
                document.getElementById('napsaForm').submit();
            }
        }

        // Print settings
        function printSettings() {
            const year = document.getElementById('year').value || 'Current';
            const monthlyRate = document.getElementById('monthly_rate').value || '0.00';
            const monthlyCeiling = document.getElementById('monthly_ceiling').value || '0.00';
            const monthlyContribution = (parseFloat(monthlyCeiling) * parseFloat(monthlyRate) / 100).toFixed(2);
            
            const printContent = `
                <!DOCTYPE html>
                <html>
                <head>
                    <title>NAPSA Settings - ${year}</title>
                    <style>
                        body { font-family: Arial, sans-serif; margin: 20px; }
                        h1 { color: #146CAA; margin-bottom: 5px; }
                        .header { text-align: center; margin-bottom: 30px; }
                        .section { margin-bottom: 20px; }
                        .section-title { font-weight: bold; color: #323130; margin-bottom: 10px; }
                        .calculation { background: #F3F2F1; padding: 15px; border-radius: 8px; margin: 10px 0; }
                        .calculation-value { font-size: 24px; font-weight: bold; color: #146CAA; }
                        .footer { margin-top: 40px; font-size: 11px; color: #666; text-align: center; }
                        @media print {
                            .no-print { display: none; }
                        }
                    </style>
                </head>
                <body>
                    <div class="header">
                        <h1>NAPSA Settings</h1>
                        <p>Financial Year: ${document.getElementById('financial_year_id').selectedOptions[0]?.text || ''}</p>
                        <p>Year: ${year}</p>
                        <p>Effective: ${document.getElementById('start_date').value} to ${document.getElementById('end_date').value}</p>
                        <p>Generated on: ${new Date().toLocaleDateString()}</p>
                    </div>
                    
                    <div class="section">
                        <div class="section-title">Contribution Rates</div>
                        <table style="width: 100%; border-collapse: collapse;">
                            <tr>
                                <td style="padding: 8px; border-bottom: 1px solid #ddd;">Monthly Contribution Rate:</td>
                                <td style="padding: 8px; border-bottom: 1px solid #ddd; font-weight: bold;">${monthlyRate}%</td>
                            </tr>
                            <tr>
                                <td style="padding: 8px; border-bottom: 1px solid #ddd;">Monthly Ceiling:</td>
                                <td style="padding: 8px; border-bottom: 1px solid #ddd; font-weight: bold;">K${parseFloat(monthlyCeiling).toFixed(2)}</td>
                            </tr>
                        </table>
                    </div>
                    
                    <div class="section">
                        <div class="section-title">Calculations</div>
                        <div class="calculation">
                            <div>Maximum Monthly Contribution:</div>
                            <div class="calculation-value">K${monthlyContribution}</div>
                            <div style="font-size: 12px; color: #666;">(K${monthlyCeiling} × ${monthlyRate}%)</div>
                        </div>
                        <div class="calculation">
                            <div>Annual Ceiling:</div>
                            <div class="calculation-value">K${(parseFloat(monthlyCeiling) * 12).toFixed(2)}</div>
                        </div>
                        <div class="calculation">
                            <div>Maximum Annual Contribution:</div>
                            <div class="calculation-value">K${(parseFloat(monthlyContribution) * 12).toFixed(2)}</div>
                        </div>
                    </div>
                    
                    <div class="section">
                        <div class="section-title">Calculation Formula</div>
                        <div style="background: #E1F0FA; padding: 15px; border-radius: 8px; font-size: 14px;">
                            <p><strong>Monthly NAPSA = Min(Pensionable Emoluments, Monthly Ceiling) × Monthly Rate</strong></p>
                            <p>Example: For salary of K8,000 with ceiling of K6,300 at 5% rate:</p>
                            <p>Min(K8,000, K6,300) × 5% = K6,300 × 5% = K315</p>
                        </div>
                    </div>
                    
                    <div class="footer">
                        <p>Generated by IMMIA FINANCE Payroll System</p>
                        <p>This is an official document</p>
                    </div>
                    
                    <div class="no-print" style="margin-top: 20px; text-align: center;">
                        <button onclick="window.print()" style="padding: 10px 20px; background: #146CAA; color: white; border: none; border-radius: 4px; cursor: pointer;">
                            Print Document
                        </button>
                        <button onclick="window.close()" style="padding: 10px 20px; background: #666; color: white; border: none; border-radius: 4px; cursor: pointer; margin-left: 10px;">
                            Close
                        </button>
                    </div>
                </body>
                </html>
            `;
            
            const printWindow = window.open('', '_blank');
            printWindow.document.write(printContent);
            printWindow.document.close();
        }

        // Utility functions
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-primary',
                warning: 'bg-yellow-500'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Add animation styles
        if (!document.querySelector('style[data-napsa]')) {
            const style = document.createElement('style');
            style.setAttribute('data-napsa', 'true');
            style.textContent = `
                @keyframes fadeIn {
                    from { opacity: 0; transform: translateY(-10px); }
                    to { opacity: 1; transform: translateY(0); }
                }
                @keyframes fadeOut {
                    from { opacity: 1; transform: translateY(0); }
                    to { opacity: 0; transform: translateY(-10px); }
                }
                .animate-fade-in {
                    animation: fadeIn 0.3s ease-out;
                }
                .animate-fade-out {
                    animation: fadeOut 0.3s ease-in;
                }
            `;
            document.head.appendChild(style);
        }
    </script>
</body>
</html>