<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payroll Periods | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .quickbooks-table { border-collapse: separate; border-spacing: 0; width: 100%; }
        .quickbooks-table th { background-color: #F3F2F1; color: #323130; font-weight: 600; font-size: 13px; text-transform: uppercase; padding: 12px 16px; border-bottom: 1px solid #EDEBE9; text-align: left; }
        .quickbooks-table td { padding: 12px 16px; border-bottom: 1px solid #EDEBE9; font-size: 14px; color: #323130; }
        .quickbooks-table tbody tr:hover { background-color: #F3F2F1; }
        .quickbooks-input { border: 1px solid #8C8C8C; border-radius: 4px; padding: 6px 12px; font-size: 14px; }
        .quickbooks-input:focus { outline: 2px solid #146CAA; outline-offset: -1px; border-color: #146CAA; }
        .quickbooks-badge { display: inline-flex; align-items: center; padding: 4px 8px; border-radius: 12px; font-size: 12px; font-weight: 600; }
        .action-btn { width: 32px; height: 32px; border-radius: 4px; display: inline-flex; align-items: center; justify-content: center; margin: 0 2px; transition: all 0.2s; }
        .action-btn:hover { transform: translateY(-1px); }
        .action-btn-edit { background-color: #E1F0FA; color: #146CAA; }
        .action-btn-edit:hover { background-color: #146CAA; color: white; }
        .action-btn-delete { background-color: #FEE; color: #D83B01; }
        .action-btn-delete:hover { background-color: #D83B01; color: white; }
        .action-btn-view { background-color: #F3F2F1; color: #323130; }
        .action-btn-view:hover { background-color: #146CAA; color: white; }
    </style>
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Payroll Periods', 'subtitle' => 'Manage monthly payroll periods'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Action Buttons -->
            <div class="mb-8 bg-white rounded-lg border border-gray-200 p-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between">
                    <div>
                        <h2 class="text-xl font-semibold text-gray-900">Monthly Payroll Periods</h2>
                        <p class="text-gray-600 text-sm mt-1">
                            Current Financial Year: 
                            <span class="font-medium">{{ $currentFinancialYear->year ?? 'Not Set' }}</span>
                            @if($currentFinancialYear && $currentFinancialYear->start_date && $currentFinancialYear->end_date)
                                <span class="text-gray-500 ml-2">
                                    ({{ $currentFinancialYear->start_date->format('M d, Y') }} - {{ $currentFinancialYear->end_date->format('M d, Y') }})
                                </span>
                            @endif
                        </p>
                    </div>
                    <div class="mt-4 md:mt-0 flex space-x-3">
                        <button onclick="openCreatePeriodModal()" 
                                class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center">
                            <i class="fas fa-plus mr-2"></i>Create Period
                        </button>
                        <form action="{{ route('admin.accounting.payroll.periods.generate') }}" method="POST" class="inline">
                            @csrf
                            <input type="hidden" name="financial_year_id" value="{{ $currentFinancialYear->id ?? '' }}">
                            <button type="submit" 
                                    class="bg-green-600 text-white px-4 py-2 rounded text-sm font-medium hover:bg-green-700 flex items-center"
                                    {{ !$currentFinancialYear ? 'disabled' : '' }}
                                    onclick="return confirmGeneratePeriods('{{ $currentFinancialYear?->start_date?->format('M Y') }}', '{{ $currentFinancialYear?->end_date?->format('M Y') }}')">
                                <i class="fas fa-calendar-plus mr-2"></i>Generate All Months
                            </button>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Payroll Periods Table -->
            <div class="bg-white rounded-lg border border-gray-200">
                <div class="overflow-x-auto">
                    <table class="quickbooks-table">
                        <thead>
                            <tr>
                                <th>Period Name</th>
                                <th>Month</th>
                                <th>Date Range</th>
                                <th>Cutoff Date</th>
                                <th>Payment Date</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($payrollPeriods ?? [] as $period)
                                <tr class="hover:bg-gray-100 transition-colors">
                                    <td class="font-medium text-gray-900">{{ $period->period_name ?? 'N/A' }}</td>
                                    <td>
                                        <span class="quickbooks-badge bg-blue-100 text-blue-800">
                                            {{ $period->month_name ?? 'N/A' }}
                                        </span>
                                    </td>
                                    <td>{{ $period->formatted_date_range ?? 'N/A' }}</td>
                                    <td class="text-gray-700">{{ $period->cutoff_date ? $period->cutoff_date->format('M d, Y') : 'N/A' }}</td>
                                    <td class="text-gray-700">{{ $period->payment_date ? $period->payment_date->format('M d, Y') : 'N/A' }}</td>
                                    <td>{!! $period->status_badge ?? '<span class="text-gray-500">N/A</span>' !!}</td>
                                    <td>
                                        <div class="flex items-center">
                                            <button onclick="editPeriod({{ $period->id }})" 
                                                    class="action-btn action-btn-edit"
                                                    title="Edit period">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            @if($period->canProcessPayroll())
                                                <button onclick="processPayroll({{ $period->id }})" 
                                                        class="action-btn action-btn-view ml-2"
                                                        title="Process payroll">
                                                    <i class="fas fa-cogs"></i>
                                                </button>
                                            @endif
                                            @if($period->status === 'closed')
                                                <button onclick="reopenPeriod({{ $period->id }})" 
                                                        class="action-btn action-btn-view ml-2 bg-green-100 text-green-600 hover:bg-green-600 hover:text-white"
                                                        title="Reopen period">
                                                    <i class="fas fa-redo"></i>
                                                </button>
                                            @endif
                                            @if($period->status === 'open')
                                                <button onclick="closePeriod({{ $period->id }})" 
                                                        class="action-btn action-btn-view ml-2 bg-red-100 text-red-600 hover:bg-red-600 hover:text-white"
                                                        title="Close period">
                                                    <i class="fas fa-lock"></i>
                                                </button>
                                            @endif
                                            <button onclick="deletePeriod({{ $period->id }})" 
                                                    class="action-btn action-btn-delete ml-2"
                                                    title="Delete period">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="7" class="text-center py-12">
                                        <div class="py-8 text-center">
                                            <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                                <i class="fas fa-calendar text-gray-500 text-2xl"></i>
                                            </div>
                                            <h3 class="text-lg font-medium text-gray-900 mb-2">No payroll periods</h3>
                                            <p class="text-gray-600 mb-6 max-w-md mx-auto">
                                                Create your first monthly payroll period or generate all months for the financial year.
                                            </p>
                                            <div class="space-x-3">
                                                <button onclick="openCreatePeriodModal()" 
                                                        class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90">
                                                    <i class="fas fa-plus mr-2"></i>Create Period
                                                </button>
                                                @if($currentFinancialYear)
                                                    <form action="{{ route('admin.accounting.payroll.periods.generate') }}" method="POST" class="inline">
                                                        @csrf
                                                        <input type="hidden" name="financial_year_id" value="{{ $currentFinancialYear->id }}">
                                                        <button type="submit" 
                                                                class="bg-green-600 text-white px-4 py-2 rounded text-sm font-medium hover:bg-green-700"
                                                                onclick="return confirmGeneratePeriods('{{ $currentFinancialYear->start_date->format('M Y') }}', '{{ $currentFinancialYear->end_date->format('M Y') }}')">
                                                            <i class="fas fa-calendar-plus mr-2"></i>Generate All Months
                                                        </button>
                                                    </form>
                                                @endif
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                @if($payrollPeriods && $payrollPeriods->hasPages())
                    <div class="border-t border-gray-200 p-4">
                        {{ $payrollPeriods->links() }}
                    </div>
                @endif
            </div>
        </main>
    </div>

    <!-- Create Period Modal -->
    <div id="createPeriodModal" class="fixed inset-0 bg-gray-900 bg-opacity-50 hidden items-center justify-center z-50 p-4 overflow-y-auto">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="border-b border-gray-200 p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <h2 class="text-xl font-semibold text-gray-900">Create Monthly Payroll Period</h2>
                        <p class="text-gray-600 text-sm">Add a new monthly payroll period</p>
                    </div>
                    <button onclick="closeCreatePeriodModal()" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-times text-lg"></i>
                    </button>
                </div>
            </div>

            <div class="p-6">
                <form id="createPeriodForm" action="{{ route('admin.accounting.payroll.periods.store') }}" method="POST">
                    @csrf
                    
                    <div class="space-y-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Financial Year</label>
                            <select name="financial_year_id" class="quickbooks-input w-full" required>
                                <option value="">Select Financial Year</option>
                                @foreach($financialYears ?? [] as $year)
                                    <option value="{{ $year->id }}" {{ ($currentFinancialYear && $year->id == $currentFinancialYear->id) ? 'selected' : '' }}>
                                        {{ $year->year }} ({{ $year->start_date->format('M Y') }} - {{ $year->end_date->format('M Y') }})
                                    </option>
                                @endforeach
                            </select>
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Period Name</label>
                            <input type="text" 
                                   name="period_name" 
                                   class="quickbooks-input w-full" 
                                   placeholder="e.g., January 2025 Payroll"
                                   required>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Start Date</label>
                                <input type="date" 
                                       name="start_date" 
                                       class="quickbooks-input w-full" 
                                       id="periodStartDate"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">End Date</label>
                                <input type="date" 
                                       name="end_date" 
                                       class="quickbooks-input w-full" 
                                       id="periodEndDate"
                                       required>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Cutoff Date</label>
                                <input type="date" 
                                       name="cutoff_date" 
                                       class="quickbooks-input w-full" 
                                       id="periodCutoffDate"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Payment Date</label>
                                <input type="date" 
                                       name="payment_date" 
                                       class="quickbooks-input w-full" 
                                       id="periodPaymentDate"
                                       required>
                            </div>
                        </div>
                    </div>

                    <div class="border-t border-gray-200 pt-6 mt-6">
                        <div class="flex justify-end space-x-3">
                            <button type="button" 
                                    onclick="closeCreatePeriodModal()" 
                                    class="px-4 py-2 border border-gray-300 rounded text-sm font-medium text-gray-700 hover:bg-gray-50">
                                Cancel
                            </button>
                            <button type="submit" 
                                    class="px-4 py-2 bg-primary text-white rounded text-sm font-medium hover:bg-primary/90">
                                Create Period
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Period Modal -->
    <div id="editPeriodModal" class="fixed inset-0 bg-gray-900 bg-opacity-50 hidden items-center justify-center z-50 p-4 overflow-y-auto">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="border-b border-gray-200 p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <h2 class="text-xl font-semibold text-gray-900">Edit Payroll Period</h2>
                        <p class="text-gray-600 text-sm">Update payroll period details</p>
                    </div>
                    <button onclick="closeEditPeriodModal()" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-times text-lg"></i>
                    </button>
                </div>
            </div>

            <div class="p-6">
                <form id="editPeriodForm">
                    @csrf
                    @method('PUT')
                    
                    <div class="space-y-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Period Name</label>
                            <input type="text" 
                                   id="editPeriodName" 
                                   name="period_name" 
                                   class="quickbooks-input w-full" 
                                   required>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Start Date</label>
                                <input type="date" 
                                       id="editStartDate" 
                                       name="start_date" 
                                       class="quickbooks-input w-full" 
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">End Date</label>
                                <input type="date" 
                                       id="editEndDate" 
                                       name="end_date" 
                                       class="quickbooks-input w-full" 
                                       required>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Cutoff Date</label>
                                <input type="date" 
                                       id="editCutoffDate" 
                                       name="cutoff_date" 
                                       class="quickbooks-input w-full" 
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Payment Date</label>
                                <input type="date" 
                                       id="editPaymentDate" 
                                       name="payment_date" 
                                       class="quickbooks-input w-full" 
                                       required>
                            </div>
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                            <select id="editStatus" name="status" class="quickbooks-input w-full" required>
                                <option value="open">Open</option>
                                <option value="closed">Closed</option>
                                <option value="processing">Processing</option>
                            </select>
                        </div>
                    </div>

                    <div class="border-t border-gray-200 pt-6 mt-6">
                        <div class="flex justify-end space-x-3">
                            <button type="button" 
                                    onclick="closeEditPeriodModal()" 
                                    class="px-4 py-2 border border-gray-300 rounded text-sm font-medium text-gray-700 hover:bg-gray-50">
                                Cancel
                            </button>
                            <button type="submit" 
                                    class="px-4 py-2 bg-primary text-white rounded text-sm font-medium hover:bg-primary/90">
                                Update Period
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="deleteModal" class="fixed inset-0 bg-gray-900 bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="w-10 h-10 bg-red-100 rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-exclamation-triangle text-danger"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray-900" id="deleteModalTitle">Delete Payroll Period</h3>
                        <p class="text-gray-600 text-sm" id="deleteModalMessage">Are you sure you want to delete this payroll period?</p>
                    </div>
                </div>
            </div>
            
            <div class="border-t border-gray-200 p-6 flex justify-end space-x-3">
                <button onclick="closeDeleteModal()" class="px-4 py-2 text-gray-700 hover:bg-gray-100 rounded text-sm font-medium">
                    Cancel
                </button>
                <form id="deleteForm" method="POST" style="display: none;">
                    @csrf
                    @method('DELETE')
                </form>
                <button onclick="confirmDelete()" class="bg-danger text-white px-4 py-2 rounded text-sm font-medium hover:bg-danger/90">
                    Delete
                </button>
            </div>
        </div>
    </div>

    <script>
        let currentEditPeriodId = null;
        let currentDeletePeriodId = null;

        // Modal Functions
        function openCreatePeriodModal() {
            // Set default dates
            const today = new Date();
            const startDate = new Date(today.getFullYear(), today.getMonth(), 1);
            const endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0);
            const cutoffDate = new Date(today.getFullYear(), today.getMonth(), 25);
            const paymentDate = new Date(today.getFullYear(), today.getMonth() + 1, 5);
            
            document.getElementById('periodStartDate').value = startDate.toISOString().split('T')[0];
            document.getElementById('periodEndDate').value = endDate.toISOString().split('T')[0];
            document.getElementById('periodCutoffDate').value = cutoffDate.toISOString().split('T')[0];
            document.getElementById('periodPaymentDate').value = paymentDate.toISOString().split('T')[0];
            
            document.getElementById('createPeriodModal').classList.remove('hidden');
        }

        function closeCreatePeriodModal() {
            document.getElementById('createPeriodModal').classList.add('hidden');
        }

        function editPeriod(periodId) {
            currentEditPeriodId = periodId;
            
            fetch(`/admin/accounting/payroll/periods/${periodId}/edit`, {
                headers: {
                    'Accept': 'application/json'
                }
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    const period = data.period;
                    
                    document.getElementById('editPeriodName').value = period.period_name;
                    document.getElementById('editStartDate').value = period.start_date;
                    document.getElementById('editEndDate').value = period.end_date;
                    document.getElementById('editCutoffDate').value = period.cutoff_date;
                    document.getElementById('editPaymentDate').value = period.payment_date;
                    document.getElementById('editStatus').value = period.status;
                    
                    // Update form action
                    document.getElementById('editPeriodForm').action = 
                        '{{ route("admin.accounting.payroll.periods.update", ":id") }}'.replace(':id', periodId);
                    
                    document.getElementById('editPeriodModal').classList.remove('hidden');
                } else {
                    showNotification(data.message || 'Failed to load period data', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Failed to load period data', 'error');
            });
        }

        function closeEditPeriodModal() {
            document.getElementById('editPeriodModal').classList.add('hidden');
            currentEditPeriodId = null;
        }

        // Handle edit form submission
        document.getElementById('editPeriodForm')?.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const form = this;
            const formData = new FormData(form);
            
            fetch(form.action, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                },
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification(data.message || 'Period updated successfully', 'success');
                    closeEditPeriodModal();
                    setTimeout(() => {
                        window.location.reload();
                    }, 1500);
                } else {
                    showNotification(data.message || 'Failed to update period', 'error');
                }
            })
            .catch(error => {
                showNotification('Failed to update period', 'error');
            });
        });

        function deletePeriod(periodId) {
            currentDeletePeriodId = periodId;
            
            // Get period name for confirmation message
            const periodRow = document.querySelector(`tr:has(button[onclick="deletePeriod(${periodId})"])`);
            const periodName = periodRow?.querySelector('td:first-child')?.textContent || 'this period';
            
            document.getElementById('deleteModalTitle').textContent = 'Delete Payroll Period';
            document.getElementById('deleteModalMessage').textContent = 
                `Are you sure you want to delete "${periodName}"? This action cannot be undone.`;
            
            // Update form action
            document.getElementById('deleteForm').action = 
                '{{ route("admin.accounting.payroll.periods.destroy", ":id") }}'.replace(':id', periodId);
            
            document.getElementById('deleteModal').classList.remove('hidden');
        }

        function closeDeleteModal() {
            document.getElementById('deleteModal').classList.add('hidden');
            currentDeletePeriodId = null;
        }

        function confirmDelete() {
            document.getElementById('deleteForm').submit();
        }

        // Generate periods confirmation function
        function confirmGeneratePeriods(start, end) {
            if (!start || !end) {
                return confirm('Generate monthly periods for the current financial year?');
            }
            return confirm(`Generate monthly periods for the entire financial year? This will create all months from ${start} to ${end}`);
        }

        function processPayroll(periodId) {
            if (confirm('Are you sure you want to process payroll for this period?')) {
                showNotification('Processing payroll...', 'info');
                // Implement payroll processing logic here
            }
        }

        function closePeriod(periodId) {
            if (confirm('Are you sure you want to close this payroll period? This will prevent further changes.')) {
                fetch(`/admin/accounting/payroll/periods/${periodId}/close`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification('Period closed successfully', 'success');
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);
                    } else {
                        showNotification(data.message || 'Failed to close period', 'error');
                    }
                })
                .catch(error => {
                    showNotification('Failed to close period', 'error');
                });
            }
        }

        function reopenPeriod(periodId) {
            if (confirm('Are you sure you want to reopen this payroll period?')) {
                fetch(`/admin/accounting/payroll/periods/${periodId}/reopen`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification('Period reopened successfully', 'success');
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);
                    } else {
                        showNotification(data.message || 'Failed to reopen period', 'error');
                    }
                })
                .catch(error => {
                    showNotification('Failed to reopen period', 'error');
                });
            }
        }

        // Utility functions
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-primary',
                warning: 'bg-yellow-500'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Add animation styles
        if (!document.querySelector('style[data-payroll-periods]')) {
            const style = document.createElement('style');
            style.setAttribute('data-payroll-periods', 'true');
            style.textContent = `
                @keyframes fadeIn {
                    from { opacity: 0; transform: translateY(-10px); }
                    to { opacity: 1; transform: translateY(0); }
                }
                @keyframes fadeOut {
                    from { opacity: 1; transform: translateY(0); }
                    to { opacity: 0; transform: translateY(-10px); }
                }
                .animate-fade-in {
                    animation: fadeIn 0.3s ease-out;
                }
                .animate-fade-out {
                    animation: fadeOut 0.3s ease-in;
                }
            `;
            document.head.appendChild(style);
        }

        // Initialize date defaults for create modal
        document.addEventListener('DOMContentLoaded', function() {
            // Auto-generate period name based on start date
            const periodNameInput = document.querySelector('input[name="period_name"]');
            const startDateInput = document.getElementById('periodStartDate');
            
            if (periodNameInput && startDateInput) {
                startDateInput.addEventListener('change', function() {
                    if (!periodNameInput.value) {
                        const date = new Date(this.value);
                        const monthName = date.toLocaleString('default', { month: 'long' });
                        const year = date.getFullYear();
                        periodNameInput.value = `${monthName} ${year} Payroll`;
                    }
                });
            }
        });
    </script>
</body>
</html>