<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payroll Adjustments - {{ $period->period_name }} | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#146CAA',
                        primaryLight: '#E1F0FA',
                        secondary: '#6b7280',
                        success: '#107C10',
                        warning: '#F68C1E',
                        danger: '#D83B01',
                        info: '#0078D4',
                        gray50: '#FAF9F8',
                        gray100: '#F3F2F1',
                        gray200: '#EDEBE9',
                        gray300: '#E1DFDD',
                        gray400: '#C8C6C4',
                        gray500: '#A19F9D',
                        gray600: '#605E5C',
                        gray700: '#323130',
                        gray800: '#201F1E',
                        gray900: '#000000'
                    }
                }
            }
        }
    </script>
    <style>
        .quickbooks-input {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            transition: all 0.2s;
        }
        .quickbooks-input:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-input-readonly {
            border: 1px solid #E1DFDD;
            background-color: #F3F2F1;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            color: #605E5C;
        }
        .quickbooks-select {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            transition: all 0.2s;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='16' height='16' viewBox='0 0 24 24' fill='none' stroke='%23605E5C' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3E%3Cpath d='m6 9 6 6 6-6'%3E%3C/path%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: right 12px center;
            background-size: 16px;
            padding-right: 36px;
        }
        .quickbooks-select:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-textarea {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            resize: vertical;
            min-height: 60px;
        }
        .quickbooks-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
        }
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
            vertical-align: middle;
        }
        .quickbooks-table tbody tr:hover {
            background-color: #f8fafc;
        }
        .quickbooks-checkbox {
            width: 18px;
            height: 18px;
            border: 2px solid #605E5C;
            border-radius: 3px;
            cursor: pointer;
            accent-color: #146CAA;
        }
        .quickbooks-checkbox:checked {
            background-color: #146CAA;
            border-color: #146CAA;
        }
        .form-section {
            background: white;
            border-radius: 8px;
            padding: 24px;
            margin-bottom: 16px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .form-section h3 {
            font-size: 18px;
            font-weight: 600;
            color: #323130;
            margin-bottom: 20px;
            padding-bottom: 12px;
            border-bottom: 1px solid #EDEBE9;
        }
        .required-field::after {
            content: ' *';
            color: #D83B01;
        }
        .badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .filter-badge {
            cursor: pointer;
            transition: all 0.2s;
        }
        .filter-badge:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .employee-selected {
            background-color: #E1F0FA !important;
            border-left: 3px solid #146CAA !important;
        }
        .action-btn {
            width: 32px;
            height: 32px;
            border-radius: 4px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            margin: 0 2px;
            transition: all 0.2s;
        }
        .action-btn:hover {
            transform: translateY(-1px);
        }
        .action-btn-edit {
            background-color: #E1F0FA;
            color: #146CAA;
        }
        .action-btn-edit:hover {
            background-color: #146CAA;
            color: white;
        }
        .action-btn-delete {
            background-color: #FEE;
            color: #D83B01;
        }
        .action-btn-delete:hover {
            background-color: #D83B01;
            color: white;
        }
        .action-btn-view {
            background-color: #F3F2F1;
            color: #323130;
        }
        .action-btn-view:hover {
            background-color: #146CAA;
            color: white;
        }
        .action-btn-approve {
            background-color: #E7F5E6;
            color: #107C10;
        }
        .action-btn-approve:hover {
            background-color: #107C10;
            color: white;
        }
        
        /* Loading Spinner */
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        .animate-spin {
            animation: spin 1s linear infinite;
        }
        
        /* Print Styles */
        @media print {
            .no-print { display: none !important; }
        }
        
        /* Current Employee Badge */
        .current-employee-badge {
            background: linear-gradient(135deg, #146CAA 0%, #0d5c95 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 6px;
            font-weight: 600;
            display: inline-flex;
            align-items: center;
        }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Payroll Adjustments', 'subtitle' => $period->period_name])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Breadcrumb Navigation -->
            <div class="mb-6 no-print">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="{{ route('admin.accounting.payroll.preparation.index') }}" 
                               class="inline-flex items-center text-sm font-medium text-gray-700 hover:text-primary">
                                <i class="fas fa-home mr-2"></i>
                                Payroll Preparation
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 mx-1"></i>
                                <a href="{{ route('admin.accounting.payroll.preparation.preliminary-summary', $period->id) }}" 
                                   class="ml-1 text-sm font-medium text-gray-700 hover:text-primary md:ml-2">
                                    {{ $period->period_name }} Summary
                                </a>
                            </div>
                        </li>
                        @if(isset($selectedEmployee) && $selectedEmployee)
                            <li aria-current="page">
                                <div class="flex items-center">
                                    <i class="fas fa-chevron-right text-gray-400 mx-1"></i>
                                    <span class="ml-1 text-sm font-medium text-primary md:ml-2">
                                        {{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}
                                    </span>
                                </div>
                            </li>
                        @endif
                    </ol>
                </nav>
            </div>

            <!-- Back Button -->
            <div class="mb-6 flex flex-wrap gap-3 no-print">
                <a href="{{ route('admin.accounting.payroll.preparation.index') }}" 
                   class="text-primary hover:text-primary/90 font-medium text-sm flex items-center">
                    <i class="fas fa-arrow-left mr-2"></i>Back to Payroll Preparation
                </a>
                <a href="{{ route('admin.accounting.payroll.preparation.preliminary-summary', $period->id) }}" 
                   class="text-primary hover:text-primary/90 font-medium text-sm flex items-center ml-4">
                    <i class="fas fa-chart-bar mr-2"></i>Back to Summary
                </a>
                @if(isset($selectedEmployee) && $selectedEmployee)
                    <a href="{{ route('admin.accounting.payroll.preparation.create', $period->id) }}" 
                       class="text-gray-600 hover:text-gray-800 font-medium text-sm flex items-center ml-4">
                        <i class="fas fa-users mr-2"></i>View All Employees
                    </a>
                @endif
            </div>

            <!-- Period Info Card -->
            <div class="bg-white rounded-lg border border-gray200 p-6 mb-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between">
                    <div>
                        <h1 class="text-xl font-bold text-gray900">{{ $period->period_name }}</h1>
                        <div class="flex flex-wrap items-center gap-4 mt-2">
                            <span class="inline-flex items-center text-sm text-gray600">
                                <i class="fas fa-calendar-alt mr-1"></i>
                                {{ $period->start_date->format('M d, Y') }} - {{ $period->end_date->format('M d, Y') }}
                            </span>
                            <span class="inline-flex items-center text-sm text-gray600">
                                <i class="fas fa-money-bill-wave mr-1"></i>
                                Payment Date: {{ $period->payment_date->format('M d, Y') }}
                            </span>
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                {{ $period->status === 'open' ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800' }}">
                                {{ ucfirst($period->status) }}
                            </span>
                            @if(isset($selectedEmployee) && $selectedEmployee)
                                <span class="current-employee-badge">
                                    <i class="fas fa-user mr-2"></i>
                                    {{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}
                                    <span class="text-xs ml-2 opacity-90">#{{ $selectedEmployee->employee_number }}</span>
                                </span>
                            @endif
                        </div>
                    </div>
                    <div class="mt-4 md:mt-0">
                        <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-blue-100 text-blue-800">
                            <i class="fas fa-users mr-2"></i>
                            {{ $employees->count() }} Active Employees
                        </span>
                    </div>
                </div>
            </div>

            <!-- Two Column Layout -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Left Column: Add Adjustment Form & Adjustments -->
                <div class="lg:col-span-2">
                    <!-- Add Adjustment Form -->
                    <div class="form-section" id="addAdjustmentSection">
                        <h3>
                            @if(isset($selectedEmployee) && $selectedEmployee)
                                Add Adjustment for {{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}
                            @else
                                Add New Adjustment
                            @endif
                        </h3>
                        
                        <form id="addAdjustmentForm" action="{{ route('admin.accounting.payroll.preparation.store', $period->id) }}" method="POST">
                            @csrf
                            
                            <!-- Employee Input Field (Read-only when employee is pre-selected) -->
                            <div class="mb-6">
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Employee</label>
                                @if(isset($selectedEmployee) && $selectedEmployee)
                                    <!-- Read-only input when employee is pre-selected -->
                                    <div class="relative">
                                        <input type="text" 
                                               value="{{ $selectedEmployee->employee_number }} - {{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}"
                                               class="quickbooks-input-readonly"
                                               readonly>
                                        <input type="hidden" name="employee_id" value="{{ $selectedEmployee->id }}">
                                        <div class="absolute right-3 top-1/2 transform -translate-y-1/2">
                                            <a href="{{ route('admin.accounting.payroll.preparation.create', $period->id) }}" 
                                               class="text-primary hover:text-primary/90 text-sm">
                                                Change Employee
                                            </a>
                                        </div>
                                    </div>
                                @else
                                    <!-- Dropdown when no employee is selected -->
                                    <select name="employee_id" id="employeeSelect" class="quickbooks-select" required onchange="onEmployeeSelectChange(this.value)">
                                        <option value="">Select Employee</option>
                                        @foreach($employees as $employee)
                                            <option value="{{ $employee->id }}">
                                                {{ $employee->employee_number }} - {{ $employee->first_name }} {{ $employee->last_name }}
                                            </option>
                                        @endforeach
                                    </select>
                                @endif
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray700 mb-2 required-field">Adjustment Type</label>
                                    <select name="earning_type_id" id="earningTypeSelect" class="quickbooks-select" required>
                                        <option value="">Select Type</option>
                                        @foreach($earningTypes as $type)
                                            <option value="{{ $type->id }}" data-taxable="{{ $type->is_taxable }}">
                                                {{ $type->name }} ({{ $type->code }})
                                            </option>
                                        @endforeach
                                    </select>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray700 mb-2 required-field">Amount</label>
                                    <div class="relative">
                                        <input type="number" 
                                               name="amount" 
                                               class="quickbooks-input pl-8" 
                                               placeholder="0.00"
                                               step="0.01"
                                               min="0"
                                               required>
                                        <span class="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 text-sm">{{ $company->currency_symbol ?? 'K' }}</span>
                                    </div>
                                </div>
                            </div>

                            <div class="mb-6">
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Description</label>
                                <input type="text" 
                                       name="description" 
                                       class="quickbooks-input" 
                                       placeholder="e.g., Overtime for weekend work, Year-end bonus"
                                       required>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray700 mb-4">Taxable</label>
                                    <div class="flex items-center space-x-4">
                                        <div class="flex items-center">
                                            <input type="radio" id="taxable_yes" name="is_taxable" value="1" class="quickbooks-checkbox mr-2" checked>
                                            <label for="taxable_yes" class="text-sm text-gray600">Yes</label>
                                        </div>
                                        <div class="flex items-center">
                                            <input type="radio" id="taxable_no" name="is_taxable" value="0" class="quickbooks-checkbox mr-2">
                                            <label for="taxable_no" class="text-sm text-gray600">No</label>
                                        </div>
                                    </div>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray700 mb-4">Adjustment Type</label>
                                    <div class="flex items-center space-x-4">
                                        <div class="flex items-center">
                                            <input type="radio" id="one_time_yes" name="is_one_time" value="1" class="quickbooks-checkbox mr-2" checked>
                                            <label for="one_time_yes" class="text-sm text-gray600">One-time</label>
                                        </div>
                                        <div class="flex items-center">
                                            <input type="radio" id="one_time_no" name="is_one_time" value="0" class="quickbooks-checkbox mr-2">
                                            <label for="one_time_no" class="text-sm text-gray600">Recurring (for this period)</label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="mb-6">
                                <label class="block text-sm font-medium text-gray700 mb-2">Notes (Optional)</label>
                                <textarea name="notes" 
                                          class="quickbooks-textarea" 
                                          placeholder="Additional information about this adjustment"></textarea>
                            </div>

                            <div class="flex justify-end space-x-3">
                                <button type="button" 
                                        onclick="resetForm()"
                                        class="bg-gray-200 text-gray-700 px-6 py-2 rounded text-sm font-medium hover:bg-gray-300 flex items-center">
                                    <i class="fas fa-redo mr-2"></i> Reset Form
                                </button>
                                <button type="submit" 
                                        class="bg-primary text-white px-6 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center">
                                    <i class="fas fa-plus mr-2"></i> Add Adjustment
                                </button>
                            </div>
                        </form>
                    </div>

                    <!-- Recurring Earnings Section -->
                    <div class="form-section" id="recurringSection">
                        <div class="flex justify-between items-center mb-6">
                            <h3>
                                @if(isset($selectedEmployee) && $selectedEmployee)
                                    Recurring Earnings for {{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}
                                @else
                                    Recurring Earnings
                                @endif
                            </h3>
                            <div class="flex items-center space-x-2">
                                <div class="text-sm text-gray600">
                                    <span id="recurringCount">{{ $recurringEarnings->count() }}</span> recurring earning(s)
                                </div>
                                <div class="flex items-center space-x-1" id="employeeFilterBadges">
                                    <span class="text-xs text-gray500">Filter:</span>
                                    <button type="button" 
                                            onclick="viewAllRecurringEarnings()"
                                            class="filter-badge inline-flex items-center px-2 py-1 rounded-full text-xs font-medium 
                                                   {{ !isset($selectedEmployee) ? 'bg-primary text-white hover:bg-primary/90' : 'bg-gray-200 text-gray-700 hover:bg-gray-300' }}">
                                        All
                                    </button>
                                    @foreach($employees as $emp)
                                        <button type="button" 
                                                onclick="filterRecurringEarnings({{ $emp->id }})"
                                                class="filter-badge inline-flex items-center px-2 py-1 rounded-full text-xs font-medium 
                                                       {{ (isset($selectedEmployee) && $selectedEmployee && $selectedEmployee->id == $emp->id) ? 'bg-primary text-white hover:bg-primary/90' : 'bg-gray-200 text-gray-700 hover:bg-gray-300' }}"
                                                data-employee-id="{{ $emp->id }}">
                                            {{ substr($emp->first_name, 0, 1) }}{{ substr($emp->last_name, 0, 1) }}
                                        </button>
                                    @endforeach
                                </div>
                            </div>
                        </div>

                        <!-- Recurring earnings table -->
                        <div id="recurringEarningsTable">
                            @if($recurringEarnings->count() > 0)
                                <div class="overflow-x-auto">
                                    <table class="quickbooks-table">
                                        <thead>
                                            <tr>
                                                @if(!isset($selectedEmployee))
                                                    <th>Employee</th>
                                                @endif
                                                <th>Earning Type</th>
                                                <th>Frequency</th>
                                                <th>Monthly Amount</th>
                                                <th>Taxable</th>
                                                <th>Effective Date</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            @foreach($recurringEarnings as $earning)
                                                @php
                                                    $employee = isset($selectedEmployee) ? $selectedEmployee : $earning->employee;
                                                @endphp
                                                <tr data-employee-id="{{ $employee->id }}">
                                                    @if(!isset($selectedEmployee))
                                                        <td>
                                                            <div class="font-medium">{{ $employee->first_name }} {{ $employee->last_name }}</div>
                                                            <div class="text-xs text-gray500">#{{ $employee->employee_number }}</div>
                                                        </td>
                                                    @endif
                                                    <td>
                                                        <div class="font-medium">{{ $earning->earningType->name }}</div>
                                                        <div class="text-xs text-gray500">{{ $earning->earningType->code }}</div>
                                                    </td>
                                                    <td>
                                                        <span class="text-sm font-medium capitalize">{{ $earning->frequency }}</span>
                                                    </td>
                                                    <td class="font-medium">
                                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($earning->getMonthlyAmount(), 2) }}
                                                    </td>
                                                    <td>
                                                        @if($earning->earningType->is_taxable)
                                                            <span class="badge bg-red-100 text-red-800">Yes</span>
                                                        @else
                                                            <span class="badge bg-green-100 text-green-800">No</span>
                                                        @endif
                                                    </td>
                                                    <td class="text-sm">
                                                        {{ $earning->effective_date->format('M d, Y') }}
                                                        @if($earning->end_date)
                                                            <div class="text-xs text-gray500">to {{ $earning->end_date->format('M d, Y') }}</div>
                                                        @endif
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                        @if($recurringEarnings->count() > 0)
                                            <tfoot class="bg-gray-50">
                                                <tr>
                                                    <td colspan="{{ isset($selectedEmployee) ? 2 : 3 }}" class="text-right font-medium text-gray-700 py-3">
                                                        Total Monthly Recurring:
                                                    </td>
                                                    <td class="font-bold text-primary py-3">
                                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($recurringEarnings->sum(function($e) { return $e->getMonthlyAmount(); }), 2) }}
                                                    </td>
                                                    <td colspan="2" class="py-3">
                                                        <div class="text-sm text-gray-600">
                                                            {{ $recurringEarnings->count() }} earning(s)
                                                        </div>
                                                    </td>
                                                </tr>
                                            </tfoot>
                                        @endif
                                    </table>
                                </div>
                            @else
                                <div class="text-center py-8 text-gray500">
                                    <i class="fas fa-file-invoice-dollar text-3xl mb-3 text-gray300"></i>
                                    @if(isset($selectedEmployee) && $selectedEmployee)
                                        <p>No recurring earnings for {{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}</p>
                                    @else
                                        <p>No recurring earnings found</p>
                                    @endif
                                    <p class="text-sm mt-1">
                                        Recurring earnings are automatically loaded from employee profiles
                                    </p>
                                </div>
                            @endif
                        </div>
                    </div>

                    <!-- Payroll Adjustments Section -->
                    <div class="form-section" id="adjustmentsSection">
                        <div class="flex justify-between items-center mb-6">
                            <h3>
                                @if(isset($selectedEmployee) && $selectedEmployee)
                                    Payroll Adjustments for {{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}
                                @else
                                    Payroll Adjustments
                                @endif
                            </h3>
                            <div class="text-sm text-gray600">
                                @if(isset($selectedEmployee) && $selectedEmployee)
                                    @php
                                        $employeeAdjustmentsCount = $adjustments->where('employee_id', $selectedEmployee->id)->count();
                                    @endphp
                                    <span id="adjustmentsCount">{{ $employeeAdjustmentsCount }}</span> adjustment(s)
                                @else
                                    <span id="adjustmentsCount">{{ $adjustments->count() }}</span> adjustment(s)
                                @endif
                            </div>
                        </div>

                        <!-- Adjustments table -->
                        <div id="adjustmentsTableContainer">
                            @if(isset($selectedEmployee) && $selectedEmployee)
                                @php
                                    $employeeAdjustmentsList = $adjustments->where('employee_id', $selectedEmployee->id);
                                @endphp
                                
                                @if($employeeAdjustmentsList->count() > 0)
                                    <div class="overflow-x-auto">
                                        <table class="quickbooks-table">
                                            <thead>
                                                <tr>
                                                    <th>Type</th>
                                                    <th>Description</th>
                                                    <th>Amount</th>
                                                    <th>Taxable</th>
                                                    <th>Status</th>
                                                    <th class="no-print">Actions</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                @foreach($employeeAdjustmentsList as $adjustment)
                                                    <tr>
                                                        <td>
                                                            <span class="text-sm font-medium text-gray900">
                                                                {{ $adjustment->earningType->name }}
                                                            </span>
                                                            <div class="text-xs text-gray500">{{ $adjustment->earningType->code }}</div>
                                                        </td>
                                                        <td>
                                                            <div class="text-sm text-gray900">{{ $adjustment->description }}</div>
                                                            @if($adjustment->notes)
                                                                <div class="text-xs text-gray500">{{ $adjustment->notes }}</div>
                                                            @endif
                                                        </td>
                                                        <td class="font-medium">
                                                            {{ $company->currency_symbol ?? 'K' }}{{ number_format($adjustment->amount, 2) }}
                                                        </td>
                                                        <td>
                                                            @if($adjustment->is_taxable)
                                                                <span class="badge bg-red-100 text-red-800">Yes</span>
                                                            @else
                                                                <span class="badge bg-green-100 text-green-800">No</span>
                                                            @endif
                                                        </td>
                                                        <td>
                                                            @if($adjustment->approved_by)
                                                                <span class="badge bg-green-100 text-green-800">
                                                                    <i class="fas fa-check mr-1"></i> Approved
                                                                </span>
                                                                <div class="text-xs text-gray500 mt-1">
                                                                    {{ optional($adjustment->approved_at)->format('M d, Y') ?? 'N/A' }}
                                                                </div>
                                                            @else
                                                                <span class="badge bg-yellow-100 text-yellow-800">
                                                                    <i class="fas fa-clock mr-1"></i> Pending
                                                                </span>
                                                            @endif
                                                        </td>
                                                        <td class="no-print">
                                                            <div class="flex items-center">
                                                                @if(!$adjustment->approved_by)
                                                                    <form action="{{ route('admin.accounting.payroll.preparation.approve', $adjustment->id) }}" 
                                                                          method="POST" 
                                                                          class="inline">
                                                                        @csrf
                                                                        <button type="submit" 
                                                                                class="action-btn action-btn-approve"
                                                                                title="Approve"
                                                                                onclick="return confirm('Approve this adjustment?')">
                                                                            <i class="fas fa-check"></i>
                                                                        </button>
                                                                    </form>
                                                                    
                                                                    <a href="{{ route('admin.accounting.payroll.preparation.edit', $adjustment->id) }}" 
                                                                       class="action-btn action-btn-edit"
                                                                       title="Edit">
                                                                        <i class="fas fa-edit"></i>
                                                                    </a>
                                                                    
                                                                    <form action="{{ route('admin.accounting.payroll.preparation.destroy', $adjustment->id) }}" 
                                                                          method="POST" 
                                                                          class="inline"
                                                                          onsubmit="return confirm('Are you sure you want to delete this adjustment?')">
                                                                        @csrf
                                                                        @method('DELETE')
                                                                        <button type="submit" 
                                                                                class="action-btn action-btn-delete"
                                                                                title="Delete">
                                                                            <i class="fas fa-trash"></i>
                                                                        </button>
                                                                    </form>
                                                                @else
                                                                    <form action="{{ route('admin.accounting.payroll.preparation.unapprove', $adjustment->id) }}" 
                                                                          method="POST" 
                                                                          class="inline">
                                                                        @csrf
                                                                        <button type="submit" 
                                                                                class="action-btn action-btn-view"
                                                                                title="Unapprove"
                                                                                onclick="return confirm('Unapprove this adjustment?')">
                                                                            <i class="fas fa-undo"></i>
                                                                        </button>
                                                                    </form>
                                                                @endif
                                                            </div>
                                                        </td>
                                                    </tr>
                                                @endforeach
                                            </tbody>
                                        </table>
                                    </div>
                                    
                                    <!-- Bulk Actions -->
                                    @if($employeeAdjustmentsList->whereNull('approved_by')->count() > 0)
                                        <div class="mt-6 p-4 bg-gray50 border border-gray200 rounded-lg no-print">
                                            <form action="{{ route('admin.accounting.payroll.preparation.bulk-approve', $period->id) }}" method="POST" id="bulkApproveForm">
                                                @csrf
                                                <input type="hidden" name="adjustment_ids" id="bulkAdjustmentIds" value="{{ $employeeAdjustmentsList->whereNull('approved_by')->pluck('id')->toJson() }}">
                                                <div class="flex items-center justify-between">
                                                    <div class="text-sm text-gray600">
                                                        {{ $employeeAdjustmentsList->whereNull('approved_by')->count() }} pending adjustment(s) for this employee
                                                    </div>
                                                    <button type="submit" 
                                                            class="bg-green-600 text-white px-4 py-2 rounded text-sm font-medium hover:bg-green-700 flex items-center"
                                                            onclick="return confirm('Approve all pending adjustments for this employee?')">
                                                        <i class="fas fa-check-double mr-2"></i> Approve All Pending
                                                    </button>
                                                </div>
                                            </form>
                                        </div>
                                    @endif
                                @else
                                    <div class="text-center py-8 text-gray500">
                                        <i class="fas fa-file-invoice-dollar text-3xl mb-3 text-gray300"></i>
                                        <p>No adjustments for {{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}</p>
                                        <p class="text-sm mt-1">Use the form above to add payroll adjustments</p>
                                    </div>
                                @endif
                            @else
                                <div class="text-center py-8 text-gray500">
                                    <i class="fas fa-file-invoice-dollar text-3xl mb-3 text-gray300"></i>
                                    <p>No employee selected</p>
                                    <p class="text-sm mt-1">Select an employee to view their adjustments</p>
                                </div>
                            @endif
                        </div>
                    </div>
                </div>

                <!-- Right Column: Summary & Actions -->
                <div>
                    <!-- Selected Employee Info -->
                    @if(isset($selectedEmployee) && $selectedEmployee)
                        <div class="form-section">
                            <h3>Employee Information</h3>
                            <div class="space-y-3">
                                <div>
                                    <div class="text-xs text-gray500 mb-1">Employee Number</div>
                                    <div class="font-medium">{{ $selectedEmployee->employee_number }}</div>
                                </div>
                                <div>
                                    <div class="text-xs text-gray500 mb-1">Name</div>
                                    <div class="font-medium">{{ $selectedEmployee->first_name }} {{ $selectedEmployee->last_name }}</div>
                                </div>
                                <div>
                                    <div class="text-xs text-gray500 mb-1">Email</div>
                                    <div class="text-sm">{{ $selectedEmployee->email }}</div>
                                </div>
                                <div>
                                    <div class="text-xs text-gray500 mb-1">Employment Type</div>
                                    <div class="text-sm capitalize">{{ $selectedEmployee->employment_type }}</div>
                                </div>
                                @if($selectedEmployee->phone)
                                    <div>
                                        <div class="text-xs text-gray500 mb-1">Phone</div>
                                        <div class="text-sm">{{ $selectedEmployee->phone }}</div>
                                    </div>
                                @endif
                                <div class="pt-3 border-t border-gray200">
                                    <a href="{{ route('admin.accounting.payroll.employees.edit', $selectedEmployee->id) }}" 
                                       class="text-primary hover:text-primary/90 text-sm flex items-center">
                                        <i class="fas fa-edit mr-2"></i> Edit Employee Details
                                    </a>
                                </div>
                            </div>
                        </div>
                    @endif

                    <!-- Summary Card -->
                    <div class="form-section">
                        <h3>Summary</h3>
                        
                        @php
                            // Calculate adjustments totals
                            $totalAdjustmentAmount = $adjustments->sum('amount');
                            $adjustmentTaxable = $adjustments->where('is_taxable', true)->sum('amount');
                            $adjustmentNonTaxable = $adjustments->where('is_taxable', false)->sum('amount');
                            
                            // Get selected employee's adjustments total
                            $selectedEmployeeAdjustmentsTotal = isset($selectedEmployee) ? 
                                $adjustments->where('employee_id', $selectedEmployee->id)->sum('amount') : 0;
                            
                            // Counts
                            $totalAdjustments = $adjustments->count();
                            $approvedAdjustments = $adjustments->whereNotNull('approved_by')->count();
                            $pendingAdjustments = $adjustments->whereNull('approved_by')->count();
                            
                            // Calculate recurring earnings totals
                            $totalRecurringAmount = $recurringEarnings->sum(function($e) { return $e->getMonthlyAmount(); });
                            $recurringTaxable = $recurringEarnings->where('earningType.is_taxable', true)->sum(function($e) { return $e->getMonthlyAmount(); });
                            $recurringNonTaxable = $recurringEarnings->where('earningType.is_taxable', false)->sum(function($e) { return $e->getMonthlyAmount(); });
                        @endphp
                        
                        <div class="space-y-4">
                            <div class="flex justify-between items-center">
                                <span class="text-sm text-gray600">Active Employees:</span>
                                <span class="font-medium">{{ $employees->count() }}</span>
                            </div>
                            
                            <!-- Recurring Earnings Summary -->
                            <div class="border-t border-gray200 pt-4">
                                <div class="flex justify-between items-center mb-2">
                                    <span class="text-sm font-medium text-gray600">Recurring Earnings:</span>
                                    <span class="font-medium">{{ $recurringEarnings->count() }}</span>
                                </div>
                                <div class="flex justify-between items-center text-sm mb-1">
                                    <span class="text-gray500">Total Amount:</span>
                                    <span class="font-medium">{{ $company->currency_symbol ?? 'K' }}{{ number_format($totalRecurringAmount, 2) }}</span>
                                </div>
                                <div class="flex justify-between items-center text-xs text-gray500 mb-1">
                                    <span>Taxable: {{ $company->currency_symbol ?? 'K' }}{{ number_format($recurringTaxable, 2) }}</span>
                                    <span>Non-taxable: {{ $company->currency_symbol ?? 'K' }}{{ number_format($recurringNonTaxable, 2) }}</span>
                                </div>
                            </div>
                            
                            <!-- Adjustments Summary -->
                            <div class="border-t border-gray200 pt-4">
                                <div class="flex justify-between items-center mb-2">
                                    <span class="text-sm font-medium text-gray600">Adjustments:</span>
                                    <span class="font-medium">{{ $totalAdjustments }}</span>
                                </div>
                                <div class="flex justify-between items-center mb-1">
                                    <span class="text-sm text-gray600">Approved:</span>
                                    <span class="font-medium text-green-600">{{ $approvedAdjustments }}</span>
                                </div>
                                <div class="flex justify-between items-center mb-1">
                                    <span class="text-sm text-gray600">Pending:</span>
                                    <span class="font-medium text-yellow-600">{{ $pendingAdjustments }}</span>
                                </div>
                                <div class="flex justify-between items-center text-sm mb-1">
                                    <span class="text-gray500">Total Amount:</span>
                                    <span class="font-medium">{{ $company->currency_symbol ?? 'K' }}{{ number_format($totalAdjustmentAmount, 2) }}</span>
                                </div>
                                <div class="flex justify-between items-center text-xs text-gray500">
                                    <span>Taxable: {{ $company->currency_symbol ?? 'K' }}{{ number_format($adjustmentTaxable, 2) }}</span>
                                    <span>Non-taxable: {{ $company->currency_symbol ?? 'K' }}{{ number_format($adjustmentNonTaxable, 2) }}</span>
                                </div>
                            </div>
                            
                            <!-- Selected Employee Summary -->
                            @if(isset($selectedEmployee) && $selectedEmployee)
                                <div class="border-t border-gray200 pt-4 border-dashed">
                                    <div class="flex justify-between items-center mb-2">
                                        <span class="text-sm font-medium text-gray600">This Employee:</span>
                                        <span class="text-xs text-primary font-medium">{{ $selectedEmployee->first_name }}</span>
                                    </div>
                                    <div class="flex justify-between items-center text-sm mb-1">
                                        <span class="text-gray500">Adjustments:</span>
                                        <span class="font-medium">{{ $adjustments->where('employee_id', $selectedEmployee->id)->count() }}</span>
                                    </div>
                                    <div class="flex justify-between items-center text-sm mb-1">
                                        <span class="text-gray500">Total:</span>
                                        <span class="font-medium text-primary">{{ $company->currency_symbol ?? 'K' }}{{ number_format($selectedEmployeeAdjustmentsTotal, 2) }}</span>
                                    </div>
                                    <div class="flex justify-between items-center text-xs text-gray500">
                                        <span>Pending: {{ $adjustments->where('employee_id', $selectedEmployee->id)->whereNull('approved_by')->count() }}</span>
                                        <span>Approved: {{ $adjustments->where('employee_id', $selectedEmployee->id)->whereNotNull('approved_by')->count() }}</span>
                                    </div>
                                </div>
                            @endif
                            
                            <!-- Grand Total -->
                            <div class="border-t border-gray200 pt-4 mt-4">
                                <div class="flex justify-between items-center mb-2">
                                    <span class="text-sm font-bold text-gray700">TOTAL PAYROLL:</span>
                                    <span class="font-bold text-lg text-primary">{{ $company->currency_symbol ?? 'K' }}{{ number_format($totalRecurringAmount + $totalAdjustmentAmount, 2) }}</span>
                                </div>
                                <div class="flex justify-between items-center text-sm mb-1">
                                    <span class="text-gray600">Recurring Total:</span>
                                    <span class="font-medium text-blue-600">{{ $company->currency_symbol ?? 'K' }}{{ number_format($totalRecurringAmount, 2) }}</span>
                                </div>
                                <div class="flex justify-between items-center text-sm">
                                    <span class="text-gray600">Adjustments Total:</span>
                                    <span class="font-medium text-green-600">{{ $company->currency_symbol ?? 'K' }}{{ number_format($totalAdjustmentAmount, 2) }}</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- All Employees -->
                    <div class="form-section">
                        <div class="flex justify-between items-center mb-4">
                            <h3>All Employees</h3>
                            <span class="text-xs text-gray500">{{ $employees->count() }} employees</span>
                        </div>
                        
                        <div class="space-y-2 max-h-[300px] overflow-y-auto pr-2">
                            @foreach($employees as $employee)
                                @php
                                    $employeeAdjCount = $adjustments->where('employee_id', $employee->id)->count();
                                    $employeeAdjPending = $adjustments->where('employee_id', $employee->id)->whereNull('approved_by')->count();
                                    $isSelected = isset($selectedEmployee) && $selectedEmployee && $selectedEmployee->id == $employee->id;
                                @endphp
                                <a href="{{ route('admin.accounting.payroll.preparation.create', ['periodId' => $period->id, 'employeeId' => $employee->id]) }}" 
                                   class="block border border-gray200 rounded p-3 hover:bg-gray50 transition-colors {{ $isSelected ? 'employee-selected' : '' }}">
                                    <div class="flex justify-between items-center">
                                        <div>
                                            <div class="font-medium text-sm">{{ $employee->first_name }} {{ $employee->last_name }}</div>
                                            <div class="text-xs text-gray500 mt-1">{{ $employee->employee_number }}</div>
                                        </div>
                                        <div class="text-right">
                                            @if($employeeAdjCount > 0)
                                                <div class="text-sm font-medium">{{ $employeeAdjCount }} adj</div>
                                                @if($employeeAdjPending > 0)
                                                    <div class="text-xs text-yellow-600">{{ $employeeAdjPending }} pending</div>
                                                @endif
                                            @else
                                                <div class="text-xs text-gray500">No adjustments</div>
                                            @endif
                                        </div>
                                    </div>
                                </a>
                            @endforeach
                        </div>
                    </div>

                    <!-- Quick Actions -->
                    <div class="form-section no-print">
                        <h3>Quick Actions</h3>
                        
                        <div class="space-y-3">
                            <a href="{{ route('admin.accounting.payroll.processing.create', $period->id) }}" 
                               class="block w-full bg-green-600 text-white text-center px-4 py-3 rounded text-sm font-medium hover:bg-green-700 flex items-center justify-center">
                                <i class="fas fa-calculator mr-2"></i> Process Payroll
                            </a>
                            
                            <a href="{{ route('admin.accounting.payroll.preparation.preliminary-summary', $period->id) }}" 
                               class="block w-full bg-primary text-white text-center px-4 py-3 rounded text-sm font-medium hover:bg-primary/90 flex items-center justify-center">
                                <i class="fas fa-chart-bar mr-2"></i> View Summary
                            </a>
                            
                            @if(isset($selectedEmployee) && $selectedEmployee)
                                <a href="{{ route('admin.accounting.payroll.employees.edit', $selectedEmployee->id) }}" 
                                   class="block w-full border border-gray300 text-gray700 text-center px-4 py-3 rounded text-sm font-medium hover:bg-gray50 flex items-center justify-center">
                                    <i class="fas fa-user-edit mr-2"></i> Edit Employee
                                </a>
                            @endif
                            
                            <button type="button" 
                                    onclick="window.print()"
                                    class="block w-full border border-gray300 text-gray700 text-center px-4 py-3 rounded text-sm font-medium hover:bg-gray50 flex items-center justify-center">
                                <i class="fas fa-print mr-2"></i> Print Adjustments
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- AJAX Scripts -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Auto-select taxable based on earning type selection
            const earningTypeSelect = document.getElementById('earningTypeSelect');
            const taxableYes = document.getElementById('taxable_yes');
            const taxableNo = document.getElementById('taxable_no');
            
            if (earningTypeSelect) {
                earningTypeSelect.addEventListener('change', function() {
                    const selectedOption = this.options[this.selectedIndex];
                    const isTaxableDefault = selectedOption.dataset.taxable === 'true';
                    
                    if (isTaxableDefault) {
                        taxableYes.checked = true;
                    } else {
                        taxableNo.checked = true;
                    }
                });
            }
            
            // Form validation
            const addAdjustmentForm = document.getElementById('addAdjustmentForm');
            if (addAdjustmentForm) {
                addAdjustmentForm.addEventListener('submit', function(e) {
                    let isValid = true;
                    
                    // Validate required fields
                    const requiredFields = addAdjustmentForm.querySelectorAll('[required]');
                    requiredFields.forEach(field => {
                        if (!field.value.trim()) {
                            isValid = false;
                            field.classList.add('border-red-500');
                        } else {
                            field.classList.remove('border-red-500');
                        }
                    });
                    
                    // Validate amount
                    const amountField = addAdjustmentForm.querySelector('input[name="amount"]');
                    if (amountField && (parseFloat(amountField.value) <= 0 || isNaN(parseFloat(amountField.value)))) {
                        isValid = false;
                        amountField.classList.add('border-red-500');
                    }
                    
                    if (!isValid) {
                        e.preventDefault();
                        showToast('Please fill in all required fields with valid values.', 'error');
                    }
                });
            }
            
            // Auto-focus first field if no employee is pre-selected
            @if(!isset($selectedEmployee) || !$selectedEmployee)
                const firstField = addAdjustmentForm?.querySelector('[required]');
                if (firstField) {
                    firstField.focus();
                }
            @else
                // Scroll to adjustments section when employee is pre-selected
                setTimeout(() => {
                    document.getElementById('adjustmentsSection').scrollIntoView({ 
                        behavior: 'smooth', 
                        block: 'start' 
                    });
                }, 300);
            @endif
            
            // Calculate initial recurring summary if employee is selected
            @if(isset($selectedEmployee) && $selectedEmployee)
                updateRecurringSummaryTotals();
            @endif
        });
        
        // Function to handle employee select change (when no employee is pre-selected)
        function onEmployeeSelectChange(employeeId) {
            if (employeeId) {
                // Reload page with selected employee
                window.location.href = `{{ route('admin.accounting.payroll.preparation.create', ['periodId' => $period->id, 'employeeId' => '']) }}${employeeId}`;
            }
        }
        
        // Function to filter recurring earnings by employee
        function filterRecurringEarnings(employeeId) {
            // Update filter badge styles
            const filterBadges = document.querySelectorAll('#employeeFilterBadges .filter-badge');
            filterBadges.forEach(badge => {
                if (employeeId === 'all' && badge.textContent.trim() === 'All') {
                    badge.classList.remove('bg-gray-200', 'text-gray-700');
                    badge.classList.add('bg-primary', 'text-white');
                } else if (badge.dataset.employeeId == employeeId) {
                    badge.classList.remove('bg-gray-200', 'text-gray-700');
                    badge.classList.add('bg-primary', 'text-white');
                } else {
                    badge.classList.remove('bg-primary', 'text-white');
                    badge.classList.add('bg-gray-200', 'text-gray-700');
                }
            });
            
            // Show loading indicator
            const tableContainer = document.getElementById('recurringEarningsTable');
            tableContainer.innerHTML = `
                <div class="text-center py-8">
                    <div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
                    <p class="mt-2 text-sm text-gray600">Loading recurring earnings...</p>
                </div>
            `;
            
            // Use Laravel's route helper to generate correct URL
            const url = `{{ route('admin.accounting.payroll.preparation.recurring-earnings', ['periodId' => $period->id]) }}?employee_id=${employeeId}`;
            
            // Make AJAX request
            fetch(url, {
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                }
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`Network response was not ok: ${response.status} ${response.statusText}`);
                }
                return response.json();
            })
            .then(data => {
                if (data.html) {
                    tableContainer.innerHTML = data.html;
                    document.getElementById('recurringCount').textContent = data.count;
                    
                    // Calculate and update summary totals
                    updateRecurringSummaryTotals(data.html, employeeId, data.count);
                } else {
                    throw new Error('Invalid response format');
                }
            })
            .catch(error => {
                console.error('Error loading recurring earnings:', error);
                tableContainer.innerHTML = `
                    <div class="text-center py-8 text-red-500">
                        <i class="fas fa-exclamation-triangle text-2xl mb-2"></i>
                        <p>Failed to load recurring earnings</p>
                        <p class="text-sm mt-1">Please try again or contact support</p>
                    </div>
                `;
                document.getElementById('recurringCount').textContent = '0';
            });
        }
        
        // Function to view all recurring earnings
        function viewAllRecurringEarnings() {
            // Navigate to the page without employee selection
            window.location.href = `{{ route('admin.accounting.payroll.preparation.create', $period->id) }}`;
        }
        
        // Function to update summary totals based on filtered data
        function updateRecurringSummaryTotals(html, employeeId, count) {
            // Create a temporary div to parse the HTML
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = html;
            
            // Get all rows from the table
            const rows = tempDiv.querySelectorAll('tbody tr');
            let totalAmount = 0;
            let taxableAmount = 0;
            let nonTaxableAmount = 0;
            
            // Parse amounts from each row
            rows.forEach(row => {
                const amountCell = row.querySelector('td:nth-child({{ isset($selectedEmployee) ? 3 : 4 }})'); // Monthly Amount column
                const taxableCell = row.querySelector('td:nth-child({{ isset($selectedEmployee) ? 4 : 5 }})'); // Taxable column
                
                if (amountCell) {
                    const amountText = amountCell.textContent.trim();
                    // Extract numeric value from currency string (e.g., "K1,234.56" -> 1234.56)
                    const amountMatch = amountText.match(/[0-9,]+\.?[0-9]*/);
                    if (amountMatch) {
                        const amount = parseFloat(amountMatch[0].replace(/,/g, ''));
                        const validAmount = isNaN(amount) ? 0 : amount;
                        totalAmount += validAmount;
                        
                        // Check if taxable
                        if (taxableCell) {
                            const taxableText = taxableCell.textContent.trim();
                            if (taxableText.includes('Yes')) {
                                taxableAmount += validAmount;
                            } else {
                                nonTaxableAmount += validAmount;
                            }
                        }
                    }
                }
            });
            
            // Update summary display
            const currencySymbol = '{{ $company->currency_symbol ?? "K" }}';
            document.getElementById('recurringCount').textContent = count;
            
            // Update the recurring summary in the right column if it exists
            const recurringSummaryCount = document.querySelector('.border-t.border-gray200.pt-4:nth-child(2) .font-medium');
            const recurringSummaryTotal = document.querySelector('.border-t.border-gray200.pt-4:nth-child(2) .text-sm.mb-1 .font-medium');
            const recurringSummaryTaxable = document.querySelector('.border-t.border-gray200.pt-4:nth-child(2) .text-xs.text-gray500.mb-1 span:first-child');
            const recurringSummaryNonTaxable = document.querySelector('.border-t.border-gray200.pt-4:nth-child(2) .text-xs.text-gray500.mb-1 span:last-child');
            
            if (recurringSummaryCount) recurringSummaryCount.textContent = count;
            if (recurringSummaryTotal) recurringSummaryTotal.textContent = currencySymbol + totalAmount.toFixed(2);
            if (recurringSummaryTaxable) recurringSummaryTaxable.textContent = 'Taxable: ' + currencySymbol + taxableAmount.toFixed(2);
            if (recurringSummaryNonTaxable) recurringSummaryNonTaxable.textContent = 'Non-taxable: ' + currencySymbol + nonTaxableAmount.toFixed(2);
        }
        
        // Function to reset the entire form
        function resetForm() {
            // Reset the "Add Adjustment" form
            const addAdjustmentForm = document.getElementById('addAdjustmentForm');
            if (addAdjustmentForm) {
                // Only reset form fields, keep employee selection
                addAdjustmentForm.querySelectorAll('input:not([type="hidden"]), select, textarea').forEach(field => {
                    if (field.name !== 'employee_id') {
                        field.value = '';
                    }
                });
                
                // Set default radio buttons
                document.getElementById('taxable_yes').checked = true;
                document.getElementById('one_time_yes').checked = true;
                
                // Reset earning type select
                document.getElementById('earningTypeSelect').value = '';
            }
            
            // Show success message
            showToast('Form has been reset successfully.', 'success');
            
            // Focus on the first field
            const firstField = document.querySelector('#earningTypeSelect');
            if (firstField) {
                firstField.focus();
            }
        }
        
        // Function to show toast messages
        function showToast(message, type = 'info') {
            // Remove existing toast
            const existingToast = document.getElementById('customToast');
            if (existingToast) {
                existingToast.remove();
            }
            
            // Create toast element
            const toast = document.createElement('div');
            toast.id = 'customToast';
            toast.className = `fixed top-4 right-4 z-50 px-6 py-3 rounded-lg shadow-lg text-white font-medium text-sm flex items-center ${
                type === 'success' ? 'bg-green-500' : 
                type === 'error' ? 'bg-red-500' : 
                'bg-blue-500'
            }`;
            
            // Add icon based on type
            const icon = type === 'success' ? 'fa-check-circle' : 
                         type === 'error' ? 'fa-exclamation-circle' : 
                         'fa-info-circle';
            
            toast.innerHTML = `
                <i class="fas ${icon} mr-2"></i>
                ${message}
            `;
            
            // Add to document
            document.body.appendChild(toast);
            
            // Remove toast after 3 seconds
            setTimeout(() => {
                toast.remove();
            }, 3000);
        }
        
        // Handle bulk approval form submission
        document.addEventListener('submit', function(e) {
            if (e.target.id === 'bulkApproveForm') {
                if (!confirm('Are you sure you want to approve all pending adjustments?')) {
                    e.preventDefault();
                }
            }
        });
        
        // Auto-submit forms on approval/unapprove (optional - makes it faster)
        document.addEventListener('click', function(e) {
            if (e.target.closest('.action-btn-approve') || e.target.closest('.action-btn-view')) {
                const form = e.target.closest('form');
                if (form && !form.id) { // Only auto-submit simple approval forms
                    e.preventDefault();
                    form.submit();
                }
            }
        });
    </script>
</body>
</html>