<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payroll Run Details | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#146CAA',
                        primaryLight: '#E1F0FA',
                        secondary: '#6b7280',
                        success: '#107C10',
                        warning: '#F68C1E',
                        danger: '#D83B01',
                        info: '#0078D4',
                        gray50: '#FAF9F8',
                        gray100: '#F3F2F1',
                        gray200: '#EDEBE9',
                        gray300: '#E1DFDD',
                        gray400: '#C8C6C4',
                        gray500: '#A19F9D',
                        gray600: '#605E5C',
                        gray700: '#323130',
                        gray800: '#201F1E',
                        gray900: '#000000'
                    }
                }
            }
        }
    </script>
    <style>
        .quickbooks-table {
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
        }
        .quickbooks-table tbody tr:hover {
            background-color: #F3F2F1;
        }
        .quickbooks-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .info-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            border: 1px solid #EDEBE9;
            margin-bottom: 16px;
        }
        .info-card h3 {
            font-size: 16px;
            font-weight: 600;
            color: #323130;
            margin-bottom: 16px;
            padding-bottom: 12px;
            border-bottom: 1px solid #EDEBE9;
        }
        .info-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #F3F2F1;
        }
        .info-row:last-child {
            border-bottom: none;
        }
        .info-label {
            font-size: 14px;
            color: #605E5C;
        }
        .info-value {
            font-size: 14px;
            color: #323130;
            font-weight: 500;
            text-align: right;
        }
        .timeline {
            position: relative;
            padding-left: 30px;
        }
        .timeline::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 2px;
            background-color: #EDEBE9;
        }
        .timeline-item {
            position: relative;
            margin-bottom: 20px;
        }
        .timeline-item::before {
            content: '';
            position: absolute;
            left: -30px;
            top: 5px;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background-color: #146CAA;
            border: 2px solid white;
        }
        .timeline-date {
            font-size: 12px;
            color: #605E5C;
            margin-bottom: 4px;
        }
        .timeline-content {
            font-size: 14px;
            color: #323130;
        }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Payroll Run Details', 'subtitle' => 'Complete details of payroll run'])

        <main class="p-6">
            <!-- Back Button -->
            <div class="mb-6">
                <a href="{{ route('admin.accounting.payroll.processing.index') }}" 
                   class="text-primary hover:text-primary/90 font-medium text-sm flex items-center">
                    <i class="fas fa-arrow-left mr-2"></i>Back to Payroll Processing
                </a>
            </div>

            <!-- Payroll Run Header -->
            <div class="bg-white rounded-lg border border-gray200 p-6 mb-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between">
                    <div class="mb-4 md:mb-0">
                        <h1 class="text-2xl font-bold text-gray900">{{ $payrollRun->run_number }}</h1>
                        <div class="flex items-center space-x-3 mt-2">
                            <span class="quickbooks-badge bg-blue-100 text-blue-800">
                                {{ $payrollRun->payrollPeriod->period_name ?? 'N/A' }}
                            </span>
                            @php
                                $statusColors = [
                                    'draft' => 'bg-gray-100 text-gray-800',
                                    'processing' => 'bg-yellow-100 text-yellow-800',
                                    'calculated' => 'bg-blue-100 text-blue-800',
                                    'approved' => 'bg-green-100 text-green-800',
                                    'paid' => 'bg-success text-white',
                                    'reversed' => 'bg-red-100 text-red-800'
                                ];
                                $statusColor = $statusColors[$payrollRun->status] ?? 'bg-gray-100 text-gray-800';
                            @endphp
                            <span class="quickbooks-badge {{ $statusColor }}">
                                {{ ucfirst($payrollRun->status) }}
                            </span>
                            @if($payrollRun->status === 'paid')
                                <span class="quickbooks-badge bg-green-100 text-green-800">
                                    Paid on {{ $payrollRun->paid_at->format('M d, Y') }}
                                </span>
                            @endif
                        </div>
                    </div>
                    <div class="flex space-x-3">
                        @if($payrollRun->status === 'approved')
                            <button onclick="processPayment({{ $payrollRun->id }})" 
                                    class="bg-success text-white px-4 py-2 rounded text-sm font-medium hover:bg-success/90 flex items-center">
                                <i class="fas fa-money-check mr-2"></i>Mark as Paid
                            </button>
                        @endif
                        <button onclick="exportPayroll()" 
                                class="border border-gray300 text-gray700 px-4 py-2 rounded text-sm font-medium hover:bg-gray50 flex items-center">
                            <i class="fas fa-download mr-2"></i>Export
                        </button>
                        @if($payrollRun->status === 'paid')
                            <button onclick="exportBankFile()" 
                                    class="border border-blue-300 text-blue-700 px-4 py-2 rounded text-sm font-medium hover:bg-blue-50 flex items-center">
                                <i class="fas fa-university mr-2"></i>Bank File
                            </button>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Main Content Grid -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Left Column -->
                <div class="lg:col-span-2 space-y-6">
                    <!-- Period Information -->
                    <div class="info-card">
                        <h3>Period Information</h3>
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div>
                                <div class="info-label">Period Name</div>
                                <div class="info-value">{{ $payrollRun->payrollPeriod->period_name ?? 'N/A' }}</div>
                            </div>
                            <div>
                                <div class="info-label">Date Range</div>
                                <div class="info-value">{{ $payrollRun->payrollPeriod->formatted_date_range ?? 'N/A' }}</div>
                            </div>
                            <div>
                                <div class="info-label">Run Date</div>
                                <div class="info-value">{{ $payrollRun->run_date->format('M d, Y') }}</div>
                            </div>
                        </div>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <div class="info-label">Cutoff Date</div>
                                <div class="info-value">{{ $payrollRun->payrollPeriod->cutoff_date->format('M d, Y') ?? 'N/A' }}</div>
                            </div>
                            <div>
                                <div class="info-label">Payment Date</div>
                                <div class="info-value">{{ $payrollRun->payrollPeriod->payment_date->format('M d, Y') ?? 'N/A' }}</div>
                            </div>
                        </div>
                    </div>

                    <!-- Financial Summary -->
                    <div class="info-card">
                        <h3>Financial Summary</h3>
                        <div class="space-y-4">
                            <!-- Earnings -->
                            <div>
                                <h4 class="text-sm font-medium text-gray700 mb-2">Earnings</h4>
                                <div class="space-y-1">
                                    <div class="info-row">
                                        <span class="info-label">Basic Salary</span>
                                        <span class="info-value">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->payrollEntries->sum('basic_salary'), 2) }}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Housing Allowance</span>
                                        <span class="info-value">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->payrollEntries->sum('housing_allowance'), 2) }}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Transport Allowance</span>
                                        <span class="info-value">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->payrollEntries->sum('transport_allowance'), 2) }}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Other Allowances</span>
                                        <span class="info-value">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->payrollEntries->sum('other_allowances'), 2) }}</span>
                                    </div>
                                    <div class="info-row" style="border-top: 2px solid #EDEBE9; padding-top: 12px;">
                                        <span class="info-label font-semibold">Total Gross Salary</span>
                                        <span class="info-value font-bold">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_gross_salary, 2) }}</span>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Deductions -->
                            <div>
                                <h4 class="text-sm font-medium text-gray700 mb-2">Deductions</h4>
                                <div class="space-y-1">
                                    <div class="info-row">
                                        <span class="info-label">PAYE Tax</span>
                                        <span class="info-value text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_paye_tax, 2) }}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">NAPSA (Employee)</span>
                                        <span class="info-value text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_napsa_employee, 2) }}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">NHIMA (Employee)</span>
                                        <span class="info-value text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_nhima_employee, 2) }}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Other Deductions</span>
                                        <span class="info-value text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_other_deductions, 2) }}</span>
                                    </div>
                                    <div class="info-row" style="border-top: 2px solid #EDEBE9; padding-top: 12px;">
                                        <span class="info-label font-semibold">Total Deductions</span>
                                        <span class="info-value font-bold text-danger">
                                            {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_paye_tax + $payrollRun->total_napsa_employee + $payrollRun->total_nhima_employee + $payrollRun->total_other_deductions, 2) }}
                                        </span>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Net Pay -->
                            <div class="bg-gray50 p-4 rounded-lg">
                                <div class="flex items-center justify-between">
                                    <span class="text-lg font-semibold text-gray900">Total Net Pay</span>
                                    <span class="text-2xl font-bold text-success">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_net_pay, 2) }}</span>
                                </div>
                            </div>
                            
                            <!-- Employer Contributions -->
                            <div>
                                <h4 class="text-sm font-medium text-gray700 mb-2">Employer Contributions</h4>
                                <div class="space-y-1">
                                    <div class="info-row">
                                        <span class="info-label">NAPSA (Employer)</span>
                                        <span class="info-value">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_napsa_employer, 2) }}</span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">NHIMA (Employer)</span>
                                        <span class="info-value">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_nhima_employer, 2) }}</span>
                                    </div>
                                    <div class="info-row" style="border-top: 2px solid #EDEBE9; padding-top: 12px;">
                                        <span class="info-label font-semibold">Total Employer Cost</span>
                                        <span class="info-value font-bold">
                                            {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_gross_salary + $payrollRun->total_napsa_employer + $payrollRun->total_nhima_employer, 2) }}
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Employee Breakdown -->
                    <div class="info-card">
                        <h3>Employee Breakdown</h3>
                        <div class="overflow-x-auto">
                            <table class="quickbooks-table">
                                <thead>
                                    <tr>
                                        <th>Employee</th>
                                        <th>Gross Salary</th>
                                        <th>PAYE</th>
                                        <th>NAPSA</th>
                                        <th>NHIMA</th>
                                        <th>Net Pay</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($payrollRun->payrollEntries as $entry)
                                        <tr class="hover:bg-gray100 transition-colors">
                                            <td>
                                                <div class="font-medium text-gray900">{{ $entry->employee->first_name }} {{ $entry->employee->last_name }}</div>
                                                <div class="text-xs text-gray500">{{ $entry->employee->employee_number }}</div>
                                            </td>
                                            <td class="font-medium text-gray900">
                                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->gross_salary, 2) }}
                                            </td>
                                            <td class="text-danger font-medium">
                                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->paye_tax, 2) }}
                                            </td>
                                            <td class="text-gray700 font-medium">
                                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->napsa_employee, 2) }}
                                            </td>
                                            <td class="text-gray700 font-medium">
                                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->nhima_employee, 2) }}
                                            </td>
                                            <td class="font-bold text-success">
                                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->net_pay, 2) }}
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Right Column -->
                <div class="space-y-6">
                    <!-- Run Information -->
                    <div class="info-card">
                        <h3>Run Information</h3>
                        <div class="space-y-4">
                            <div>
                                <div class="info-label">Run Number</div>
                                <div class="info-value font-mono">{{ $payrollRun->run_number }}</div>
                            </div>
                            <div>
                                <div class="info-label">Employees Count</div>
                                <div class="info-value">{{ $payrollRun->payrollEntries->count() }}</div>
                            </div>
                            <div>
                                <div class="info-label">Created By</div>
                                <div class="info-value">{{ $payrollRun->creator->name ?? 'System' }}</div>
                            </div>
                            <div>
                                <div class="info-label">Created On</div>
                                <div class="info-value">{{ $payrollRun->created_at->format('M d, Y H:i') }}</div>
                            </div>
                            @if($payrollRun->approved_by)
                                <div>
                                    <div class="info-label">Approved By</div>
                                    <div class="info-value">{{ $payrollRun->approvedBy->name ?? 'N/A' }}</div>
                                </div>
                                <div>
                                    <div class="info-label">Approved On</div>
                                    <div class="info-value">{{ $payrollRun->approved_at->format('M d, Y H:i') }}</div>
                                </div>
                            @endif
                            @if($payrollRun->paid_by)
                                <div>
                                    <div class="info-label">Paid By</div>
                                    <div class="info-value">{{ $payrollRun->paidBy->name ?? 'N/A' }}</div>
                                </div>
                                <div>
                                    <div class="info-label">Paid On</div>
                                    <div class="info-value">{{ $payrollRun->paid_at->format('M d, Y H:i') }}</div>
                                </div>
                                @if($payrollRun->payment_reference)
                                    <div>
                                        <div class="info-label">Payment Reference</div>
                                        <div class="info-value">{{ $payrollRun->payment_reference }}</div>
                                    </div>
                                @endif
                            @endif
                        </div>
                    </div>

                    <!-- Activity Timeline -->
                    <div class="info-card">
                        <h3>Activity Timeline</h3>
                        <div class="timeline">
                            <div class="timeline-item">
                                <div class="timeline-date">{{ $payrollRun->created_at->format('M d, Y H:i') }}</div>
                                <div class="timeline-content">Payroll run created</div>
                            </div>
                            
                            @if($payrollRun->approved_at)
                                <div class="timeline-item">
                                    <div class="timeline-date">{{ $payrollRun->approved_at->format('M d, Y H:i') }}</div>
                                    <div class="timeline-content">Payroll approved for payment</div>
                                </div>
                            @endif
                            
                            @if($payrollRun->paid_at)
                                <div class="timeline-item">
                                    <div class="timeline-date">{{ $payrollRun->paid_at->format('M d, Y H:i') }}</div>
                                    <div class="timeline-content">Payroll marked as paid</div>
                                </div>
                            @endif
                            
                            <div class="timeline-item">
                                <div class="timeline-date">{{ $payrollRun->updated_at->format('M d, Y H:i') }}</div>
                                <div class="timeline-content">Last updated</div>
                            </div>
                        </div>
                    </div>

                    <!-- Actions -->
                    <div class="info-card">
                        <h3>Actions</h3>
                        <div class="space-y-3">
                            @if($payrollRun->status === 'approved')
                                <button onclick="processPayment({{ $payrollRun->id }})" 
                                        class="w-full bg-success text-white px-4 py-2 rounded text-sm font-medium hover:bg-success/90 flex items-center justify-center">
                                    <i class="fas fa-money-check mr-2"></i>Mark as Paid
                                </button>
                            @endif
                            
                            <button onclick="exportPayroll()" 
                                    class="w-full border border-gray300 text-gray700 px-4 py-2 rounded text-sm font-medium hover:bg-gray50 flex items-center justify-center">
                                <i class="fas fa-download mr-2"></i>Export Report
                            </button>
                            
                            @if($payrollRun->status === 'paid')
                                <button onclick="exportBankFile()" 
                                        class="w-full border border-blue-300 text-blue-700 px-4 py-2 rounded text-sm font-medium hover:bg-blue-50 flex items-center justify-center">
                                    <i class="fas fa-university mr-2"></i>Generate Bank File
                                </button>
                            @endif
                            
                            @if($payrollRun->status === 'paid')
                                <button onclick="reversePayroll()" 
                                        class="w-full border border-red-300 text-red-700 px-4 py-2 rounded text-sm font-medium hover:bg-red-50 flex items-center justify-center">
                                    <i class="fas fa-undo mr-2"></i>Reverse Payroll
                                </button>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Payment Processing Modal -->
    <div id="paymentModal" class="fixed inset-0 bg-gray900 bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-money-check-alt text-success"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray900">Mark as Paid</h3>
                        <p class="text-gray600 text-sm">Confirm payment details for payroll run</p>
                    </div>
                </div>
                
                <form id="paymentForm">
                    @csrf
                    <div class="space-y-4">
                        <div>
                            <label class="text-sm font-medium text-gray700 block mb-2">Payment Date</label>
                            <input type="date" 
                                   id="payment_date"
                                   name="payment_date" 
                                   class="quickbooks-input w-full"
                                   value="{{ date('Y-m-d') }}">
                        </div>
                        <div>
                            <label class="text-sm font-medium text-gray700 block mb-2">Payment Reference</label>
                            <input type="text" 
                                   id="payment_reference"
                                   name="payment_reference" 
                                   class="quickbooks-input w-full"
                                   placeholder="e.g., BANK-TRANS-001">
                        </div>
                    </div>
                </form>
            </div>
            
            <div class="border-t border-gray200 p-6 flex justify-end space-x-3">
                <button onclick="closePaymentModal()" class="px-4 py-2 text-gray700 hover:bg-gray100 rounded text-sm font-medium">
                    Cancel
                </button>
                <button onclick="confirmPayment()" class="bg-success text-white px-4 py-2 rounded text-sm font-medium hover:bg-success/90">
                    Mark as Paid
                </button>
            </div>
        </div>
    </div>

    <script>
        function processPayment(runId) {
            document.getElementById('paymentModal').classList.remove('hidden');
        }

        function closePaymentModal() {
            document.getElementById('paymentModal').classList.add('hidden');
        }

        function confirmPayment() {
            const formData = new FormData(document.getElementById('paymentForm'));
            
            fetch(`/admin/accounting/payroll/processing/pay/{{ $payrollRun->id }}`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json',
                },
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification(data.message, 'success');
                    setTimeout(() => {
                        window.location.reload();
                    }, 1500);
                } else {
                    showNotification(data.message, 'error');
                }
                closePaymentModal();
            })
            .catch(error => {
                showNotification('Failed to process payment', 'error');
                closePaymentModal();
            });
        }

        function exportPayroll() {
            const exportType = confirm('Export as:\n\nClick OK for Excel\nClick Cancel for PDF');
            
            if (exportType) {
                // Export to Excel
                window.open(`/admin/accounting/payroll/processing/export/{{ $payrollRun->id }}?format=excel`, '_blank');
            } else {
                // Export to PDF
                window.open(`/admin/accounting/payroll/processing/export/{{ $payrollRun->id }}?format=pdf`, '_blank');
            }
        }

        function exportBankFile() {
            window.open(`/admin/accounting/payroll/processing/export-bank/{{ $payrollRun->id }}`, '_blank');
        }

        function reversePayroll() {
            if (confirm('Are you sure you want to reverse this payroll run? This action cannot be undone.')) {
                fetch(`/admin/accounting/payroll/processing/reverse/{{ $payrollRun->id }}`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json',
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification(data.message, 'success');
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);
                    } else {
                        showNotification(data.message, 'error');
                    }
                })
                .catch(error => {
                    showNotification('Failed to reverse payroll', 'error');
                });
            }
        }

        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-success',
                error: 'bg-danger',
                info: 'bg-primary',
                warning: 'bg-warning'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Add animation styles
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeIn {
                from { opacity: 0; transform: translateY(-10px); }
                to { opacity: 1; transform: translateY(0); }
            }
            @keyframes fadeOut {
                from { opacity: 1; transform: translateY(0); }
                to { opacity: 0; transform: translateY(-10px); }
            }
            .animate-fade-in {
                animation: fadeIn 0.3s ease-out;
            }
            .animate-fade-out {
                animation: fadeOut 0.3s ease-in;
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>