<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payroll Reports | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#146CAA',
                        primaryLight: '#E1F0FA',
                        secondary: '#6b7280',
                        success: '#107C10',
                        warning: '#F68C1E',
                        danger: '#D83B01',
                        info: '#0078D4',
                        gray50: '#FAF9F8',
                        gray100: '#F3F2F1',
                        gray200: '#EDEBE9',
                        gray300: '#E1DFDD',
                        gray400: '#C8C6C4',
                        gray500: '#A19F9D',
                        gray600: '#605E5C',
                        gray700: '#323130',
                        gray800: '#201F1E',
                        gray900: '#000000'
                    }
                }
            }
        }
    </script>
    <style>
        .quickbooks-input {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
        }
        .quickbooks-input:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .report-card {
            background: white;
            border-radius: 8px;
            padding: 24px;
            border: 1px solid #EDEBE9;
            transition: all 0.3s ease;
            height: 100%;
        }
        .report-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            border-color: #146CAA;
        }
        .report-icon {
            width: 48px;
            height: 48px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 16px;
        }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Payroll Reports', 'subtitle' => 'Generate and view payroll reports'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            <!-- Period Selection -->
            <div class="bg-white rounded-lg border border-gray200 p-6 mb-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between space-y-4 md:space-y-0">
                    <div>
                        <h2 class="text-lg font-semibold text-gray900">Report Period</h2>
                        <p class="text-gray600 text-sm mt-1">Select period for payroll reports</p>
                    </div>
                    <div class="flex items-center space-x-4">
                        <select id="financialYearSelect" class="quickbooks-input">
                            <option value="">Select Financial Year</option>
                            @foreach($financialYears ?? [] as $year)
                                <option value="{{ $year->id }}" {{ $currentFinancialYear && $year->id == $currentFinancialYear->id ? 'selected' : '' }}>
                                    {{ $year->year }}
                                </option>
                            @endforeach
                        </select>
                        
                        <select id="periodSelect" class="quickbooks-input">
                            <option value="">All Periods</option>
                            @foreach($payrollPeriods ?? [] as $period)
                                <option value="{{ $period->id }}">
                                    {{ $period->period_name }}
                                </option>
                            @endforeach
                        </select>
                        
                        <button onclick="applyReportFilters()" 
                                class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90">
                            Apply Filters
                        </button>
                    </div>
                </div>
            </div>

            <!-- Report Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
                <!-- Payroll Summary -->
                <a href="{{ route('admin.accounting.payroll.reports.payroll-summary') }}" class="report-card block">
                    <div class="report-icon bg-blue-100">
                        <i class="fas fa-chart-bar text-blue-600 text-xl"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray900 mb-2">Payroll Summary</h3>
                    <p class="text-gray600 text-sm mb-4">
                        Comprehensive overview of payroll expenses, deductions, and net pay for selected period.
                    </p>
                    <div class="flex items-center text-primary text-sm font-medium">
                        <span>Generate Report</span>
                        <i class="fas fa-arrow-right ml-2"></i>
                    </div>
                </a>

                <!-- PAYE Report -->
                <a href="{{ route('admin.accounting.payroll.reports.paye-report') }}" class="report-card block">
                    <div class="report-icon bg-red-100">
                        <i class="fas fa-percentage text-red-600 text-xl"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray900 mb-2">PAYE Report</h3>
                    <p class="text-gray600 text-sm mb-4">
                        Detailed PAYE tax calculations and summaries for ZRA compliance and filings.
                    </p>
                    <div class="flex items-center text-primary text-sm font-medium">
                        <span>Generate Report</span>
                        <i class="fas fa-arrow-right ml-2"></i>
                    </div>
                </a>

                <!-- NAPSA Report -->
                <a href="{{ route('admin.accounting.payroll.reports.napsa-report') }}" class="report-card block">
                    <div class="report-icon bg-green-100">
                        <i class="fas fa-shield-alt text-green-600 text-xl"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray900 mb-2">NAPSA Report</h3>
                    <p class="text-gray600 text-sm mb-4">
                        Employee and employer NAPSA contributions for pension fund remittance.
                    </p>
                    <div class="flex items-center text-primary text-sm font-medium">
                        <span>Generate Report</span>
                        <i class="fas fa-arrow-right ml-2"></i>
                    </div>
                </a>

                <!-- NHIMA Report -->
                <a href="{{ route('admin.accounting.payroll.reports.nhima-report') }}" class="report-card block">
                    <div class="report-icon bg-purple-100">
                        <i class="fas fa-heartbeat text-purple-600 text-xl"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray900 mb-2">NHIMA Report</h3>
                    <p class="text-gray600 text-sm mb-4">
                        Health insurance contributions for NHIMA compliance and monthly submissions.
                    </p>
                    <div class="flex items-center text-primary text-sm font-medium">
                        <span>Generate Report</span>
                        <i class="fas fa-arrow-right ml-2"></i>
                    </div>
                </a>

                <!-- Employee Payslips -->
                <div class="report-card">
                    <div class="report-icon bg-yellow-100">
                        <i class="fas fa-file-invoice-dollar text-yellow-600 text-xl"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray900 mb-2">Employee Payslips</h3>
                    <p class="text-gray600 text-sm mb-4">
                        Generate individual payslips for employees with detailed earnings and deductions.
                    </p>
                    <div class="space-y-3">
                        <select id="employeeSelect" class="quickbooks-input w-full">
                            <option value="">Select Employee</option>
                            @foreach($employees ?? [] as $employee)
                                <option value="{{ $employee->id }}">
                                    {{ $employee->employee_number }} - {{ $employee->first_name }} {{ $employee->last_name }}
                                </option>
                            @endforeach
                        </select>
                        <button onclick="generatePayslip()" 
                                class="w-full bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90">
                            Generate Payslip
                        </button>
                    </div>
                </div>

                <!-- Bank Payment File -->
                <div class="report-card">
                    <div class="report-icon bg-gray-100">
                        <i class="fas fa-university text-gray-600 text-xl"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray900 mb-2">Bank Payment File</h3>
                    <p class="text-gray600 text-sm mb-4">
                        Generate CSV file for bank upload with employee bank details and net pay amounts.
                    </p>
                    <div class="space-y-3">
                        <select id="payrollRunSelect" class="quickbooks-input w-full">
                            <option value="">Select Payroll Run</option>
                            @foreach($payrollRuns ?? [] as $run)
                                @if($run->status === 'approved' || $run->status === 'paid')
                                    <option value="{{ $run->id }}">
                                        {{ $run->run_number }} - {{ $run->payrollPeriod->period_name ?? 'N/A' }}
                                    </option>
                                @endif
                            @endforeach
                        </select>
                        <button onclick="generateBankFile()" 
                                class="w-full bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90">
                            Generate Bank File
                        </button>
                    </div>
                </div>
            </div>

            <!-- Recent Payroll Runs -->
            <div class="bg-white rounded-lg border border-gray200 overflow-hidden">
                <div class="border-b border-gray200 p-6">
                    <h3 class="text-lg font-semibold text-gray900">Recent Payroll Runs</h3>
                    <p class="text-gray600 text-sm mt-1">Latest processed payroll runs for quick access</p>
                </div>
                
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead>
                            <tr class="bg-gray50">
                                <th class="text-left py-3 px-6 text-sm font-medium text-gray700">Run #</th>
                                <th class="text-left py-3 px-6 text-sm font-medium text-gray700">Period</th>
                                <th class="text-left py-3 px-6 text-sm font-medium text-gray700">Date</th>
                                <th class="text-left py-3 px-6 text-sm font-medium text-gray700">Employees</th>
                                <th class="text-left py-3 px-6 text-sm font-medium text-gray700">Gross Salary</th>
                                <th class="text-left py-3 px-6 text-sm font-medium text-gray700">Net Pay</th>
                                <th class="text-left py-3 px-6 text-sm font-medium text-gray700">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray200">
                            @forelse($recentPayrollRuns ?? [] as $run)
                                <tr class="hover:bg-gray50">
                                    <td class="py-3 px-6">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                            {{ $run->run_number }}
                                        </span>
                                    </td>
                                    <td class="py-3 px-6 text-sm text-gray900">
                                        {{ $run->payrollPeriod->period_name ?? 'N/A' }}
                                    </td>
                                    <td class="py-3 px-6 text-sm text-gray600">
                                        {{ $run->run_date->format('M d, Y') }}
                                    </td>
                                    <td class="py-3 px-6 text-sm text-gray600">
                                        {{ $run->payrollEntries->count() ?? 0 }}
                                    </td>
                                    <td class="py-3 px-6 text-sm font-medium text-gray900">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($run->total_gross_salary, 2) }}
                                    </td>
                                    <td class="py-3 px-6 text-sm font-medium text-success">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($run->total_net_pay, 2) }}
                                    </td>
                                    <td class="py-3 px-6 text-sm">
                                        <a href="{{ route('admin.accounting.payroll.processing.show', $run->id) }}" 
                                           class="text-primary hover:text-primary/90 font-medium">
                                            View Details
                                        </a>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="7" class="py-8 text-center text-gray600">
                                        No recent payroll runs found
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>

    <script>
        function applyReportFilters() {
            const financialYearId = document.getElementById('financialYearSelect').value;
            const periodId = document.getElementById('periodSelect').value;
            
            let url = window.location.href.split('?')[0];
            const params = new URLSearchParams();
            
            if (financialYearId) params.append('financial_year_id', financialYearId);
            if (periodId) params.append('period_id', periodId);
            
            const queryString = params.toString();
            if (queryString) {
                url += '?' + queryString;
            }
            
            window.location.href = url;
        }
        
        function generatePayslip() {
            const employeeId = document.getElementById('employeeSelect').value;
            if (!employeeId) {
                showNotification('Please select an employee', 'error');
                return;
            }
            
            const periodId = document.getElementById('periodSelect').value;
            if (!periodId) {
                showNotification('Please select a payroll period', 'error');
                return;
            }
            
            window.open(`/admin/accounting/payroll/reports/employee-payslip/${employeeId}?period_id=${periodId}`, '_blank');
        }
        
        function generateBankFile() {
            const runId = document.getElementById('payrollRunSelect').value;
            if (!runId) {
                showNotification('Please select a payroll run', 'error');
                return;
            }
            
            window.open(`/admin/accounting/payroll/processing/export-bank/${runId}`, '_blank');
        }
        
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-success',
                error: 'bg-danger',
                info: 'bg-primary',
                warning: 'bg-warning'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Add animation styles
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeIn {
                from { opacity: 0; transform: translateY(-10px); }
                to { opacity: 1; transform: translateY(0); }
            }
            @keyframes fadeOut {
                from { opacity: 1; transform: translateY(0); }
                to { opacity: 0; transform: translateY(-10px); }
            }
            .animate-fade-in {
                animation: fadeIn 0.3s ease-out;
            }
            .animate-fade-out {
                animation: fadeOut 0.3s ease-in;
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>