<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Credit Score Tiers | IMMIA FINANCE</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Credit Score Tiers', 'subtitle' => 'Manage credit score tiers and loan parameters'])

        <main class="p-6">
            <!-- Action Bar -->
            <div class="flex justify-between items-center mb-6">
                <div>
                    <h2 class="text-xl font-bold text-gray-900">All Credit Score Tiers</h2>
                    <p class="text-gray-600">{{ $tiers->count() }} credit tiers found</p>
                </div>
                <button onclick="openCreateModal()" 
                   class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-semibold flex items-center space-x-2">
                    <i class="fas fa-plus"></i>
                    <span>New Credit Tier</span>
                </button>
            </div>

            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg mb-6">
                    {{ session('success') }}
                </div>
            @endif

            @if(session('error'))
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-6">
                    {{ session('error') }}
                </div>
            @endif

            <!-- Credit Tiers Table -->
            <div class="bg-white rounded-xl shadow-sm border">
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50 border-b">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tier Name</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Score Range</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Interest Rate</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Max Loan Amount</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Max Term</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            @forelse($tiers as $tier)
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm font-medium text-gray-900">{{ $tier->name }}</div>
                                    @if($tier->description)
                                    <div class="text-sm text-gray-500">{{ Str::limit($tier->description, 50) }}</div>
                                    @endif
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    @php
                                        $scoreColors = [
                                            'Excellent' => 'bg-green-100 text-green-800',
                                            'Good' => 'bg-blue-100 text-blue-800',
                                            'Fair' => 'bg-yellow-100 text-yellow-800',
                                            'Poor' => 'bg-orange-100 text-orange-800',
                                            'Very Poor' => 'bg-red-100 text-red-800'
                                        ];
                                    @endphp
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $scoreColors[$tier->name] ?? 'bg-gray-100 text-gray-800' }}">
                                        {{ $tier->min_score }} - {{ $tier->max_score }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                    {{ number_format($tier->base_interest_rate, 2) }}%
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                    ZMW {{ number_format($tier->max_loan_amount, 2) }}
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    {{ $tier->max_loan_term }} months
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $tier->is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' }}">
                                        {{ $tier->is_active ? 'Active' : 'Inactive' }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div class="flex items-center space-x-2">
                                        <button onclick="openEditModal({{ $tier->id }})" 
                                                class="text-green-600 hover:text-green-900">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button onclick="toggleTierStatus({{ $tier->id }})" 
                                                class="text-{{ $tier->is_active ? 'orange' : 'green' }}-600 hover:text-{{ $tier->is_active ? 'orange' : 'green' }}-900">
                                            <i class="fas fa-{{ $tier->is_active ? 'pause' : 'play' }}"></i>
                                        </button>
                                        <button onclick="confirmDelete({{ $tier->id }})" 
                                                class="text-red-600 hover:text-red-900">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            @empty
                            <tr>
                                <td colspan="7" class="px-6 py-4 text-center text-sm text-gray-500">
                                    <div class="flex flex-col items-center justify-center py-8">
                                        <i class="fas fa-chart-line text-4xl text-gray-300 mb-4"></i>
                                        <p class="text-gray-500">No credit score tiers found</p>
                                        <button onclick="openCreateModal()" 
                                                class="mt-2 text-blue-600 hover:text-blue-700 font-medium">
                                            Create your first credit tier
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>

    <!-- Create/Edit Modal -->
    <div id="tierModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <div class="flex justify-between items-center pb-3 border-b">
                    <h3 id="modalTitle" class="text-lg font-medium text-gray-900">Add Credit Score Tier</h3>
                    <button onclick="closeModal()" class="text-gray-400 hover:text-gray-600">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <form id="tierForm" class="mt-4 space-y-4">
                    @csrf
                    <input type="hidden" id="tier_id" name="id">
                    
                    <div>
                        <label for="name" class="block text-sm font-medium text-gray-700">Tier Name *</label>
                        <input type="text" id="name" name="name" required 
                               class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label for="min_score" class="block text-sm font-medium text-gray-700">Min Score *</label>
                            <input type="number" id="min_score" name="min_score" min="300" max="849" required 
                                   class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        </div>
                        <div>
                            <label for="max_score" class="block text-sm font-medium text-gray-700">Max Score *</label>
                            <input type="number" id="max_score" name="max_score" min="301" max="850" required 
                                   class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        </div>
                    </div>
                    
                    <div>
                        <label for="base_interest_rate" class="block text-sm font-medium text-gray-700">Base Interest Rate (%) *</label>
                        <input type="number" id="base_interest_rate" name="base_interest_rate" step="0.01" min="0" max="100" required 
                               class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                    </div>
                    
                    <div>
                        <label for="max_loan_amount" class="block text-sm font-medium text-gray-700">Max Loan Amount (ZMW) *</label>
                        <input type="number" id="max_loan_amount" name="max_loan_amount" step="0.01" min="0" required 
                               class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                    </div>
                    
                    <div>
                        <label for="max_loan_term" class="block text-sm font-medium text-gray-700">Max Loan Term (months) *</label>
                        <input type="number" id="max_loan_term" name="max_loan_term" min="1" max="120" required 
                               class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                    </div>
                    
                    <div>
                        <label for="description" class="block text-sm font-medium text-gray-700">Description</label>
                        <textarea id="description" name="description" rows="3" 
                                  class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500"></textarea>
                    </div>
                    
                    <div class="flex justify-end space-x-3 pt-4 border-t">
                        <button type="button" onclick="closeModal()" class="bg-gray-300 hover:bg-gray-400 text-gray-800 font-medium py-2 px-4 rounded-md transition duration-200">Cancel</button>
                        <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white font-medium py-2 px-4 rounded-md transition duration-200">Save Tier</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="deleteModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100">
                    <i class="fas fa-exclamation-triangle text-red-600"></i>
                </div>
                <div class="mt-3 text-center">
                    <h3 class="text-lg font-medium text-gray-900">Delete Tier</h3>
                    <div class="mt-2 px-7 py-3">
                        <p class="text-sm text-gray-500">Are you sure you want to delete this credit score tier? This action cannot be undone.</p>
                    </div>
                </div>
                <div class="flex justify-center space-x-3 mt-4">
                    <button onclick="closeDeleteModal()" class="bg-gray-300 hover:bg-gray-400 text-gray-800 font-medium py-2 px-4 rounded-md transition duration-200">Cancel</button>
                    <button onclick="deleteTier()" class="bg-red-600 hover:bg-red-700 text-white font-medium py-2 px-4 rounded-md transition duration-200">Delete</button>
                </div>
            </div>
        </div>
    </div>

    <script>
    let currentTierId = null;

    // Modal functions
    function openCreateModal() {
        document.getElementById('modalTitle').textContent = 'Add Credit Score Tier';
        document.getElementById('tierForm').reset();
        document.getElementById('tier_id').value = '';
        document.getElementById('tierModal').classList.remove('hidden');
    }

    function openEditModal(tierId) {
        fetch(`/admin/settings/credit-score-tiers/${tierId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const tier = data.tier;
                    document.getElementById('modalTitle').textContent = 'Edit Credit Score Tier';
                    document.getElementById('tier_id').value = tier.id;
                    document.getElementById('name').value = tier.name;
                    document.getElementById('min_score').value = tier.min_score;
                    document.getElementById('max_score').value = tier.max_score;
                    document.getElementById('base_interest_rate').value = tier.base_interest_rate;
                    document.getElementById('max_loan_amount').value = tier.max_loan_amount;
                    document.getElementById('max_loan_term').value = tier.max_loan_term;
                    document.getElementById('description').value = tier.description || '';
                    document.getElementById('tierModal').classList.remove('hidden');
                }
            });
    }

    function closeModal() {
        document.getElementById('tierModal').classList.add('hidden');
    }

    function confirmDelete(tierId) {
        currentTierId = tierId;
        document.getElementById('deleteModal').classList.remove('hidden');
    }

    function closeDeleteModal() {
        document.getElementById('deleteModal').classList.add('hidden');
        currentTierId = null;
    }

    function deleteTier() {
        if (!currentTierId) return;
        
        fetch(`/admin/settings/credit-score-tiers/${currentTierId}`, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Tier deleted successfully!', 'success');
                location.reload();
            } else {
                showAlert(data.message, 'error');
            }
            closeDeleteModal();
        });
    }

    function toggleTierStatus(tierId) {
        fetch(`/admin/settings/credit-score-tiers/${tierId}/toggle-status`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Tier status updated successfully!', 'success');
                location.reload();
            } else {
                showAlert(data.message, 'error');
            }
        });
    }

    // Form submission
    document.getElementById('tierForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const tierId = document.getElementById('tier_id').value;
        const url = tierId ? `/admin/settings/credit-score-tiers/${tierId}` : '/admin/settings/credit-score-tiers';
        const method = tierId ? 'PUT' : 'POST';
        
        fetch(url, {
            method: method,
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(Object.fromEntries(formData))
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert(data.message, 'success');
                closeModal();
                location.reload();
            } else {
                showAlert(data.message, 'error');
            }
        })
        .catch(error => {
            showAlert('An error occurred. Please try again.', 'error');
        });
    });

    // Alert function
    function showAlert(message, type) {
        // Create alert element
        const alertDiv = document.createElement('div');
        alertDiv.className = `mb-6 px-4 py-3 rounded-lg border ${
            type === 'success' ? 'bg-green-100 border-green-400 text-green-700' : 'bg-red-100 border-red-400 text-red-700'
        }`;
        
        alertDiv.innerHTML = `
            <div class="flex justify-between items-center">
                <div class="flex items-center">
                    <i class="fas fa-${type === 'success' ? 'check' : 'exclamation-triangle'} mr-2"></i>
                    <span>${message}</span>
                </div>
                <button onclick="this.parentElement.parentElement.remove()" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        // Insert after the action bar
        const actionBar = document.querySelector('main .mb-6');
        actionBar.parentNode.insertBefore(alertDiv, actionBar.nextSibling);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (alertDiv.parentNode) {
                alertDiv.remove();
            }
        }, 5000);
    }

    // Close modals when clicking outside
    document.addEventListener('click', function(e) {
        if (e.target.id === 'tierModal') {
            closeModal();
        }
        if (e.target.id === 'deleteModal') {
            closeDeleteModal();
        }
    });
    </script>
</body>
</html>