<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Currency Rate | IMMIA FINANCE</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Edit Currency Rate', 'subtitle' => 'Update exchange rate between currencies'])

        <main class="p-6">
            <div class="max-w-2xl mx-auto">
                <!-- Back Button -->
                <div class="mb-6">
                    <a href="{{ route('admin.accounting.settings.currency-rates.index') }}" 
                       class="inline-flex items-center text-blue-600 hover:text-blue-700 font-medium transition-colors duration-200">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Back to Currency Rates
                    </a>
                </div>

                <!-- Success/Error Messages -->
                @if($errors->any())
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-6">
                        <div class="flex items-center mb-2">
                            <i class="fas fa-exclamation-triangle mr-2"></i>
                            <strong>Please fix the following errors:</strong>
                        </div>
                        <ul class="list-disc list-inside text-sm">
                            @foreach($errors->all() as $error)
                                <li>{{ $error }}</li>
                            @endforeach
                        </ul>
                    </div>
                @endif

                @if(session('success'))
                    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg mb-6 flex items-center justify-between">
                        <div class="flex items-center">
                            <i class="fas fa-check-circle mr-2"></i>
                            <span>{{ session('success') }}</span>
                        </div>
                        <button type="button" onclick="this.parentElement.style.display='none'" class="text-green-700 hover:text-green-900">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                @endif

                @if(session('error'))
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-6 flex items-center justify-between">
                        <div class="flex items-center">
                            <i class="fas fa-exclamation-circle mr-2"></i>
                            <span>{{ session('error') }}</span>
                        </div>
                        <button type="button" onclick="this.parentElement.style.display='none'" class="text-red-700 hover:text-red-900">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                @endif

                <!-- Form Card -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-semibold text-gray-900">Edit Exchange Rate</h3>
                        <p class="text-sm text-gray-600 mt-1">Update exchange rate between two currencies</p>
                    </div>
                    
                    <form action="{{ route('admin.accounting.settings.currency-rates.update', $currencyRate->id) }}" method="POST">
                        @csrf
                        @method('PUT')
                        <div class="p-6 space-y-6">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <!-- Base Currency -->
                                <div>
                                    <label for="base_currency_id" class="block text-sm font-medium text-gray-700 mb-2">
                                        Base Currency *
                                    </label>
                                    <select name="base_currency_id" id="base_currency_id" 
                                            class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors duration-200" required>
                                        <option value="">Select Base Currency</option>
                                        @foreach($currencies as $currency)
                                            <option value="{{ $currency->id }}" {{ $currencyRate->base_currency_id == $currency->id ? 'selected' : '' }}>
                                                {{ $currency->code }} - {{ $currency->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('base_currency_id')
                                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                    <p class="text-xs text-gray-500 mt-1">The currency you're converting FROM</p>
                                </div>

                                <!-- Target Currency -->
                                <div>
                                    <label for="target_currency_id" class="block text-sm font-medium text-gray-700 mb-2">
                                        Target Currency *
                                    </label>
                                    <select name="target_currency_id" id="target_currency_id" 
                                            class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors duration-200" required>
                                        <option value="">Select Target Currency</option>
                                        @foreach($currencies as $currency)
                                            <option value="{{ $currency->id }}" {{ $currencyRate->target_currency_id == $currency->id ? 'selected' : '' }}>
                                                {{ $currency->code }} - {{ $currency->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('target_currency_id')
                                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                    <p class="text-xs text-gray-500 mt-1">The currency you're converting TO</p>
                                </div>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <!-- Exchange Rate -->
                                <div>
                                    <label for="rate" class="block text-sm font-medium text-gray-700 mb-2">
                                        Exchange Rate *
                                    </label>
                                    <input type="number" name="rate" id="rate" step="0.000001" min="0.000001"
                                           value="{{ old('rate', $currencyRate->rate) }}"
                                           class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors duration-200"
                                           placeholder="0.000000" required>
                                    @error('rate')
                                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                    <p class="text-xs text-gray-500 mt-1">1 Base = X Target</p>
                                </div>

                                <!-- Rate Date -->
                                <div>
                                    <label for="rate_date" class="block text-sm font-medium text-gray-700 mb-2">
                                        Rate Date *
                                    </label>
                                    <input type="date" name="rate_date" id="rate_date" 
                                           value="{{ old('rate_date', $currencyRate->rate_date->format('Y-m-d')) }}"
                                           class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors duration-200" required>
                                    @error('rate_date')
                                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>
                            </div>

                            <!-- Current Rate Display -->
                            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                <h4 class="text-sm font-medium text-blue-900 mb-2 flex items-center">
                                    <i class="fas fa-info-circle mr-2"></i>
                                    Current Rate
                                </h4>
                                <p class="text-lg font-semibold text-blue-700">
                                    1 {{ $currencyRate->baseCurrency->code ?? 'N/A' }} = {{ number_format($currencyRate->rate, 6) }} {{ $currencyRate->targetCurrency->code ?? 'N/A' }}
                                </p>
                                <p class="text-xs text-blue-600 mt-1">
                                    Last updated: {{ $currencyRate->updated_at->format('M d, Y \a\t h:i A') }}
                                </p>
                            </div>

                            <!-- Rate Preview -->
                            <div class="bg-gray-50 border border-gray-200 rounded-lg p-4">
                                <h4 class="text-sm font-medium text-gray-700 mb-2 flex items-center">
                                    <i class="fas fa-eye mr-2"></i>
                                    Rate Preview
                                </h4>
                                <p class="text-lg font-semibold text-gray-900" id="ratePreview">
                                    1 <span id="baseCurrencyCode">{{ $currencyRate->baseCurrency->code ?? '[Base]' }}</span> = 
                                    <span id="rateValue">{{ number_format($currencyRate->rate, 6) }}</span> 
                                    <span id="targetCurrencyCode">{{ $currencyRate->targetCurrency->code ?? '[Target]' }}</span>
                                </p>
                            </div>
                        </div>

                        <!-- Form Actions -->
                        <div class="px-6 py-4 bg-gray-50 border-t border-gray-200 rounded-b-xl flex justify-end space-x-3">
                            <a href="{{ route('admin.accounting.settings.currency-rates.index') }}" 
                               class="px-6 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 font-medium transition-colors duration-200 flex items-center space-x-2">
                                <i class="fas fa-times"></i>
                                <span>Cancel</span>
                            </a>
                            <button type="submit" 
                                    class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium flex items-center space-x-2 transition-colors duration-200">
                                <i class="fas fa-save"></i>
                                <span>Update Exchange Rate</span>
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Additional Information Card -->
                <div class="mt-6 bg-white rounded-xl shadow-sm border border-gray-200">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-semibold text-gray-900">Rate Information</h3>
                    </div>
                    <div class="p-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 text-sm text-gray-600">
                            <div>
                                <label class="font-medium text-gray-500">Created By:</label>
                                <p>{{ $currencyRate->creator->name ?? 'System' }}</p>
                            </div>
                            <div>
                                <label class="font-medium text-gray-500">Created At:</label>
                                <p>{{ $currencyRate->created_at->format('M d, Y \a\t h:i A') }}</p>
                            </div>
                            <div>
                                <label class="font-medium text-gray-500">Last Updated:</label>
                                <p>{{ $currencyRate->updated_at->format('M d, Y \a\t h:i A') }}</p>
                            </div>
                            <div>
                                <label class="font-medium text-gray-500">Rate ID:</label>
                                <p class="font-mono">{{ $currencyRate->id }}</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Update rate preview in real-time
        document.addEventListener('DOMContentLoaded', function() {
            const baseCurrencySelect = document.getElementById('base_currency_id');
            const targetCurrencySelect = document.getElementById('target_currency_id');
            const rateInput = document.getElementById('rate');
            const rateValue = document.getElementById('rateValue');
            const baseCurrencyCode = document.getElementById('baseCurrencyCode');
            const targetCurrencyCode = document.getElementById('targetCurrencyCode');

            function updateRatePreview() {
                const baseCurrency = baseCurrencySelect.options[baseCurrencySelect.selectedIndex];
                const targetCurrency = targetCurrencySelect.options[targetCurrencySelect.selectedIndex];
                const baseCode = baseCurrency.text.split(' - ')[0] || '[Base]';
                const targetCode = targetCurrency.text.split(' - ')[0] || '[Target]';
                const rate = rateInput.value || '0.000000';
                
                baseCurrencyCode.textContent = baseCode;
                targetCurrencyCode.textContent = targetCode;
                rateValue.textContent = parseFloat(rate).toFixed(6);
            }

            baseCurrencySelect.addEventListener('change', updateRatePreview);
            targetCurrencySelect.addEventListener('change', updateRatePreview);
            rateInput.addEventListener('input', updateRatePreview);
            
            // Initial update
            updateRatePreview();
        });
    </script>
</body>
</html>