@php
    $pageTitle = 'Disburse Loan Application';
    $pageSubtitle = 'Process loan disbursement for application #' . $application->application_number;
    $themeBlue = '#0077C5';
    $themeBlueDark = '#005A94';
@endphp

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{{ $pageTitle }} | IMMIA FINANCE</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

    <style>
        :root {
            --zedbank-blue: {{ $themeBlue }}; 
            --zedbank-blue-dark: {{ $themeBlueDark }}; 
        }
        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { .lg\:ml-64 { margin-left: 0 !important; } }
        .compact-input { padding: 10px 14px; font-size: 0.875rem; } 
        .value-text { font-size: 0.95rem; color: #111827; font-weight: 500; }
        .date-info { font-size: 0.8rem; color: #6b7280; margin-top: 0.1rem; }
        .backdate-indicator {
            animation: pulse 2s infinite;
        }
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.7; }
            100% { opacity: 1; }
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64 min-h-screen">
        
        @include('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ])

        <main class="p-6">
            <div class="max-w-4xl mx-auto">
                <div class="mb-6">
                    <a href="{{ route('admin.applications.approved') }}" 
                    class="inline-flex items-center text-zedbank-blue font-medium hover:underline">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Back to Approved Applications
                    </a>
                </div>

                @if(session('success'))
                    <div class="mb-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg flex items-center">
                        <i class="fas fa-check-circle mr-2"></i>
                        {{ session('success') }}
                    </div>
                @endif

                @if(session('error'))
                    <div class="mb-6 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg flex items-center">
                        <i class="fas fa-exclamation-circle mr-2"></i>
                        {{ session('error') }}
                    </div>
                @endif

                <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden mb-6">
                    <div class="p-4 border-b bg-gray-50 flex justify-between items-center">
                        <h2 class="text-lg font-bold text-gray-800">Financial Summary</h2>
                        <span class="px-3 py-1 bg-green-100 text-green-800 rounded-full text-xs font-bold uppercase">
                            {{ $application->status }}
                        </span>
                    </div>
                    
                    {{-- TIMELINE DATES SECTION --}}
                    @php
                        // Calculate backdate information
                        $submittedAt = $application->submitted_at;
                        $approvedAt = $application->approved_at;
                        
                        $isAppBackdated = $submittedAt && $submittedAt->isPast() && $submittedAt->diffInDays(now()) > 0;
                        $appDaysAgo = $isAppBackdated ? $submittedAt->diffInDays(now()) : 0;
                        
                        $isApprovalBackdated = $approvedAt && $approvedAt->isPast() && $approvedAt->diffInDays(now()) > 0;
                        $approvalDaysAgo = $isApprovalBackdated ? $approvedAt->diffInDays(now()) : 0;
                        
                        // Calculate processing timeline
                        $processingTime = $approvedAt && $submittedAt ? $approvedAt->diffInDays($submittedAt) : null;
                    @endphp
                    
                    <div class="p-4 border-b border-gray-100 bg-gradient-to-r from-blue-50 to-indigo-50">
                        <h3 class="text-sm font-bold text-gray-700 mb-3 uppercase">Application Timeline</h3>
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                            {{-- Application Date --}}
                            <div class="bg-white p-3 rounded-lg border border-gray-200">
                                <div class="flex justify-between items-start mb-1">
                                    <p class="text-xs text-gray-500 font-bold uppercase">Application Date</p>
                                    @if($isAppBackdated)
                                        <span class="inline-flex items-center px-2 py-0.5 rounded-full text-[10px] font-medium bg-yellow-100 text-yellow-800 backdate-indicator">
                                            <i class="fas fa-history mr-1"></i> Backdated
                                        </span>
                                    @endif
                                </div>
                                <p class="value-text text-gray-900">
                                    {{ $submittedAt ? $submittedAt->format('M d, Y') : 'N/A' }}
                                </p>
                                @if($isAppBackdated)
                                    <p class="date-info">
                                        <i class="fas fa-clock mr-1"></i> {{ $appDaysAgo }} {{ $appDaysAgo == 1 ? 'day' : 'days' }} ago
                                    </p>
                                @endif
                            </div>
                            
                            {{-- Approval Date --}}
                            <div class="bg-white p-3 rounded-lg border border-gray-200">
                                <div class="flex justify-between items-start mb-1">
                                    <p class="text-xs text-gray-500 font-bold uppercase">Approval Date</p>
                                    @if($isApprovalBackdated)
                                        <span class="inline-flex items-center px-2 py-0.5 rounded-full text-[10px] font-medium bg-yellow-100 text-yellow-800 backdate-indicator">
                                            <i class="fas fa-history mr-1"></i> Backdated
                                        </span>
                                    @endif
                                </div>
                                <p class="value-text text-gray-900">
                                    {{ $approvedAt ? $approvedAt->format('M d, Y') : 'Pending' }}
                                </p>
                                @if($isApprovalBackdated)
                                    <p class="date-info">
                                        <i class="fas fa-clock mr-1"></i> {{ $approvalDaysAgo }} {{ $approvalDaysAgo == 1 ? 'day' : 'days' }} ago
                                    </p>
                                @endif
                            </div>
                            
                            {{-- Processing Time --}}
                            <div class="bg-white p-3 rounded-lg border border-gray-200">
                                <p class="text-xs text-gray-500 font-bold uppercase mb-1">Processing Time</p>
                                <p class="value-text text-gray-900">
                                    @if($processingTime !== null)
                                        {{ $processingTime }} {{ $processingTime == 1 ? 'day' : 'days' }}
                                    @else
                                        N/A
                                    @endif
                                </p>
                                <p class="date-info">
                                    @if($submittedAt && $approvedAt)
                                        From {{ $submittedAt->format('M d') }} to {{ $approvedAt->format('M d') }}
                                    @else
                                        Timeline not available
                                    @endif
                                </p>
                            </div>
                        </div>
                    </div>

                    <div class="p-6 grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div>
                            <p class="text-xs text-gray-500 uppercase font-bold">Borrower</p>
                            <p class="font-semibold text-gray-800">{{ $application->customer->first_name }} {{ $application->customer->surname }}</p>
                        </div>
                        <div>
                            <p class="text-xs text-gray-500 uppercase font-bold">Loan Product</p>
                            <p class="font-semibold text-gray-800">{{ $application->product->product ?? 'N/A' }}</p>
                        </div>
                        <div>
                            <p class="text-xs text-gray-500 uppercase font-bold">Principal Amount</p>
                            <p class="font-bold text-zedbank-blue">ZMW {{ number_format($application->loan_amount, 2) }}</p>
                        </div>
                        <div>
                            <p class="text-xs text-gray-500 uppercase font-bold">Interest Rate</p>
                            <p class="font-semibold text-gray-800">{{ number_format($application->interest_rate, 2) }}%</p>
                        </div>
                        <div>
                            <p class="text-xs text-gray-500 uppercase font-bold">Payment Frequency</p>
                            <p class="font-semibold text-gray-800 capitalize">{{ str_replace('_', ' ', $application->payment_frequency) }}</p>
                        </div>
                        <div>
                            <p class="text-xs text-gray-500 uppercase font-bold">Tenure</p>
                            <p class="font-semibold text-gray-800">{{ $application->loan_tenure }} {{ in_array($application->payment_frequency, ['daily', 'one_off']) ? 'Days' : 'Months' }}</p>
                        </div>
                        <div>
                            <p class="text-xs text-gray-500 uppercase font-bold">Expected Installment</p>
                            <p class="font-bold text-green-600">ZMW {{ number_format($application->installment_amount, 2) }}</p>
                        </div>
                    </div>

                    <div class="p-6 border-t border-gray-100 bg-gray-50/30">
                        <h3 class="text-sm font-bold text-gray-700 mb-3 uppercase italic">Fees & Penalties</h3>
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div class="bg-white p-3 rounded-lg border border-gray-100">
                                <p class="text-xs text-gray-500 font-bold uppercase">Processing Fee</p>
                                <p class="value-text">{{ number_format($application->processing_fee, 2) }}% <span class="text-[10px] text-gray-400">({{ str_replace('_', ' ', $application->processing_fee_basis) }})</span></p>
                            </div>
                            <div class="bg-white p-3 rounded-lg border border-gray-100">
                                <p class="text-xs text-gray-500 font-bold uppercase">Admin Fee</p>
                                <p class="value-text">ZMW {{ number_format($application->adminfee, 2) }}</p>
                            </div>
                            <div class="bg-white p-3 rounded-lg border border-gray-100">
                                <p class="text-xs text-gray-500 font-bold uppercase">Insurance Fee</p>
                                <p class="value-text">ZMW {{ number_format($application->insurancefee, 2) }}</p>
                            </div>
                            <div class="bg-red-50 p-3 rounded-lg border border-red-100">
                                <p class="text-xs text-red-500 font-bold uppercase">Penalty Rate</p>
                                <p class="value-text text-red-800">{{ number_format($application->penalty_rate, 2) }}%</p>
                            </div>
                            <div class="bg-red-50 p-3 rounded-lg border border-red-100 md:col-span-2">
                                <p class="text-xs text-red-500 font-bold uppercase">Penalty Basis</p>
                                <p class="value-text text-red-800 capitalize">{{ str_replace('_', ' ', $application->penalty_basis) }}</p>
                            </div>
                        </div>
                    </div>

                    <div class="p-6 border-t border-gray-100">
                        <h3 class="text-sm font-bold text-gray-700 mb-3 uppercase italic">Disbursement Destination (Requested)</h3>
                        <div class="bg-blue-50 p-4 rounded-lg border border-blue-100">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <p class="text-xs text-blue-500 font-bold uppercase">Payment Method</p>
                                    <p class="font-bold text-blue-900 text-lg">{{ $application->suggestedpaymentmethod }}</p>
                                </div>
                                @if($application->suggestedpaymentmethod == 'Bank')
                                    <div>
                                        <p class="text-xs text-blue-500 font-bold uppercase">Account Details</p>
                                        <div class="text-sm text-blue-900">
                                            <p><strong>Bank:</strong> {{ $application->bank_name }}</p>
                                            <p><strong>Holder:</strong> {{ $application->account_holder }}</p>
                                            <p><strong>Account:</strong> {{ $application->account_number }}</p>
                                            <p><strong>Branch/Swift:</strong> {{ $application->bank_branch }} / {{ $application->swift_code ?? 'N/A' }}</p>
                                        </div>
                                    </div>
                                @elseif($application->suggestedpaymentmethod == 'Mobile')
                                    <div>
                                        <p class="text-xs text-blue-500 font-bold uppercase">Mobile Money Details</p>
                                        <div class="text-sm text-blue-900">
                                            <p><strong>Channel:</strong> {{ $application->mobile_channel }}</p>
                                            <p><strong>Number:</strong> {{ $application->mobile_number }}</p>
                                            <p class="mt-2 italic font-medium border-t border-blue-200 pt-2">
                                                <i class="fas fa-mobile-alt mr-2"></i> Funds to be disbursed via Mobile Money.
                                            </p>
                                        </div>
                                    </div>
                                @else
                                    <div class="flex items-center">
                                        <p class="text-sm text-blue-800 italic font-medium"><i class="fas fa-hand-holding-usd mr-2"></i> Funds to be disbursed via cash counter.</p>
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                    <h2 class="text-xl font-bold text-gray-800 mb-4">Funding Details</h2>
                    
                    <form action="{{ route('admin.applications.process-disbursement', $application->id) }}" method="POST" id="disbursementForm">
                        @csrf
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label for="disbursement_date" class="block text-sm font-medium text-gray-700 mb-2">Disbursement Date *</label>
                                <input type="date" name="disbursement_date" id="disbursement_date" 
                                       value="{{ date('Y-m-d') }}"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500"
                                       required max="{{ date('Y-m-d') }}">
                                <p class="text-xs text-gray-500 mt-1">
                                    @php
                                        $appDate = $application->submitted_at ? \Carbon\Carbon::parse($application->submitted_at)->format('Y-m-d') : null;
                                        $approvalDate = $application->approved_at ? \Carbon\Carbon::parse($application->approved_at)->format('Y-m-d') : null;
                                    @endphp
                                    
                                    @if($appDate && $approvalDate)
                                        Application: {{ $appDate }} | Approval: {{ $approvalDate }}
                                    @elseif($appDate)
                                        Application submitted: {{ $appDate }}
                                    @endif
                                </p>
                            </div>

                            <div>
                                <label for="payment_account_id" class="block text-sm font-medium text-gray-700 mb-2">Source Fund Account *</label>
                                <select name="payment_account_id" id="payment_account_id" 
                                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500"
                                        required>
                                    <option value="">Select Cash/Bank Account</option>
                                    @foreach($cashAccounts as $account)
                                        <option value="{{ $account->cashBankId }}" data-balance="{{ $account->current_balance }}">
                                            {{ $account->accountName }} ({{ $account->currency }}) - ZMW {{ number_format($account->current_balance, 2) }}
                                        </option>
                                    @endforeach
                                </select>
                                <p id="account-balance-warning" class="mt-1 text-xs text-red-600 hidden">
                                    <i class="fas fa-exclamation-triangle mr-1"></i> Insufficient funds in this account.
                                </p>
                            </div>

                            <div class="md:col-span-2">
                                <label for="disbursement_notes" class="block text-sm font-medium text-gray-700 mb-2">Internal Disbursement Notes</label>
                                <textarea name="disbursement_notes" id="disbursement_notes" rows="2"
                                          class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500"
                                          placeholder="Transaction references or internal comments..."></textarea>
                            </div>
                        </div>

                        {{-- Date Validation Warning --}}
                        @php
                            $disbursementWarning = '';
                            if ($appDate && $approvalDate) {
                                $appDateObj = \Carbon\Carbon::parse($appDate);
                                $approvalDateObj = \Carbon\Carbon::parse($approvalDate);
                                $today = \Carbon\Carbon::today();
                                
                                if ($approvalDateObj->lt($appDateObj)) {
                                    $disbursementWarning = '⚠️ Approval date is before application date. Please verify dates.';
                                } elseif ($approvalDateObj->gt($today)) {
                                    $disbursementWarning = '⚠️ Approval date is in the future. Please verify dates.';
                                } elseif ($appDateObj->gt($today)) {
                                    $disbursementWarning = '⚠️ Application date is in the future. Please verify dates.';
                                }
                            }
                        @endphp
                        
                        @if($disbursementWarning)
                            <div class="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                                <div class="flex items-start">
                                    <i class="fas fa-exclamation-triangle text-yellow-600 mt-0.5 mr-2"></i>
                                    <div>
                                        <p class="text-sm font-medium text-yellow-800">{{ $disbursementWarning }}</p>
                                    </div>
                                </div>
                            </div>
                        @endif

                        <div class="mt-8 flex justify-end space-x-4">
                            <button type="button" id="previewButton" class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-medium transition flex items-center gap-2">
                                <i class="fas fa-eye"></i> Preview
                            </button>
                            <button type="submit" id="submitButton" class="px-6 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 font-medium transition flex items-center gap-2 disabled:opacity-50">
                                <i class="fas fa-money-bill-wave"></i> Confirm Disbursement
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>

    <div id="previewModal" class="fixed inset-0 bg-black/50 overflow-y-auto h-full w-full z-50 hidden flex items-center justify-center">
        <div class="relative p-5 border w-full max-w-xl shadow-lg rounded-xl bg-white animate-in fade-in zoom-in duration-200">
            <h3 class="text-2xl font-bold text-gray-900 mb-4 border-b pb-2">Confirm Disbursement</h3>
            <div class="space-y-4">
                <div class="bg-gray-50 p-4 rounded-lg">
                    <div class="grid grid-cols-2 gap-2 text-sm">
                        <span class="text-gray-500">Beneficiary:</span>
                        <span class="font-bold text-gray-800">{{ $application->customer->first_name }} {{ $application->customer->surname }}</span>
                        <span class="text-gray-500">Loan Amount:</span>
                        <span class="font-bold text-blue-700">ZMW {{ number_format($application->loan_amount, 2) }}</span>
                        <span class="text-gray-500">Application Date:</span>
                        <span class="font-bold text-gray-800">{{ $submittedAt ? $submittedAt->format('M d, Y') : 'N/A' }}</span>
                        <span class="text-gray-500">Approval Date:</span>
                        <span class="font-bold text-green-700">{{ $approvedAt ? $approvedAt->format('M d, Y') : 'N/A' }}</span>
                    </div>
                </div>
                <div class="bg-blue-50 p-4 rounded-lg border border-blue-100">
                    <div class="grid grid-cols-2 gap-2 text-sm text-blue-900">
                        <span>Funding From:</span>
                        <span class="font-bold" id="p-account">-</span>
                        <span>Disbursement Date:</span>
                        <span class="font-bold" id="p-date">-</span>
                        <span>Processing Time:</span>
                        <span class="font-bold">
                            @if($processingTime !== null)
                                {{ $processingTime }} {{ $processingTime == 1 ? 'day' : 'days' }}
                            @else
                                N/A
                            @endif
                        </span>
                    </div>
                </div>
            </div>
            <div class="mt-6 flex justify-end space-x-3">
                <button type="button" id="closeModal" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg">Cancel</button>
                <button type="button" id="confirmDisbursement" class="px-6 py-2 bg-purple-600 text-white rounded-lg font-bold">Finalize Transaction</button>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const acctSelect = document.getElementById('payment_account_id');
            const warning = document.getElementById('account-balance-warning');
            const loanAmt = {{ $application->loan_amount }};
            const disbursementDateInput = document.getElementById('disbursement_date');

            acctSelect.addEventListener('change', function() {
                const bal = parseFloat(this.options[this.selectedIndex].dataset.balance || 0);
                warning.classList.toggle('hidden', bal >= loanAmt);
            });

            // Check if disbursement date is backdated
            disbursementDateInput.addEventListener('change', function() {
                const selectedDate = new Date(this.value);
                const today = new Date();
                today.setHours(0, 0, 0, 0);
                
                if (selectedDate < today) {
                    const daysAgo = Math.floor((today - selectedDate) / (1000 * 60 * 60 * 24));
                    if (daysAgo > 30) {
                        if (!confirm(`⚠️ WARNING: You're disbursing ${daysAgo} days in the past.\n\nThis is a significant backdate. Continue anyway?`)) {
                            this.value = '{{ date("Y-m-d") }}';
                        }
                    }
                }
            });

            const modal = document.getElementById('previewModal');
            document.getElementById('previewButton').addEventListener('click', () => {
                if(!acctSelect.value) return alert('Select source funding account.');
                
                // Get selected date and check for backdating
                const selectedDate = new Date(disbursementDateInput.value);
                const today = new Date();
                today.setHours(0, 0, 0, 0);
                
                document.getElementById('p-account').textContent = acctSelect.options[acctSelect.selectedIndex].text;
                document.getElementById('p-date').textContent = disbursementDateInput.value;
                
                // Show backdate warning in modal if applicable
                if (selectedDate < today) {
                    const daysAgo = Math.floor((today - selectedDate) / (1000 * 60 * 60 * 24));
                    let warningHtml = '';
                    
                    if (daysAgo > 90) {
                        warningHtml = `
                            <div class="mt-4 p-3 bg-red-50 border border-red-200 rounded-lg">
                                <div class="flex items-start">
                                    <i class="fas fa-exclamation-triangle text-red-600 mt-0.5 mr-2"></i>
                                    <div>
                                        <p class="text-sm font-medium text-red-800">⚠️ Extensive Backdating</p>
                                        <p class="text-xs text-red-700 mt-1">
                                            Disbursement is being backdated ${daysAgo} days (over 3 months). 
                                            This will create immediate past-due installments.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        `;
                    } else if (daysAgo > 30) {
                        warningHtml = `
                            <div class="mt-4 p-3 bg-orange-50 border border-orange-200 rounded-lg">
                                <div class="flex items-start">
                                    <i class="fas fa-exclamation-triangle text-orange-600 mt-0.5 mr-2"></i>
                                    <div>
                                        <p class="text-sm font-medium text-orange-800">Backdating Notice</p>
                                        <p class="text-xs text-orange-700 mt-1">
                                            Disbursement is being backdated ${daysAgo} days. 
                                            Past-due installments will be created immediately.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        `;
                    } else if (daysAgo > 7) {
                        warningHtml = `
                            <div class="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                                <div class="flex items-start">
                                    <i class="fas fa-exclamation-triangle text-yellow-600 mt-0.5 mr-2"></i>
                                    <div>
                                        <p class="text-sm font-medium text-yellow-800">Backdating Information</p>
                                        <p class="text-xs text-yellow-700 mt-1">
                                            Disbursement is being backdated ${daysAgo} days.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        `;
                    }
                    
                    // Insert warning if it exists
                    const existingWarning = document.getElementById('backdate-warning-modal');
                    if (existingWarning) existingWarning.remove();
                    
                    if (warningHtml) {
                        const warningDiv = document.createElement('div');
                        warningDiv.id = 'backdate-warning-modal';
                        warningDiv.innerHTML = warningHtml;
                        document.querySelector('#previewModal .space-y-4').appendChild(warningDiv);
                    }
                }
                
                modal.classList.remove('hidden');
            });

            document.getElementById('closeModal').addEventListener('click', () => {
                const warningDiv = document.getElementById('backdate-warning-modal');
                if (warningDiv) warningDiv.remove();
                modal.classList.add('hidden');
            });
            
            document.getElementById('confirmDisbursement').addEventListener('click', () => {
                document.getElementById('disbursementForm').submit();
            });
        });
    </script>
</body>
</html>